// -*- C++ -*-

/* 
 * GChemPaint library
 * molecule.cc 
 *
 * Copyright (C) 2001-2004
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "molecule.h"
#include "document.h"

gcpMolecule::gcpMolecule(TypeId Type): Object(Type)
{
}

gcpMolecule::gcpMolecule(gcpAtom* pAtom): Object(MoleculeType)
{
	AddAtom(pAtom);
	gcpChain* pChain = new gcpChain(this, pAtom); //will find the cycles
	delete pChain;
}

gcpMolecule::~gcpMolecule()
{
	std::list<gcpBond*>::iterator n;
	for (n = m_Bonds.begin(); n != m_Bonds.end(); n++) (*n)->RemoveAllCycles();
	while (!m_Cycles.empty())
	{
		delete m_Cycles.front();
		m_Cycles.pop_front();
	}
	while (!m_Chains.empty())
	{
		delete m_Chains.front();
		m_Chains.pop_front();
	}
}

void gcpMolecule::AddAtom(gcpAtom* pAtom)
{
	m_Atoms.push_back(pAtom);
	AddChild(pAtom);
}

void gcpMolecule::AddFragment(gcpFragment* pFragment)
{
	m_Fragments.push_back(pFragment);
	AddChild(pFragment);
}

void gcpMolecule::AddBond(gcpBond* pBond)
{
	m_Bonds.push_back(pBond);
	AddChild(pBond);
}

void gcpMolecule::Remove(Object* pObject)
{
	switch (pObject->GetType())
	{
		case AtomType:
			m_Atoms.remove((gcpAtom*)pObject);
			break;
		case BondType:
			m_Bonds.remove((gcpBond*)pObject);
			break;
		case FragmentType:
			m_Fragments.remove((gcpFragment*)pObject);
			break;
	}
	pObject->SetParent(GetParent());
}

void gcpMolecule::UpdateCycles(gcpBond* pBond)	//FIXME: function not totally implemented
{
	gcpChain* pChain = new gcpChain(this, pBond); //will find the cycles
	delete pChain;
}

void gcpMolecule::Merge(gcpMolecule* pMolecule)
{
	gcpAtom* pAtom;
	gcpFragment* pFragment;
	gcpBond* pBond;
	gcpChain* pChain;
	gcpCycle* pCycle;
	pAtom = m_Atoms.front();
	while (!pMolecule->m_Atoms.empty())
	{
		pAtom = pMolecule->m_Atoms.front();
		AddAtom(pAtom);
		pMolecule->m_Atoms.pop_front();
	}
	while (!pMolecule->m_Fragments.empty())
	{
		pFragment = pMolecule->m_Fragments.front();
		AddFragment(pFragment);
		pMolecule->m_Fragments.pop_front();
	}
	while (!pMolecule->m_Bonds.empty())
	{
		pBond = pMolecule->m_Bonds.front();
		AddBond(pBond);
		pMolecule->m_Bonds.pop_front();
	}
	while (!pMolecule->m_Chains.empty())	//FIXME: Chains should change
	{
		pChain = pMolecule->m_Chains.front();
		m_Chains.push_back(pChain);
		pMolecule->m_Chains.pop_front();
	}
	while (!pMolecule->m_Cycles.empty())
	{
		pCycle = pMolecule->m_Cycles.front();
		m_Cycles.push_back(pCycle);
		pMolecule->m_Cycles.pop_front();
	}
	delete pMolecule;
}

xmlNodePtr gcpMolecule::Save(xmlDocPtr xml)
{
	xmlNodePtr node;
	node = xmlNewDocNode(xml, NULL, (xmlChar*)"molecule", NULL);
	if (!node) return NULL;
	SaveId(node);
	
	if (!SaveChildren(xml, node))
	{
		xmlFreeNode(node);
		return NULL;
	}
	return node;
}

bool gcpMolecule::Load(xmlNodePtr node)
{
	char* tmp;
	xmlNodePtr child;
	Object* pObject;
	gcpDocument* pDoc = (gcpDocument*)GetDocument();

	tmp = (char*)xmlGetProp(node, (xmlChar*)"id");
	if (tmp) SetId(tmp);
	child = GetNodeByName(node, "atom");
	while (child)
	{
		pObject = new gcpAtom();
		if (!pObject->Load(child)) 
		{
			delete pObject;
			return false;
		}
		if (pDoc) pDoc->AddAtom((gcpAtom*)pObject);
		AddAtom((gcpAtom*)pObject);
		child = GetNextNodeByName(child->next, "atom");
	}

	child = GetNodeByName(node, "fragment");
	while (child)
	{
		pObject = new gcpFragment();
		AddChild(pObject);
		if (!pObject->Load(child)) 
		{
			delete pObject;
			return false;
		}
		if (pDoc) pDoc->AddFragment((gcpFragment*)pObject);
		AddFragment((gcpFragment*)pObject);
		((gcpFragment*)pObject)->AnalContent();
		child = GetNextNodeByName(child->next, "fragment");
	}

	child = GetNodeByName(node, "bond");
	while (child)
	{
		pObject = new gcpBond();
		AddBond((gcpBond*)pObject);
		if (!pObject->Load(child))
		{
			delete pObject;
			m_Bonds.remove((gcpBond*)pObject);
			return false;
		}
		if (pDoc) pDoc->AddBond((gcpBond*)pObject);
		child = GetNextNodeByName(child->next, "bond");
	}
	if (!m_Atoms.empty())
	{
		gcpAtom* pAtom = m_Atoms.front();
		std::list<gcpAtom*>::iterator i = m_Atoms.begin();
		i++;
		for (; i != m_Atoms.end(); i++) (*i)->SetParent(NULL);
		gcpChain* pChain = new gcpChain(this, pAtom); //will find the cycles
		delete pChain;
	}
	return true;
}

void gcpMolecule::UpdateCycles()
{
	std::list<gcpBond*>::iterator n;
	for (n = m_Bonds.begin(); n != m_Bonds.end(); n++) (*n)->RemoveAllCycles();
	while (!m_Cycles.empty())
	{
		delete m_Cycles.front();
		m_Cycles.pop_front();
	}
	if (!m_Atoms.empty())
	{
		std::list<gcpAtom*>::iterator i = m_Atoms.begin();
		i++;
		for (; i != m_Atoms.end(); i++) (*i)->SetParent(NULL);
		gcpChain* pChain = new gcpChain(this, *(m_Atoms.begin())); //will find the cycles
		delete pChain;
	}
}

void gcpMolecule::Clear()
{
	m_Bonds.clear();
	m_Atoms.clear();
	m_Fragments.clear();
}

void gcpMolecule::SetSelected(GtkWidget* w, int state)
{
	map<string, Object*>::iterator i;
	Object *child = GetFirstChild(i);
	while (child)
	{
		child->SetSelected(w, state);
		child= GetNextChild(i);
	}
}
