// -*- C++ -*-

/* 
 * GChemPaint selection plugin
 * selectiontool.cc
 *
 * Copyright (C) 2001-2004
 *
 * Developed by Jean Bréfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "selectiontool.h"
#include "lib/molecule.h"
#include "lib/settings.h"
#include "lib/document.h"
#include "lib/application.h"
#include <math.h>
#include <stdexcept>

static bool on_flip(GtkWidget *btn, gcpApplication* App)
{
	gcpSelectionTool *tool = (gcpSelectionTool*) App->GetTool ("Select");
	tool->OnFlip (strcmp (gtk_widget_get_name (btn), "VertFlip"));
}

static bool on_rotate(GtkWidget *btn, gcpApplication* App)
{
	gcpSelectionTool *tool = (gcpSelectionTool*) App->GetTool ("Select");
	tool->Rotate (gtk_toggle_tool_button_get_active (GTK_TOGGLE_TOOL_BUTTON (btn)));
}

static bool on_merge(GtkWidget *btn, gcpApplication* App)
{
	gcpSelectionTool *tool = (gcpSelectionTool*) App->GetTool ("Select");
	tool->Merge ();
}

gcpSelectionTool::gcpSelectionTool(gcpApplication *App): gcpTool(App, "Select")
{
	m_pApp->SetCallback("flip", (GCallback) on_flip);
	m_pApp->SetCallback("rotate", (GCallback) on_rotate);
	m_pApp->SetCallback("merge", (GCallback) on_merge);
	m_bRotate = false;
	m_UIManager = gtk_ui_manager_new ();
}

gcpSelectionTool::~gcpSelectionTool()
{
	g_object_unref (m_UIManager);
}

bool gcpSelectionTool::OnClicked()
{
	if (m_pObject)
	{
/*		switch (m_pObject->GetType())
		{
			case BondType:
			case AtomType:
				gcpMolecule* pMol = (gcpMolecule*)m_pObject->GetMolecule();
				if (pMol)
				m_pObject = pMol;
				break;
		}*/
		Object* pObj = m_pObject->GetGroup();
		if (pObj) m_pObject = pObj;
		if (!m_pData->IsSelected(m_pObject))
		{
			m_pData->UnselectAll();
			m_pData->SetSelected(m_pObject);
			m_pApp->ActivateMenu("Copy", true);
			m_pApp->ActivateMenu("Cut", true);
			m_pApp->ActivateMenu("Erase", true);
		}
	}
	else
	{
		m_pData->UnselectAll();
		m_pApp->ActivateMenu("Copy", false);
		m_pApp->ActivateMenu("Cut", false);
		m_pApp->ActivateMenu("Erase", false);
	}
	if (m_bRotate) {
		// Calculate center of selection
		ArtDRect rect;
		m_pData->GetSelectionBounds(rect);
		m_cx = (rect.x0 + rect.x1) / 2.;
		m_cy = (rect.y0 + rect.y1) / 2.;
		m_dAngle = 0.;
		m_x0 -= m_cx;
		m_y0 -= m_cy;
		if (m_x0 == 0)
			m_dAngleInit = (m_y0 <= 0) ? 90 : 270;
		else
			m_dAngleInit = atan(-m_y0/m_x0) * 180 / M_PI;
		if (m_x0 < 0) m_dAngleInit += 180.;
		std::list<Object*>::iterator i;
		gcpDocument* pDoc = m_pView->GetDoc();
		m_pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
		for (i = m_pData->SelectedObjects.begin(); i != m_pData->SelectedObjects.end(); i++)
			m_pOp->AddObject(*i,0);
	}
	return true;
}

void gcpSelectionTool::OnDrag()
{
	double dx = m_x - m_x1, dy = m_y - m_y1, x1, y1, x2, y2;
	m_x1 = m_x;
	m_y1 = m_y;
	if (m_pObject)
	{
		if (m_bRotate) {
			double dAngle;
			m_x-= m_cx;
			m_y -= m_cy;
			if (m_x == 0)
			{
				if (m_y == 0) return;
				dAngle = (m_y < 0) ? 90 : 270;
			}
			else
			{
				dAngle = atan(-m_y/m_x) * 180. / M_PI;
				if (m_x < 0) dAngle += 180.;
				dAngle -= m_dAngleInit;
				if (!(m_nState & GDK_CONTROL_MASK)) dAngle = rint(dAngle / 5) * 5;
			}
			if (dAngle < -180.) dAngle += 360.;
			if (dAngle > 180.) dAngle -= 360.;
			if (dAngle != m_dAngle) {
				m_pData->RotateSelection (m_cx, m_cy, dAngle - m_dAngle);
				m_dAngle = dAngle;
			}
			char tmp[32];
			snprintf(tmp, sizeof(tmp) - 1, _("Orientation: %g"), dAngle);
			m_pApp->SetStatusText(tmp);
		} else
			m_pData->MoveSelectedItems(dx, dy);
	}
	else
	{
		if (m_pItem)
		{
			gnome_canvas_item_get_bounds(m_pItem, &x1, &y1, &x2, &y2);
			g_object_set(G_OBJECT(m_pItem), "x2", m_x, "y2", m_y, NULL);
			gnome_canvas_request_redraw(GNOME_CANVAS(m_pWidget), (int)x1, (int)y1, (int)x2, (int)y2);
		}
		else
		{
			m_pItem = gnome_canvas_item_new(
									m_pData->Group,
									gnome_canvas_rect_get_type(),
									"x1", m_x0,
									"y1", m_y0,
									"x2", m_x,
									"y2", m_y,
									"outline_color", SelectColor,
									"width_units", m_pData->BondWidth,
									NULL);
		}
	}
}

void gcpSelectionTool::OnRelease()
{
	m_pApp->ClearStatus();
	if (m_pObject)
	{
		if (m_bRotate) {
			std::list<Object*>::iterator i;
			gcpDocument* pDoc = m_pView->GetDoc();
			for (i = m_pData->SelectedObjects.begin(); i != m_pData->SelectedObjects.end(); i++)
				m_pOp->AddObject(*i,1);
			pDoc->FinishOperation();
		} else {
			double dx = m_x1 - m_x0, dy = m_y1 - m_y0;
			if (dx != 0.0 && dy != 0.0)
			{
				m_pData->MoveSelectedItems(-dx, -dy);
				m_pData->MoveSelection(dx, dy);
			}
		}
	}
	else
	{
		if (m_x < m_x0)
		{
			m_x1 = m_x0;
			m_x0 = m_x;
		}
		else m_x1 = m_x;
		if (m_y < m_y0)
		{
			m_y1 = m_y0;
			m_y0 = m_y;
		}
		else m_y1 = m_y;
		double x0, y0, x1, y1;
		std::map<Object*, GnomeCanvasGroup*>::iterator j;
		for (j = m_pData->Items.begin(); j != m_pData->Items.end(); j++)
		{
			if (!m_pData->IsSelected((*j).first))
			{
				gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM((*j).second), &x0, &y0,&x1, &y1);
				if ((x0 < m_x1) && (y0 < m_y1) && (x1 > m_x0) && (y1 > m_y0))
				{
					m_pObject = (*j).first->GetGroup ();	//GetMolecule();
					if (m_pObject) {
						if (!m_pData->IsSelected(m_pObject))
							m_pData->SetSelected(m_pObject);
					}
					else
						m_pData->SetSelected((*j).first);
				}
			}
		}
	}
	AddSelection(m_pData);
	// If the selection is made of two molecules, activate the merge tool
	m_pApp->ActivateToolItem ("Merge", ((m_pData->SelectedObjects.size () == 2) &&
		(m_pData->SelectedObjects.front()->GetType() == MoleculeType) &&
		(m_pData->SelectedObjects.back()->GetType() == MoleculeType)));
}

void gcpSelectionTool::Activate()
{
	m_pApp->ActivateToolItem ("HorizFlip", true);
	m_pApp->ActivateToolItem ("VertFlip", true);
	m_pApp->ActivateToolItem ("Rotate", true);
	m_pApp->ActivateToolItem ("Merge", false);
	gcpDocument *pDoc = m_pApp->GetActiveDocument ();
	if (pDoc) {
		m_pView = m_pApp->GetActiveDocument ()->GetView ();
		GtkWidget *w = m_pView->GetWidget ();
		m_pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (w), "data");
	}
}

bool gcpSelectionTool::Deactivate()
{
	while (!SelectedWidgets.empty())
	{
		SelectedWidgets.front()->UnselectAll();
		SelectedWidgets.pop_front();
	}
	m_pApp->ActivateToolItem ("HorizFlip", false);
	m_pApp->ActivateToolItem ("VertFlip", false);
	m_pApp->ActivateToolItem ("Rotate", false);
	m_pApp->ActivateToolItem ("Merge", false);
	return true;
}

void gcpSelectionTool::AddSelection(gcpWidgetData* data)
{
	gcpWidgetData *d = m_pData;
	m_pData = data;
	m_pView = data->View;
	if (m_pData->HasSelection())
	{
		GtkClipboard *clipboard = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
		m_pView->OnCopySelection (m_pData->Canvas, clipboard);
		if (!m_pView->IsEmbedded())
		{
			m_pApp->ActivateMenu("Copy", true);
			m_pApp->ActivateMenu("Cut", true);
			m_pApp->ActivateMenu("Erase", true);
		}
	}
	SelectedWidgets.remove(m_pData);
	SelectedWidgets.push_front(m_pData);
	m_pData = d;
	if (d) m_pView = d->View;
	else m_pView = NULL;
}

void gcpSelectionTool::OnFlip (bool horizontal)
{
	if (!m_pData) {
		m_pView = m_pApp->GetActiveDocument ()->GetView ();
		GtkWidget *w = m_pView->GetWidget ();
		m_pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (w), "data");
	}
	if (!m_pData->SelectedObjects.size ())
		return;
	ArtDRect rect;
	m_pData->GetSelectionBounds(rect);
	m_cx = (rect.x0 + rect.x1) / 2.;
	m_cy = (rect.y0 + rect.y1) / 2.;
	m_x = (horizontal)? -1.: 1.;
	Matrix2D m(m_x, 0., 0., -m_x);
	std::list<Object*>::iterator i;
	gcpDocument* pDoc = m_pView->GetDoc();
	m_pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
	for (i = m_pData->SelectedObjects.begin(); i != m_pData->SelectedObjects.end(); i++) {
		m_pOp->AddObject(*i,0);
		(*i)->Transform2D (m, m_cx / m_pData->ZoomFactor, m_cy / m_pData->ZoomFactor);
		m_pView->Update(*i);
		m_pOp->AddObject(*i,1);
	}
	pDoc->FinishOperation();
}

void gcpSelectionTool::Rotate (bool rotate)
{
	m_bRotate = rotate;
}

void gcpSelectionTool::Merge ()
{
	gcpMolecule *pMol0, *pMol1;
	gcpDocument* pDoc = m_pApp->GetActiveDocument ();
	if (!m_pData) {
		m_pView = pDoc->GetView ();
		GtkWidget *w = m_pView->GetWidget ();
		m_pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (w), "data");
	}
	pMol0 = (gcpMolecule*) m_pData->SelectedObjects.front();
	pMol1 = (gcpMolecule*) m_pData->SelectedObjects.back();
	m_pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
	m_pOp->AddObject (pMol0, 0);
	m_pOp->AddObject (pMol1, 0);
	m_pData->UnselectAll ();
	if (pMol0->Merge (pMol1, true)) {
		m_pOp->AddObject (pMol0, 1);
		m_pData->SetSelected (pMol0);
		AddSelection(m_pData);
		m_pView->Update (pMol0);
		pDoc->FinishOperation();
	} else {
		pDoc->AbortOperation ();
	}
	m_pApp->ActivateToolItem ("Merge", false);
}

static void on_create_group (gcpSelectionTool* tool)
{
	tool->CreateGroup ();
}

void gcpSelectionTool::CreateGroup ()
{
	gcpDocument *pDoc = m_pView->GetDoc ();
	Object *pObj = Object::CreateObject (Object::GetTypeName (m_Type), pDoc);
	try {
		m_pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
		std::list<Object*>::iterator i;
		for (i = m_pData->SelectedObjects.begin(); i != m_pData->SelectedObjects.end(); i++)
			m_pOp->AddObject(*i,0);
		if (!pObj->Build (m_pData->SelectedObjects))
			throw logic_error (_("Creation failed!"));
		m_pView->Update (pObj);
		m_pData->UnselectAll ();
		m_pData->SetSelected (pObj);
		AddSelection(m_pData);
		m_pOp->AddObject (pObj, 1);
		pDoc->FinishOperation ();
	}
	catch (invalid_argument& e) {
			pDoc->AbortOperation ();
			delete pObj;
			GtkWidget* message = gtk_message_dialog_new (NULL, (GtkDialogFlags) 0,
								GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, e.what ());
			g_signal_connect_swapped (G_OBJECT (message), "response", G_CALLBACK (gtk_widget_destroy), G_OBJECT (message));
			gtk_widget_show(message);
	}
}

bool gcpSelectionTool::OnRightButtonClicked ()
{
	// first destroy the GtkUIManager
	GtkWidget *w;
	g_object_unref (m_UIManager);
	m_UIManager = gtk_ui_manager_new ();
	GtkActionGroup *group = gtk_action_group_new ("selection");
	if (m_pData->SelectedObjects.size () > 1) {
		set<TypeId> possible_types, types, wrong_types;
		list<Object*>::iterator  i = m_pData->SelectedObjects.begin (),
												end = m_pData->SelectedObjects.end ();
		(*i)->GetPossibleAncestorTypes (possible_types);
		set<TypeId>::iterator type;
		for (i++; i != end; i++) {
			(*i)->GetPossibleAncestorTypes (types);
			for (type = possible_types.begin(); type != possible_types.end (); type++)
				if (types.find (*type) == types.end ())
					wrong_types.insert (*type);
			for (type = wrong_types.begin(); type != wrong_types.end (); type++)
				possible_types.erase (*type);
			wrong_types.clear ();
		}
		if (possible_types.size () == 1) {
			// Add a new action.
			m_Type = *possible_types.begin();
			const string &label = Object::GetCreationLabel (m_Type);
			if (label.size ()) {
				GtkAction *action = gtk_action_new ("create_group", label.c_str (), NULL, NULL);
				gtk_action_group_add_action (group, action);
				gtk_ui_manager_insert_action_group (m_UIManager, group, 0);
				char buf[] = "<ui><popup><menuitem action='create_group'/></popup></ui>";
				m_uiIds.push_front (gtk_ui_manager_add_ui_from_string (m_UIManager, buf, -1, NULL));
				w = gtk_ui_manager_get_widget (m_UIManager, "/popup/create_group");
				g_signal_connect_swapped (w, "activate", G_CALLBACK (on_create_group), this);
			}
			w = gtk_ui_manager_get_widget (m_UIManager, "/popup");
			gtk_menu_popup (GTK_MENU (w), NULL, NULL, NULL, this, 3,  gtk_get_current_event_time ());
		}
	}
	return true;
}
