// -*- C++ -*-

/* 
 * GChemPaint library
 * reactant.cc 
 *
 * Copyright (C) 2002-2004
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "reactant.h"
#include "reaction-step.h"
#include "application.h"
#include "document.h"
#include "text.h"
#include <glib/gi18n.h>

gcpReactant::gcpReactant (): Object (ReactantType)
{
	SetId ("r1");
	m_Stoech = 0;
	Child = NULL;
	Stoechiometry = NULL;
}

gcpReactant::gcpReactant (gcpReactionStep* step, Object *object)	throw (invalid_argument): Object (ReactantType)
{
	SetId ("r1");
	step->AddChild (this);
	GetDocument ()->EmptyTranslationTable();
	static const set<TypeId>& allowed_types = Object::GetRules ("reactant", RuleMayContain);
	if (allowed_types.find (object->GetType ()) == allowed_types.end ())
		throw invalid_argument ("invalid reactant");
	AddChild (object);
	Child = object;
	Stoechiometry = NULL;
	m_Stoech = 0;
}

gcpReactant::~gcpReactant ()
{
}

static void do_add_stoechiometry (gcpReactant *reactant)
{
	reactant->AddStoechiometry ();
}

bool gcpReactant::BuildContextualMenu (GtkUIManager *UIManager, Object *object)
{
	bool result = false;
	if (m_Stoech == 0 && !Stoechiometry) {
		GtkActionGroup *group = gtk_action_group_new ("reactant");
		GtkAction *action = gtk_action_new ("stoechiometry", _("Add a stoechiometry coefficient"), NULL, NULL);
		gtk_action_group_add_action (group, action);
		gtk_ui_manager_insert_action_group (UIManager, group, 0);
		char buf[] = "<ui><popup><menuitem action='stoechiometry'/></popup></ui>";
		gtk_ui_manager_add_ui_from_string (UIManager, buf, -1, NULL);
		GtkWidget *w = gtk_ui_manager_get_widget (UIManager, "/popup/stoechiometry");
		g_signal_connect_swapped (w, "activate", G_CALLBACK (do_add_stoechiometry), this);
		result = true;
	}
	return result | GetParent ()->BuildContextualMenu (UIManager, object);
}

xmlNodePtr gcpReactant::Save (xmlDocPtr xml)
{
	if (!Child)
		return NULL;
	xmlNodePtr node = xmlNewDocNode (xml, NULL, (const xmlChar*) "reactant", NULL);
	SaveId (node);
	xmlNodePtr child = Child->Save (xml);
	xmlAddChild (node, child);
	if (Stoechiometry) {
		xmlNodePtr stoech = Stoechiometry->Save (xml);
		xmlNodeSetName (stoech, (const xmlChar*) "stoechiometry");
		xmlAddChild (node, stoech);
	}
	return node;
}

bool gcpReactant::Load (xmlNodePtr node)
{
	xmlChar* buf;
	xmlNodePtr child;

	m_IsLoading = true;
	buf = xmlGetProp (node, (xmlChar*) "id");
	if (buf) {
		SetId ((char*) buf);
		xmlFree (buf);
	}
	child = node->children;
	gcpDocument *pDoc = (gcpDocument*) GetDocument ();
	while (child) {
		if (!strcmp ((const char*) child->name, "stoechiometry")) {
			if (Stoechiometry) {
				m_IsLoading = false;
				return false;
			}
			Stoechiometry = new gcpText ();
			AddChild (Stoechiometry);
			if (!Stoechiometry->Load (child)) {
				delete Stoechiometry;
				m_IsLoading = false;
				return false;
			};
			pDoc->AddObject (Stoechiometry);
		} else {
			if (Child) {
				if (strcmp ((const char*) child->name, "text")) {
					m_IsLoading = false;
					return false;
				} else {
					child = child->next;
					continue;
				}
			}
			Child = CreateObject ((const char*) child->name, this);
			if (Child) {
				AddChild (Child);
				if (!Child->Load (child)) {
					delete Child;
					Child = NULL;
				}
			}
		}
		child = child->next;
	}
	m_IsLoading = false;
	return (Child != NULL);
}

double gcpReactant::GetYAlign ()
{
	return (Child)? Child->GetYAlign (): 0.;
}

void gcpReactant::AddStoechiometry ()
{
	gcpDocument *pDoc = (gcpDocument*) GetDocument ();
	gcpApplication * pApp = pDoc->GetApplication ();
	gcpView *pView = pDoc->GetView ();
	gcpWidgetData *pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (pDoc->GetWidget ()), "data");
	ArtDRect rect;
	pData->GetObjectBounds (this, &rect);
	double x = rect.x0 / pData->ZoomFactor;
	gcpText *text = new gcpText (x, GetYAlign () + pView->GetBaseLineOffset ());
	Stoechiometry = text;
	AddChild (text);
	pDoc->AddObject (text);
	pData->GetObjectBounds (text, &rect);
	Child->Move (rect.x1 / pData->ZoomFactor + pView->GetStoechiometryPadding () - x, 0.);
	gcpTool *tool = pApp->GetTool ("Text");
	GetParent ()->EmitSignal (OnChangedSignal);
	pApp->ActivateTool ("Text", true);
	tool->OnClicked (pView, text, rect.x0 * pData->ZoomFactor, GetYAlign () * pData->ZoomFactor, 0);
}

bool gcpReactant::OnSignal (SignalId Signal, Object *Obj)
{
	if (Signal == OnChangedSignal) {
		gcpDocument *pDoc = (gcpDocument*) GetDocument ();
		gcpView *pView = pDoc->GetView ();
		gcpWidgetData *pData = (gcpWidgetData*) g_object_get_data (G_OBJECT (pDoc->GetWidget ()), "data");
		ArtDRect rect;
		unsigned n = GetChildrenNumber ();
		if ((n == 1) && Stoechiometry) {
			// Child or stoechiometry have been deleted
		} else if ((n == 2) && Stoechiometry) {
			// Just need to space the two children
			gnome_canvas_update_now (GNOME_CANVAS (pData->Canvas));
			pData->GetObjectBounds (Stoechiometry, &rect);
			double x = rect.x1 / pData->ZoomFactor + pView->GetStoechiometryPadding ();
			pData->GetObjectBounds (Child, &rect);
			Child->Move (x - rect.x0 / pData->ZoomFactor, 0.);
			GtkTextBuffer *buf = (GtkTextBuffer*) ((gcpTextObject*) Stoechiometry)->GetTextBuffer ();
			GtkTextIter start, end;
			gtk_text_buffer_get_bounds (buf, &start, &end);
			const char *txt = gtk_text_buffer_get_text (buf, &start, &end, true);
			char *endptr;
			int n = strtol (txt, &endptr, 10);
			m_Stoech = (!*endptr)? n: 0;
		} else {
			// Most probably child has been splitted
		}
	}
	return true;
}
