// -*- C++ -*-

/* 
 * GChemPaint atoms plugin
 * chargetool.cc 
 *
 * Copyright (C) 2003-2005
 *
 * Developed by Jean Bréfort <jean.brefort@n>ormalesup.org
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gchempaint-config.h"
#include "chargetool.h"
#include "lib/settings.h"
#include "lib/document.h"
#include "lib/application.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include <math.h>

gcpChargeTool::gcpChargeTool(gcpApplication *App, string Id): gcpTool(App, Id)
{
	if (Id == string("ChargePlus")) m_glyph = "\xE2\x8a\x95";
	else if (Id == string("ChargeMinus")) m_glyph = "\xE2\x8a\x96";
	else m_glyph = 0;
}

gcpChargeTool::~gcpChargeTool()
{
}

bool gcpChargeTool::OnClicked()
{
	if (!m_pObject || (m_pObject->GetType () != AtomType))
		return false;
	gcpAtom *pAtom = (gcpAtom*) m_pObject;
	m_Charge = pAtom->GetCharge () + ((GetName() == string("ChargePlus"))? 1: -1);
	if (!pAtom->AcceptCharge (m_Charge))
		return false;
	GObject *obj;
	ArtDRect rect;
	pAtom->GetCoords (&m_x0, &m_y0);
	m_x0 *= m_dZoomFactor;
	m_y0 *= m_dZoomFactor;
	if (m_pObject->GetParent ()->GetType () == FragmentType) {
		obj = G_OBJECT (m_pData->Items[m_pObject->GetParent ()]);
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (g_object_get_data (obj, "fragment")),
				&rect.x0, &rect.y0, &rect.x1, &rect.y0);
	} else {
		obj = G_OBJECT (m_pData->Items[m_pObject]);
		GnomeCanvasItem *sym = (GnomeCanvasItem*) g_object_get_data (obj, "symbol");
		if (sym)
			gnome_canvas_item_get_bounds (sym,
					&rect.x0, &rect.y0, &rect.x1, &rect.y0);
		else
			rect.y0 = m_y0 + 6;
	}
	GnomeCanvasItem *item = (GnomeCanvasItem*) g_object_get_data (obj, "charge");
	m_dDist = 0;
	m_pData->GetObjectBounds (m_pObject, &rect);
	m_dDistMax = 1.5 * fabs (rect.y0 - m_y0);
	if (m_Charge) {
		if (item)
			gnome_canvas_item_hide (item);
		double x, y;
		m_DefaultPos = -1;
		int align = ((gcpAtom*)m_pObject)->GetChargePosition(m_DefaultPos, 0., x, y);
		if (!align) return false;
		m_Pos = m_DefaultPos;
		x *= m_dZoomFactor;
		y *= m_dZoomFactor;
		if (!m_Pos) {
			m_x = x - m_x0;
			m_y = y - m_y0;
			m_dAngle = atan (- m_y / m_x);
			if (m_x < 0)
				m_dAngle += M_PI;
		} else {
			switch (m_Pos) {
			case CHARGE_NE:
				m_dAngle = M_PI / 4;
				break;
			case CHARGE_NW:
				m_dAngle = 3 * M_PI / 4;
				break;
			case CHARGE_N:
				m_dAngle = M_PI / 2;
				break;
			case CHARGE_SE:
				m_dAngle = 7 * M_PI / 4;
				break;
			case CHARGE_SW:
				m_dAngle = 5 * M_PI / 4;
				break;
			case CHARGE_S:
				m_dAngle = 3 * M_PI / 2;
				break;
			case CHARGE_E:
				m_dAngle = 0.;
				break;
			case CHARGE_W:
				m_dAngle = M_PI;
				break;
			}
		}
		GtkAnchorType anchor;
		switch (align)
		{
			case -2:
				anchor = GTK_ANCHOR_NORTH;
				break;
			case -1:
				anchor = GTK_ANCHOR_EAST;
				break;
			case 0:
				anchor = GTK_ANCHOR_CENTER;
				break;
			case 1:
				anchor = GTK_ANCHOR_WEST;
				break;
			case 2:
				anchor = GTK_ANCHOR_SOUTH;
				break;
		}
		char *markup;
		if 	(abs (m_Charge) > 1)
			markup = g_strdup_printf ("%d<span size=\"larger\">%s</span>", abs (m_Charge), (m_Charge > 0)? "\xE2\x8a\x95": "\xE2\x8a\x96");
		else
			markup = g_strdup_printf ("<span size=\"larger\">%s</span>", (m_Charge > 0)? "\xE2\x8a\x95": "\xE2\x8a\x96");
		m_pItem = gnome_canvas_item_new(
						m_pGroup,
						gnome_canvas_text_get_type(),
						"fill_color", AddColor,
						"font-desc", m_pView->GetPangoSmallFontDesc (),
						"markup", markup,
						"anchor", anchor,
						"x", x,
						"y", y,
						NULL);
		g_free (markup);
	} else
		g_object_set (item, "fill-color", DeleteColor, NULL);
	char buf[32];
	snprintf (buf, sizeof (buf) - 1, _("Orientation: %g"), m_dAngle * 180. / M_PI);
	m_pApp->SetStatusText (buf);
	m_bChanged = true;
	return true;
}

void gcpChargeTool::OnDrag()
{
	if (m_Charge && !m_pItem)
		return;
	GObject *obj = G_OBJECT ((m_pObject->GetParent ()->GetType () == FragmentType)?
		m_pData->Items[m_pObject->GetParent ()]: m_pData->Items[m_pObject]);
	GnomeCanvasItem *item = (GnomeCanvasItem*) g_object_get_data (obj, "charge");
	int align;
	m_x -= m_x0;
	m_y -= m_y0;
	m_dDist = sqrt (square (m_x) + square (m_y));
	if (!m_pItem) {
		if (m_dDist < m_dDistMax) {
			if (!m_bChanged) {
				g_object_set (item, "fill-color", DeleteColor, NULL);
				m_bChanged = true;
			}
		} else {
			if (m_bChanged) {
				g_object_set (item, "fill-color", "black", NULL);
				m_bChanged = false;
			}
		}
	}
	double Angle = atan (- m_y / m_x);
	if (isnan (Angle))
		Angle = m_dAngle;
	else if (m_x < 0)
		Angle += M_PI;
	if (!(m_nState & GDK_CONTROL_MASK)) {
		int pos = (int) rint (Angle * 4. / M_PI);
		Angle = (double) pos * M_PI / 4.;
		if (m_nState & GDK_SHIFT_MASK)
			pos = 8;
		else if (pos < 0)
			pos += 8;
		switch (pos) {
		case 0:
			m_Pos = POSITION_E;
			break;
		case 1:
			m_Pos = POSITION_NE;
			break;
		case 2:
			m_Pos = POSITION_N;
			break;
		case 3:
			m_Pos = POSITION_NW;
			break;
		case 4:
			m_Pos = POSITION_W;
			break;
		case 5:
			m_Pos = POSITION_SW;
			break;
		case 6:
			m_Pos = POSITION_S;
			break;
		case 7:
			m_Pos = POSITION_SE;
			break;
		default:
			m_Pos = 0;
		}
	} else
		m_Pos = 0;
	if ((Angle == m_dAngle) && !(m_nState & GDK_SHIFT_MASK)) {
		if (m_dDist < m_dDistMax) {
			if (!m_bChanged) {
				gnome_canvas_item_show (m_pItem);
				if (item)
					gnome_canvas_item_hide (item);
				m_bChanged = true;
			}
		} else {
			if (m_bChanged) {
				if (item)
					gnome_canvas_item_show (item);
				gnome_canvas_item_hide (m_pItem);
				m_bChanged = false;
			}
		}
	} else {
		double x, y, x1, y1, x2, y2;
		gcpAtom *pAtom = (gcpAtom*)m_pObject;
		if (!(m_nState & GDK_SHIFT_MASK) && (m_dDist >= m_dDistMax) && m_bChanged) {
			gnome_canvas_item_hide (m_pItem);
			m_bChanged = false;
		} else if (align = pAtom->GetChargePosition (m_Pos, Angle * 180. / M_PI, x, y)) {
			m_dAngle = Angle;
			GtkAnchorType anchor;
			if (m_nState & GDK_SHIFT_MASK) {
				align = 0;
				x = m_x0 + m_dDist * cos (m_dAngle);
				y = m_y0 - m_dDist * sin (m_dAngle);
			} else {
				x = x * m_dZoomFactor;
				y = y * m_dZoomFactor;
			}
			switch (align)
			{
				case -2:
					anchor = GTK_ANCHOR_NORTH;
					break;
				case -1:
					anchor = GTK_ANCHOR_EAST;
					break;
				case 0:
					anchor = GTK_ANCHOR_CENTER;
					break;
				case 1:
					anchor = GTK_ANCHOR_WEST;
					break;
				case 2:
					anchor = GTK_ANCHOR_SOUTH;
					break;
			}
			g_object_set (m_pItem,
						"anchor", anchor,
						"x", x,
						"y", y,
						NULL);
			gnome_canvas_item_show (m_pItem);
			if (item)
				gnome_canvas_item_hide (item);
			m_bChanged = true;
		}
	}
	char tmp[32];
	snprintf(tmp, sizeof(tmp) - 1, _("Orientation: %g"), m_dAngle * 180. / M_PI);
	m_pApp->SetStatusText(tmp);
}

void gcpChargeTool::OnRelease()
{
	if (m_bChanged)
	{
		gcpAtom* pAtom = (gcpAtom*) m_pObject;
		char charge = pAtom->GetCharge();
		gcpDocument* pDoc = m_pView->GetDoc();
		gcpOperation* pOp = pDoc-> GetNewOperation(GCP_MODIFY_OPERATION);
		m_pObject = m_pObject->GetGroup ();
		pOp->AddObject(m_pObject, 0);
		pAtom->SetCharge(m_Charge);
		pAtom->SetChargePosition (m_Pos, m_Pos == m_DefaultPos, m_dAngle,
			(m_nState & GDK_SHIFT_MASK)? m_dDist / m_dZoomFactor: 0.);
		pAtom->Update();
		m_pView->Update(m_pObject);
		pAtom->EmitSignal (OnChangedSignal);
		pOp->AddObject(m_pObject, 1);
		pDoc->FinishOperation();
	}
}
