// -*- C++ -*-

/* 
 * GChemPaint library
 * atom.cc
 *
 * Copyright (C) 2001-2005
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#ifndef HAS_OPENBABEL_2
#	include <mol.h>
#	undef PACKAGE
#	undef PACKAGE_BUGREPORT
#	undef PACKAGE_NAME
#	undef PACKAGE_STRING
#	undef PACKAGE_TARNAME
#	undef PACKAGE_VERSION
#	undef VERSION
#endif

#include "gchempaint-config.h"
#include "atom.h"
#include "electron.h"
#include "bond.h"
#include "molecule.h"
#include "settings.h"
#include "view.h"
#include "document.h"
#include "libgcpcanvas/gcp-canvas-group.h"
#include "libgcpcanvas/gcp-canvas-rect-ellipse.h"
#include "libgcpcanvas/gcp-canvas-bpath.h"
#include "libgcpcanvas/gcp-canvas-rich-text.h"
#include "libgcpcanvas/gcp-canvas-text.h"
#include <math.h>
#include <gcu/element.h>
#ifdef HAS_OPENBABEL_2
#	include <openbabel/mol.h>
#endif

using namespace gcu;

extern GtkTextTagTable* TextTagTable;

gcpAtom::gcpAtom(): Atom()
{
	m_buf = gtk_text_buffer_new(TextTagTable);
	m_Valence = -1; //unspecified
	m_nlp = 0;
	m_nH = 0;
	m_HPos = GetBestSide();
	m_ChargeAuto = false;
	m_ascent = 0;
	m_CHeight = 0.;
	m_Changed = 0;
	m_AvailPosCached = false;
	m_OccupiedPos = 0;
	m_ChargePos = 0xff;
	m_ChargeAngle = 0.;
	m_ChargeDist = 0.;
	m_ChargeAutoPos = true;
}

gcpAtom::~gcpAtom()
{
	gcpDocument *pDoc = (gcpDocument*) GetDocument ();
	if (!pDoc)
		return;
	gcpView *pView = pDoc->GetView ();
	map<string, Object*>::iterator i;
	Object* electron = GetFirstChild (i);
	while (electron){
		pView->Remove (electron);
		delete electron;
		electron = GetNextChild (i);
	}
}

gcpAtom::gcpAtom(int Z, double x, double y, double z): Atom(Z, x, y, z)
{
	m_buf = gtk_text_buffer_new(TextTagTable);
	m_ChargeAuto = false;
	m_HPos = GetBestSide();
	m_nlp = 0;
	SetZ(Z);
	m_ascent = 0;
	m_CHeight = 0.;
	m_Changed = 0;
	m_AvailPosCached = false;
	m_OccupiedPos = 0;
	m_ChargePos = 0xff;
	m_ChargeAngle = 0.;
	m_ChargeDist = 0.;
	m_ChargeAutoPos = true;
}

gcpAtom::gcpAtom(OBAtom* atom, double zoom): Atom()
{
	m_buf = gtk_text_buffer_new(TextTagTable);
	m_x = atom->GetX() * zoom;
	m_y = - atom->GetY() * zoom;
	m_z = atom->GetZ() * zoom;
	m_nlp = 0;
	SetZ(atom->GetAtomicNum());
	gchar* Id = g_strdup_printf("a%d", atom->GetIdx());
	SetId(Id);
	g_free(Id);
//	m_Valence = atom->GetValence();
	m_HPos = true;
	m_ascent = 0;
	m_CHeight = 0.;
	m_Changed = 0;
	m_AvailPosCached = false;
	m_OccupiedPos = 0;
	m_ChargePos = 0xff;
	m_ChargeAngle = 0.;
	m_ChargeDist = 0.;
	m_ChargeAutoPos = true;
}

void gcpAtom::SetZ(int Z)
{
	Atom::SetZ(Z);
	m_Element = Element::GetElement (m_Z);
	if (m_Valence = m_Element->GetDefaultValence ())
		m_HPos = GetBestSide();
	else
		m_nH = 0;
	int max = m_Element->GetMaxValenceElectrons ();
	int diff = m_Element->GetTotalValenceElectrons () - m_Element->GetValenceElectrons ();
	switch (max) {
	case 2:
		m_ValenceOrbitals = 1;
		break;
	case 8:
		m_ValenceOrbitals = 4;
		break;
	case 18:
		if (!diff)
			m_ValenceOrbitals = 6;
		else
			m_ValenceOrbitals = 4;
		break;
	case 32:
		if (!diff)
			m_ValenceOrbitals = 8;
		else if (diff == 14)
			m_ValenceOrbitals = 6;
		else
			m_ValenceOrbitals = 4;
		break;
	default:
		m_ValenceOrbitals = 0; //should not occur
	}
	Update();
	EmitSignal (OnChangedSignal);
}

int gcpAtom::GetTotalBondsNumber()
{
	std::map<Atom*, Bond*>::iterator i;
	int n = 0;
	for (i = m_Bonds.begin(); i != m_Bonds.end(); i++)
		n += (*i).second->GetOrder();
	return n;
}

void gcpAtom::AddBond(Bond* pBond)
{
	Atom::AddBond(pBond);
	Update();
}

void gcpAtom::RemoveBond(Bond* pBond)
{
	Atom::RemoveBond(pBond);
	Update();
}

bool gcpAtom::GetBestSide()
{
	if (m_Bonds.size() == 0) return Element::BestSide(m_Z);
	std::map<Atom*, Bond*>::iterator i;
	double sum = 0.0;
	for (i = m_Bonds.begin(); i != m_Bonds.end(); i++)
		sum -= cos(((gcpBond*)(*i).second)->GetAngle2DRad(this));
	if (fabs(sum) > 0.1) return (sum >= 0.0);
	else return Element::BestSide(m_Z);
}

void gcpAtom::Update()
{
	if (m_ChargeAuto)
	{
		m_Charge = 0;
		m_ChargeAuto = false;
	}
	int nb, nexplp = 0, nexplu = 0; //nexplp is the number of explicit lone pairs
	//nexplu is the number of explicit unpaired electrons
	map<string, Object*>::iterator i;
	gcpElectron* electron = (gcpElectron*) GetFirstChild (i);
	while (electron){ 
		if (electron->IsPair ())
			nexplp++;
		else
			nexplu++;
		electron = (gcpElectron*) GetNextChild (i);
	}
	int nbonds = GetTotalBondsNumber ();
	if (m_Valence > 0) {
		m_nlp = (m_Element->GetValenceElectrons () - m_Valence) / 2;
		if ((m_Charge > 0) && (m_nlp > 0)) m_nlp --;
		else if (m_Charge < 0)
			m_nlp -= m_Charge;
		if (m_nlp < nexplp) // Can this occur ?
			m_nlp = nexplp;
		else if (m_nlp > m_ValenceOrbitals - nbonds - nexplu)
			m_nlp = m_ValenceOrbitals - nbonds - nexplu;
		if (m_nlp < 0)
			m_nlp = 0;
		nb = m_Element->GetValenceElectrons () - 2 * m_nlp - m_Charge;
		if (nb + m_nlp > 4) nb -= 2; //octet rule
		m_nH = nb - nbonds - nexplu;
		if (!m_Charge && m_nH == -1 && m_nlp > 0)
		{
			m_Charge = m_Element->GetValenceElectrons () - nbonds
						- m_nlp * 2 - nexplu;
			m_ChargeAuto = true;
			m_nH = 0;
		}
		if (m_nH < 0) { // extended octet or missing core electrons
			m_nH = 0;
			if (m_nlp || nexplu || nbonds) {
				m_Charge = m_Element->GetValenceElectrons () - 2 * m_nlp - nexplu - nbonds;
				m_ChargeAuto = true;
			}
		}
		m_HPos = GetBestSide();
	} else {
		m_nH = 0;
		if (m_ChargeAuto || !m_Charge) {
			m_Charge = m_Element->GetValenceElectrons () - 2 * nexplp - nexplu - nbonds;
			if (m_Charge > 0)
				m_Charge = 0;
			m_ChargeAuto = true;
		}
	}
	gcpDocument *pDoc = (gcpDocument *) GetDocument ();
	if (pDoc)
		m_Changed = pDoc->GetView ()->GetNbWidgets ();
	m_AvailPosCached = false;
}
	
bool gcpAtom::IsInCycle(gcpCycle* pCycle)
{
	map<Atom*, Bond*>::iterator i;
	for (i = m_Bonds.begin(); i != m_Bonds.end(); i++)
		if (((gcpBond*)(*i).second)->IsInCycle(pCycle)) return true;
	return false;
}

void gcpAtom::Add(GtkWidget* w)
{
	if (!w) return;
	if (m_Changed > 0) m_Changed--;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	gcpView* pView = pData->View;
	if (m_ascent <= 0)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "l", 1);
		PangoLayoutIter* iter = pango_layout_get_iter(pl);
		m_ascent = pango_layout_iter_get_baseline(iter) / PANGO_SCALE;
		pango_layout_iter_free(iter);
		g_object_unref(pl);
	}
	if (m_CHeight == 0.)
	{
		PangoContext* pc = pData->View->GetPangoContext();
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, "C", 1);
		PangoRectangle rect;
		pango_layout_get_extents(pl, &rect, NULL);
		m_CHeight =  double(rect.height / PANGO_SCALE) / 2.0;
		g_object_unref(G_OBJECT(pl));
	}
	const gchar* FontName = pView->GetSmallFontName();
	GtkTextTag* tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoSmallFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	FontName = pView->GetFontName();
	tag = gtk_text_tag_table_lookup(TextTagTable, FontName);
	if (!tag)
	{
		tag = gtk_text_tag_new(FontName);
		PangoFontDescription* pfd = pView->GetPangoFontDesc();
		g_object_set(G_OBJECT(tag),
							"family", pango_font_description_get_family(pfd),
							"size", pango_font_description_get_size(pfd),
							NULL);
		gtk_text_tag_table_add(TextTagTable, tag);
		g_object_unref((GObject*)tag);
	}
	double x, y, xc, yc;
	m_width =  m_height = 2.0 * pData->Padding;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	double dFontHeight = pData->View->GetFontHeight();
	char * szFontName = pData->View->GetFontName();
	GnomeCanvasItem *item;
	GnomeCanvasGroup *group, *chgp;
	int index;
	char *hs;
	PangoContext* pc = pData->View->GetPangoContext();
	group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(pData->Group, gnome_canvas_group_ext_get_type(), NULL));
	g_signal_connect(G_OBJECT(group), "event", G_CALLBACK(on_event), w);
	g_object_set_data(G_OBJECT(group), "object", this);
	if ((GetZ() != 6) || (GetBondsNumber() == 0))
	{
		GtkTextIter start, end;
		int sw, hw = 0, nw = 0;
		gtk_text_buffer_get_bounds(m_buf, &start, &end);
		gtk_text_buffer_delete(m_buf, &start, &end);
		const gchar* symbol = GetSymbol();
		gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, symbol, strlen(symbol),pView->GetFontName(), NULL);
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, symbol, strlen(symbol));
		PangoRectangle rect;
		pango_layout_get_extents(pl, NULL, &rect);
		m_width = sw = rect.width / PANGO_SCALE;
		m_height = rect.height / PANGO_SCALE;
		m_text_height = dFontHeight;
		int n = GetAttachedHydrogens();
		if (n > 0)
		{
			pango_layout_set_text(pl, "H", 1);
			PangoRectangle rect;
			pango_layout_get_extents(pl, NULL, &rect);
			hw = rect.width / PANGO_SCALE;
			index = (m_HPos)? strlen(symbol): 0;
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index);
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, "H", 1,pView->GetFontName(), NULL);
		}
		g_object_unref(pl);
		if (n > 1)
		{
			pango_context_set_font_description(pc, pData->View->GetPangoSmallFontDesc());
			pl = pango_layout_new(pc);
			hs = g_strdup_printf("%d", n);
			pango_layout_set_text(pl, hs, strlen(hs));
			pango_layout_get_extents(pl, NULL, &rect);
			nw = rect.width / PANGO_SCALE;
			g_object_unref(pl);
			pango_context_set_font_description(pc, pData->View->GetPangoFontDesc());
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index + 1);
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, hs, strlen(hs),pView->GetSmallFontName(), "subscript", NULL);
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index + 1);
			gtk_text_buffer_get_iter_at_offset(m_buf, &end, index + 1 + strlen (hs));
			gtk_text_buffer_remove_tag_by_name(m_buf, pView->GetFontName(), &start, &end);
			m_text_height += 2;
		}
		m_length = sw + hw + nw;
		m_lbearing = (m_HPos)? sw / 2: sw / 2 + hw + nw;

		item = gnome_canvas_item_new(
							group,
							gnome_canvas_rect_ext_get_type(),
							"x1", x - m_lbearing - pData->Padding,
							"y1", y  - m_ascent + m_CHeight - pData->Padding,
							"x2", x - m_lbearing + m_length + pData->Padding,
							"y2", y  - m_ascent + m_CHeight + m_height + pData->Padding,
							"fill_color", (pData->IsSelected(this))? SelectColor: "white",
							NULL);
		g_object_set_data(G_OBJECT(group), "rect", item);
		g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
		g_object_set_data(G_OBJECT(item), "object", this);
		
		item = gnome_canvas_item_new (
							group,
							gnome_canvas_rich_text_ext_get_type(),
							"x", x - m_lbearing,
							"y", y - m_ascent + m_CHeight,
							"width", m_length,
							"height", m_height,
							"grow_height", false,
							"editable", false,
							"cursor_visible", false,
							"text_direction", GTK_TEXT_DIR_LTR,
							NULL);
		gnome_canvas_rich_text_ext_set_buffer(GNOME_CANVAS_RICH_TEXT_EXT(item), m_buf);
		g_object_set_data(G_OBJECT(group), "symbol", item);
		g_object_set_data(G_OBJECT(item), "object", this);
		g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
	}
	else
	{
		item = gnome_canvas_item_new(
								group,
								gnome_canvas_rect_ext_get_type(),
								"x1", x - 3,
								"y1", y - 3,
								"x2", x + 3,
								"y2", y + 3,
								"fill_color",  (pData->IsSelected(this))? SelectColor: "white",
								NULL);
		g_object_set_data(G_OBJECT(group), "rect", item);
		gnome_canvas_request_redraw((GnomeCanvas*)w, (int)x-3, (int)y-3, (int)x+3, (int)y+3);
		gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(group));
		gnome_canvas_item_raise(GNOME_CANVAS_ITEM(group), 1);
		g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), w);
		g_object_set_data(G_OBJECT(item), "object", this);
	}
	pData->Items[this] = group;
	m_width /= pData->ZoomFactor;
	m_height /= pData->ZoomFactor;
	/* add charge */
	int charge = GetCharge ();
	if (charge) {
		int align = GetChargePosition (m_ChargePos, m_ChargeAngle * 180 / M_PI, x, y);
		if (m_ChargeDist != 0.) {
			align = 0;
			x = m_x + m_ChargeDist * cos (m_ChargeAngle);
			y = m_y - m_ChargeDist * sin (m_ChargeAngle);
		}
		x *= pData->ZoomFactor;
		y *= pData->ZoomFactor;
		char *fig = NULL;
		if 	(abs (charge) > 1) {
			fig = g_strdup_printf ("%d", abs (charge));
			PangoContext* pc = pData->View->GetPangoContext();
			PangoRectangle rect;
			pango_context_set_font_description (pc, pData->View->GetPangoSmallFontDesc ());
			PangoLayout* pl = pango_layout_new (pc);
			pango_layout_set_text (pl, fig, -1);
			pango_layout_get_extents (pl, NULL, &rect);
			m_ChargeWidth = rect.width / PANGO_SCALE;
			pango_context_set_font_description (pc, pData->View->GetPangoFontDesc ());
			m_ChargeTWidth = m_ChargeWidth + 1. + ChargeSignSize;
		} else {
			m_ChargeWidth = 0.;
			m_ChargeTWidth = ChargeSignSize;
		}
		switch (align) {
		case -2:
			xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
			y += ChargeSignHeight / 2.;
			break;
		case -1:
			xc = x - ChargeSignSize -1.;
			break;
		case 0:
		case -3:
			xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
			break;
		case 1:
			xc = x + m_ChargeWidth + 1.;
			break;
		case 2:
			xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
			y -= ChargeSignHeight / 2.;
			break;
		}
		x = xc - 1.;
		yc = y - ChargeSignHeight / 2. + ChargeYAlign - ChargeSignSize + 1.;
		chgp = (GnomeCanvasGroup*) gnome_canvas_item_new (
					group,
					gnome_canvas_group_ext_get_type(),
					NULL);
		g_object_set_data (G_OBJECT (group), "charge", chgp);
		if (fig) {
			item = gnome_canvas_item_new(
						chgp,
						gnome_canvas_text_ext_get_type(),
						"fill_color", (pData->IsSelected(this))? SelectColor: Color,
						"font-desc", pData->View->GetPangoSmallFontDesc (),
						"markup", fig,
						"anchor", GTK_ANCHOR_EAST,
						"x", x,
						"y", y,
						NULL);
			g_object_set_data (G_OBJECT (group), "figure", item);
			g_free (fig);
		}
		item = gnome_canvas_item_new (
					chgp,
					gnome_canvas_ellipse_ext_get_type (),
					"x1", xc,
					"y1", yc,
					"x2", xc + ChargeSignSize,
					"y2", yc + ChargeSignSize,
					"outline_color", (pData->IsSelected(this))? SelectColor: Color,
					"width_units", 0.5,
					NULL
				);
		g_object_set_data (G_OBJECT (group), "circle", item);
		ArtBpath *path = art_new (ArtBpath, 5);
		path[0].code = ART_MOVETO_OPEN;
		path[0].x3 = xc + 1.;
		path[1].code = ART_LINETO;
		path[1].x3 = xc + ChargeSignSize - 1.;
		path[0].y3 = path[1].y3 = yc + ChargeSignSize / 2.;
		if (charge > 0) {
			path[2].code = ART_MOVETO_OPEN;
			path[2].y3 = yc + 1.;
			path[3].code = ART_LINETO;
			path[3].y3 = yc + ChargeSignSize - 1.;
			path[2].x3 = path[3].x3 = xc + ChargeSignSize / 2.;
			path[4].code = ART_END;
		} else
			path[2].code = ART_END;
		GnomeCanvasPathDef *cpd = gnome_canvas_path_def_new_from_bpath (path);
		item = gnome_canvas_item_new (
					chgp,
					gnome_canvas_bpath_ext_get_type (),
					"bpath", cpd,
					"outline_color", (pData->IsSelected(this))? SelectColor: Color,
					"width_units", 1.,
					NULL
				);
		gnome_canvas_path_def_unref (cpd);
		g_object_set_data (G_OBJECT (group), "sign", item);
	}
	map<string, Object*>::iterator i;
	Object* electron = GetFirstChild (i);
	while (electron){
		electron->Add (w);
		electron = GetNextChild (i);
	}
}

void gcpAtom::Update(GtkWidget* w)
{
	if (!w) return;
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	double x, y, xc, yc;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	GnomeCanvasGroup *group = pData->Items[this];
	if (m_Changed) {
		BuildItems (pData);
	} else {
		if ((GetZ() != 6) || (GetBondsNumber() == 0)) {
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "symbol")),
								"x", x - m_lbearing,
								"y", y - m_ascent + m_CHeight,
								NULL);
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")),
								"x1", x - m_lbearing - pData->Padding,
								"y1", y  - m_ascent + m_CHeight - pData->Padding,
								"x2", x - m_lbearing + m_length + pData->Padding,
								"y2", y  - m_ascent + m_CHeight + m_text_height + pData->Padding,
								NULL);
		} else
			g_object_set(G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")),
									"x1", x - 3,
									"y1", y - 3,
									"x2", x + 3,
									"y2", y + 3,
								NULL);
	}
	void* item = g_object_get_data (G_OBJECT (group), "charge");
	int charge = GetCharge ();
	if (charge) {
		if (item) {
			int align = GetChargePosition(m_ChargePos, m_ChargeAngle * 180 / M_PI, x, y);
			if (m_ChargeDist != 0.) {
				align = 0;
				x = m_x + m_ChargeDist * cos (m_ChargeAngle);
				y = m_y - m_ChargeDist * sin (m_ChargeAngle);
			}
			x *= pData->ZoomFactor;
			y *= pData->ZoomFactor;
			GnomeCanvasItem *figure = (GnomeCanvasItem*) g_object_get_data (G_OBJECT (group), "figure");
			char *fig = NULL;
			if 	(abs (charge) > 1) {
				fig = g_strdup_printf ("%d", abs (charge));
				PangoContext* pc = pData->View->GetPangoContext();
				PangoRectangle rect;
				pango_context_set_font_description (pc, pData->View->GetPangoSmallFontDesc ());
				PangoLayout* pl = pango_layout_new (pc);
				pango_layout_set_text (pl, fig, -1);
				pango_layout_get_extents (pl, NULL, &rect);
				m_ChargeWidth = rect.width / PANGO_SCALE;
				pango_context_set_font_description (pc, pData->View->GetPangoFontDesc ());
			} else
				m_ChargeWidth = 0.;
			m_ChargeTWidth = m_ChargeWidth + 1. + ChargeSignSize;
			if (figure == NULL && fig != NULL) {
				figure = gnome_canvas_item_new(
							GNOME_CANVAS_GROUP (item),
							gnome_canvas_text_ext_get_type(),
							"fill_color", (pData->IsSelected(this))? SelectColor: Color,
							"font-desc", pData->View->GetPangoSmallFontDesc (),
							"anchor", GTK_ANCHOR_EAST,
							NULL);
				g_object_set_data (G_OBJECT (group), "figure", figure);
			} else if (figure != NULL && fig == NULL) {
				gtk_object_destroy (GTK_OBJECT (figure));
				g_object_set_data ((GObject*)group, "figure", NULL);
			}
			switch (align) {
			case -2:
				xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
				y += ChargeSignHeight / 2.;
				break;
			case -1:
				xc = x - ChargeSignSize -1.;
				break;
			case 0:
			case -3:
				xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
				break;
			case 1:
				xc = x + m_ChargeWidth + 1.;
				break;
			case 2:
				xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
				y -= ChargeSignHeight / 2.;
				break;
			}
			x = xc - 1.;
			yc = y - ChargeSignHeight / 2. + ChargeYAlign - ChargeSignSize + 1.;
			if (fig) {
				g_object_set (G_OBJECT (figure),
							"markup", fig,
							"x", x,
							"y", y,
							NULL);
				g_free (fig);
			}
			item = g_object_get_data (G_OBJECT (group), "circle");
			g_object_set (G_OBJECT (item),
						"x1", xc,
						"y1", yc,
						"x2", xc + ChargeSignSize,
						"y2", yc + ChargeSignSize,
						NULL);
			item = g_object_get_data (G_OBJECT (group), "sign");
			ArtBpath *path = art_new (ArtBpath, 5);
			path[0].code = ART_MOVETO_OPEN;
			path[0].x3 = xc + 1.;
			path[1].code = ART_LINETO;
			path[1].x3 = xc + ChargeSignSize - 1.;
			path[0].y3 = path[1].y3 = yc + ChargeSignSize / 2.;
			if (charge > 0) {
				path[2].code = ART_MOVETO_OPEN;
				path[2].y3 = yc + 1.;
				path[3].code = ART_LINETO;
				path[3].y3 = yc + ChargeSignSize - 1.;
				path[2].x3 = path[3].x3 = xc + ChargeSignSize / 2.;
				path[4].code = ART_END;
			} else
				path[2].code = ART_END;
			GnomeCanvasPathDef *cpd = gnome_canvas_path_def_new_from_bpath (path);
			g_object_set (G_OBJECT (item),
						"bpath", cpd,
						NULL
					);
			gnome_canvas_path_def_unref (cpd);
		} else {
			GnomeCanvasGroup *chgp;
			int align = GetChargePosition(m_ChargePos, m_ChargeAngle * 180 / M_PI, x, y);
			if (m_ChargeDist != 0.) {
				align = 0;
				x = (m_x + m_ChargeDist * cos (m_ChargeAngle));
				y = (m_y - m_ChargeDist * sin (m_ChargeAngle));
			}
			x *= pData->ZoomFactor;
			y *= pData->ZoomFactor;

			char *fig = NULL;
			if 	(abs (charge) > 1) {
				fig = g_strdup_printf ("%d", abs (charge));
				PangoContext* pc = pData->View->GetPangoContext();
				PangoRectangle rect;
				pango_context_set_font_description (pc, pData->View->GetPangoSmallFontDesc ());
				PangoLayout* pl = pango_layout_new (pc);
				pango_layout_set_text (pl, fig, -1);
				pango_layout_get_extents (pl, NULL, &rect);
				m_ChargeWidth = rect.width / PANGO_SCALE;
				pango_context_set_font_description (pc, pData->View->GetPangoFontDesc ());
				m_ChargeTWidth = m_ChargeWidth + 2 + ChargeSignSize;
			} else {
				m_ChargeWidth = 0.;
				m_ChargeTWidth = ChargeSignSize;
			}
			switch (align) {
			case -2:
				xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
				y += ChargeSignHeight / 2.;
				break;
			case -1:
				xc = x - ChargeSignSize -1.;
				break;
			case 0:
			case -3:
				xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
				break;
			case 1:
				xc = x + m_ChargeWidth + 1.;
				break;
			case 2:
				xc = x + m_ChargeTWidth / 2. - ChargeSignSize;
				y -= ChargeSignHeight / 2.;
				break;
			}
			x = xc - 1.;
			yc = y - ChargeSignHeight / 2. + ChargeYAlign - ChargeSignSize + 1.;
			chgp = (GnomeCanvasGroup*) gnome_canvas_item_new (
						group,
						gnome_canvas_group_ext_get_type(),
						NULL);
			g_object_set_data (G_OBJECT (group), "charge", chgp);
			if (fig) {
				item = gnome_canvas_item_new(
							chgp,
							gnome_canvas_text_ext_get_type(),
							"fill_color", (pData->IsSelected(this))? SelectColor: Color,
							"font-desc", pData->View->GetPangoSmallFontDesc (),
							"markup", fig,
							"anchor", GTK_ANCHOR_EAST,
							"x", x,
							"y", y,
							NULL);
				g_object_set_data (G_OBJECT (group), "figure", item);
				g_free (fig);
			}
			item = gnome_canvas_item_new (
						chgp,
						gnome_canvas_ellipse_ext_get_type (),
						"x1", xc,
						"y1", yc,
						"x2", xc + ChargeSignSize,
						"y2", yc + ChargeSignSize,
						"outline_color", (pData->IsSelected(this))? SelectColor: Color,
						"width_units", .5,
						NULL
					);
			g_object_set_data (G_OBJECT (group), "circle", item);
			ArtBpath *path = art_new (ArtBpath, 5);
			path[0].code = ART_MOVETO_OPEN;
			path[0].x3 = xc + 1.;
			path[1].code = ART_LINETO;
			path[1].x3 = xc + ChargeSignSize - 1.;
			path[0].y3 = path[1].y3 = yc + ChargeSignSize / 2.;
			if (charge > 0) {
				path[2].code = ART_MOVETO_OPEN;
				path[2].y3 = yc + 1.;
				path[3].code = ART_LINETO;
				path[3].y3 = yc + ChargeSignSize - 1.;
				path[2].x3 = path[3].x3 = xc + ChargeSignSize / 2.;
				path[4].code = ART_END;
			} else
				path[2].code = ART_END;
			GnomeCanvasPathDef *cpd = gnome_canvas_path_def_new_from_bpath (path);
			item = gnome_canvas_item_new (
						chgp,
						gnome_canvas_bpath_ext_get_type (),
						"bpath", cpd,
						"outline_color", (pData->IsSelected(this))? SelectColor: Color,
						"width_units", 1.,
						NULL
					);
			gnome_canvas_path_def_unref (cpd);
			g_object_set_data (G_OBJECT (group), "sign", item);
		}
	} else {
		if (item)
			gtk_object_destroy (GTK_OBJECT (item));
			g_object_set_data ((GObject*)group, "charge", NULL);
	}
	map<string, Object*>::iterator i;
	Object* electron = GetFirstChild (i);
	while (electron){
		electron->Update (w);
		electron = GetNextChild (i);
	}
}

void gcpAtom::UpdateAvailablePositions ()
{
	list<double>::iterator n;
	double angle, delta, dir;
	m_AngleList.clear ();
	if (((GetZ() != 6 || m_Bonds.size() == 0)) && m_nH)
	{
		if (m_HPos)
		{
			m_AvailPos = 0xB6;
			m_AngleList.push_front(315.0);
			m_AngleList.push_front(45.0);
		}
		else
		{
			m_AvailPos = 0x6D;
			m_AngleList.push_front(225.0);
			m_AngleList.push_front(135.0);
		}
	}
	else m_AvailPos = 0xff;
	m_AvailPos &= ~m_OccupiedPos;
	map<Atom*, Bond*>::iterator i = m_Bonds.begin();
	while (i != m_Bonds.end())
	{
		n = m_AngleList.begin();
		angle = ((gcpBond*)(*i).second)->GetAngle2D(this);
		if (angle < 0)
			angle += 360.;
		while ((n != m_AngleList.end()) && (*n < angle)) n++;
		m_AngleList.insert(n, angle);
		i++;
		if ((m_AvailPos & CHARGE_SW) && (angle >= 180.0) && (angle <= 270.0)) m_AvailPos -= CHARGE_SW;
		if ((m_AvailPos & CHARGE_SE) && (((angle >= 270.0) && (angle <= 360.0)) || (fabs(angle) < 0.1))) m_AvailPos -= CHARGE_SE;
		if ((m_AvailPos & CHARGE_S) && (angle >= 225.0) && (angle <= 315.0)) m_AvailPos -= CHARGE_S;
		if ((m_AvailPos & CHARGE_NW) && (angle >= 90.0) && (angle <= 180.0)) m_AvailPos -= CHARGE_NW;
		if ((m_AvailPos & CHARGE_NE) && (((angle >= 0.0) && (angle <= 90.0)) || (fabs(angle - 360.0) < 0.1))) m_AvailPos -= CHARGE_NE;
		if ((m_AvailPos & CHARGE_N) && (angle >= 45.0) && (angle <= 135.0)) m_AvailPos -= CHARGE_N;
		if ((m_AvailPos & CHARGE_W) && ((angle <= 225.0) && (angle >= 135.0))) m_AvailPos -= CHARGE_W;
		if ((m_AvailPos & CHARGE_E) && ((angle >= 315.0) || (angle <= 45.0))) m_AvailPos -= CHARGE_E;
	}
	m_AngleList.push_back ((angle = m_AngleList.front ()) + 360.0);
	m_InterBonds.clear ();
	for (n = m_AngleList.begin (), n++; n != m_AngleList.end (); n++) {
		delta = *n - angle;
		while (m_InterBonds.find (delta) != m_InterBonds.end ())
			delta -= 1e-8;
		dir = (*n + angle) / 2.;
		if ((m_AvailPos == 0xff) || (m_HPos && (dir < 135. || dir > 225.)) ||
			(!m_HPos && (dir > 45. && dir < 315.)))
			m_InterBonds[delta] = dir;
		angle = *n;
	}
	m_AvailPosCached = true;
}

int gcpAtom::GetChargePosition(unsigned char& Pos, double Angle, double& x, double& y)
{
	list<double>::iterator n;
	double angle;
	if (m_ChargePos != 0xff)
		m_OccupiedPos &= ~m_ChargePos;
	if (!m_AvailPosCached)
		UpdateAvailablePositions ();
	if (m_ChargePos != 0xff)
		m_OccupiedPos |= m_ChargePos;
	if (!m_ChargeAutoPos && Pos == 0xff) {
		Pos = m_ChargePos;
		if (!Pos)
			Angle = m_ChargeAngle * 180 / M_PI;
	} else if (Pos == 0xff) {
		if (m_AvailPos) {
			if (m_AvailPos & CHARGE_NE)
				Pos = CHARGE_NE;
			else if (m_AvailPos & CHARGE_NW)
				Pos = CHARGE_NW;
			else if (m_AvailPos & CHARGE_N)
				Pos = CHARGE_N;
			else if (m_AvailPos & CHARGE_SE)
				Pos = CHARGE_SE;
			else if (m_AvailPos & CHARGE_SW)
				Pos = CHARGE_SW;
			else if (m_AvailPos & CHARGE_S)
				Pos = CHARGE_S;
			else if (m_AvailPos & CHARGE_E)
				Pos = CHARGE_E;
			else if (m_AvailPos & CHARGE_W)
				Pos = CHARGE_W;
		} else {
			Pos = 0;
			angle = m_AngleList.front();
			double max = 0.0;
			//if we are there, there are at least two bonds
			for (n = m_AngleList.begin(), n++; n != m_AngleList.end(); n++)
			{
				if (*n - angle > max)
				{
					if (*n - angle - max > 0.1) x = (*n + angle) / 2;
					if (m_nH)
					{
						if (m_HPos && ((x > 225.0) || (x < 135.0))) Angle = x;
						else if (m_HPos && (x > 45.0) && (x < 315.0)) Angle = x;
					}
					else Angle = x;
					max = *n - angle;
				}
				angle = *n;
			}
		} 
	} else if (Pos) {
		if (!(Pos & m_AvailPos) && (Pos != m_ChargePos))
			return 0;
	} else {
		if (Angle > 360.)
			Angle -= 360;
		else if (Angle < 0.)
			Angle += 360;
		if (!(((GetZ() == 6) && (m_Bonds.size() != 0)) ||
			 !m_nH || ((!m_HPos && (Angle < 135. || Angle > 225.)) ||
				(m_HPos && (Angle > 45. && Angle < 315.)))))
			return 0;
	}
	switch (Pos) {
	case CHARGE_NE:
		x = m_x + m_width / 2.0;
		y = m_y - m_height / 2.0;
		return 1;
	case CHARGE_NW:
		x = m_x - m_width / 2.0;
		y = m_y - m_height / 2.0;
		return -1;
	case CHARGE_N:
		x = m_x;
		y = m_y - m_height / 2.0;
		return 2;
	case CHARGE_SE:
		x = m_x + m_width / 2.0;
		y = m_y + m_height / 2.0;
		return 1;
	case CHARGE_SW:
		x = m_x - m_width / 2.0;
		y = m_y + m_height / 2.0;
		return -1;
	case CHARGE_S:
		x = m_x;
		y = m_y + m_height / 2.0;
		return -2;
	case CHARGE_E:
		x = m_x + 12. + m_width / 2.0;
		y = m_y;
		return 1;
	case CHARGE_W:
		x = m_x - 12. - m_width / 2.0;
		y = m_y;
		return -1;
	default: {
			double t = tan (Angle / 180. * M_PI);
			double limit = atan (m_height / m_width) * 180. / M_PI;
			if (Angle < limit) {
				x = m_x +  12. + m_width / 2.;
				y = m_y - m_width / 2. * t;
				return 1;
			} else if (Angle < 180. - limit) {
				if (!isnan (t))
					x = m_x + m_height / 2. / t;
				else
					x = m_x;
				y = m_y - m_height / 2.;
				return 2;
			} else if (Angle < 180. + limit) {
				x = m_x - 12. - m_width / 2.;
				y = m_y + m_width / 2. * t;
				return -1;
			} else if (Angle < 360. - limit) {
				if (!isnan (t))
					x = m_x - m_height / 2. / t;
				else
					x = m_x;
				y = m_y + m_height / 2.;
				return -2;
			} else {
				x = m_x +  12. + m_width / 2.;
				y = m_y - m_width / 2. * t;
				return 1;
			}
		}			
	}
	return 0; // should not occur
}

int gcpAtom::GetAvailablePosition(double& x, double& y)
{
	list<double>::iterator n;
	double angle;
	if (!m_AvailPosCached)
		UpdateAvailablePositions ();
	if (m_AvailPos)
	{
		if (m_AvailPos & POSITION_N)
		{
			x = m_x;
			y = m_y - m_height / 2.0;
			return POSITION_N;
		}
		if (m_AvailPos & POSITION_S)
		{
			x = m_x;
			y = m_y + m_height / 2.0;
			return POSITION_S;
		}
		if (m_AvailPos & POSITION_E)
		{
			x = m_x + m_width / 2.0;
			y = m_y;
			return POSITION_E;
		}
		if (m_AvailPos & POSITION_W)
		{
			x = m_x - m_width / 2.0;
			y = m_y;
			return POSITION_W;
		}
		if (m_AvailPos & POSITION_NE)
		{
			x = m_x + m_width / 2.0;
			y = m_y - m_height / 2.0;
			return POSITION_NE;
		}
		if (m_AvailPos & POSITION_NW)
		{
			x = m_x - m_width / 2.0;
			y = m_y - m_height / 2.0;
			return POSITION_NW;
		}
		if (m_AvailPos & POSITION_SE)
		{
			x = m_x + m_width / 2.0;
			y = m_y + m_height / 2.0;
			return POSITION_SE;
		}
		if (m_AvailPos & POSITION_SW)
		{
			x = m_x - m_width / 2.0;
			y = m_y + m_height / 2.0;
			return POSITION_SW;
		}
	}
	angle = m_AngleList.front();
	double dir = 0.0, max = 0.0;
	//if we are there, there are at least two bonds
	for (n = m_AngleList.begin(), n++; n != m_AngleList.end(); n++)
	{
		if (*n - angle > max)
		{
			if (*n - angle - max > 0.1) x = (*n + angle) / 2;
			if (m_nH)
			{
				if (m_HPos && ((x > 225.0) || (x < 135.0))) dir = x;
				else if (m_HPos && (x > 45.0) && (x < 315.0)) dir = x;
			}
			else dir = x;
			max = *n - angle;
		}
		angle = *n;
	}
	max = sqrt(square(m_width) + square(m_height)) / 2.0 + 24;//Could do better, should replace 24 by something more intelligent
	x = m_x + max * cos(dir / 180.0 * M_PI);
	y = m_y - max * sin(dir / 180.0 * M_PI);
	return 0;
}

bool gcpAtom::LoadNode(xmlNodePtr)
{
	SetZ(GetZ());
	return true;
}

void gcpAtom::SetSelected(GtkWidget* w, int state)
{
	gcpWidgetData* pData = (gcpWidgetData*)g_object_get_data(G_OBJECT(w), "data");
	GnomeCanvasGroup* group = pData->Items[this];
	gpointer item;
	gchar *color, *chargecolor;
	
	switch (state)
	{	
		case SelStateUnselected: color = "white"; chargecolor = "black"; break;
		case SelStateSelected: chargecolor = color = SelectColor; break;
		case SelStateUpdating: chargecolor = color = AddColor; break;
		case SelStateErasing: chargecolor = color = DeleteColor; break;
	}
	g_object_set (G_OBJECT(g_object_get_data(G_OBJECT(group), "rect")),
				"fill_color", color, NULL);
	if (item = g_object_get_data(G_OBJECT(group), "figure"))
		g_object_set (item, "fill_color", chargecolor, NULL);
	if (item = g_object_get_data(G_OBJECT(group), "circle"))
		g_object_set (item, "outline_color", chargecolor, NULL);
	if (item = g_object_get_data(G_OBJECT(group), "sign"))
		g_object_set (item, "outline_color", chargecolor, NULL);
	Object::SetSelected (w, state);
}

bool gcpAtom::AcceptNewBonds(int nb)
{
	if ((m_Valence > 0) || m_ChargeAuto)
		return Element::GetMaxBonds(m_Z) >= (GetTotalBondsNumber() + GetChildrenNumber () + nb);
	map<string, Object*>::iterator i;
	gcpElectron* electron = (gcpElectron*) GetFirstChild (i);
	int nel = 0;
	while (electron){ 
		if (electron->IsPair ())
			nel += 2;
		else
			nel++;
		electron = (gcpElectron*) GetNextChild (i);
	}
	nel += GetTotalBondsNumber ();
	return (m_ValenceOrbitals - GetTotalBondsNumber () - GetChildrenNumber () > 0)
			&& (((m_Element->GetValenceElectrons() - m_Charge) > nel) || m_ChargeAuto);
}

void gcpAtom::AddToMolecule(gcpMolecule* Mol)
{
	Mol->AddAtom(this);
}

void gcpAtom::BuildItems (gcpWidgetData* pData)
{
	GnomeCanvasGroup* group = pData->Items[this];
	void* item;
	GtkTextIter start, end;
	double dFontHeight = pData->View->GetFontHeight();
	PangoContext* pc = pData->View->GetPangoContext();
	double x, y;
	m_width =  m_height = 2.0 * pData->Padding;
	GetCoords(&x, &y);
	x *= pData->ZoomFactor;
	y *= pData->ZoomFactor;
	if ((GetZ() != 6) || (GetBondsNumber() == 0))
	{
		int sw, hw = 0, nw = 0;
		int index;
		char *hs;
		gtk_text_buffer_get_bounds(m_buf, &start, &end);
		gtk_text_buffer_delete(m_buf, &start, &end);
		const gchar* symbol = GetSymbol();
		gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, symbol, strlen(symbol),pData->View->GetFontName(), NULL);
		PangoLayout *pl = pango_layout_new(pc);
		pango_layout_set_text(pl, symbol, strlen(symbol));
		PangoRectangle rect;
		pango_layout_get_extents(pl, NULL, &rect);
		m_width = sw = rect.width / PANGO_SCALE;
		m_height = rect.height / PANGO_SCALE;
		m_text_height = dFontHeight;
		int n = GetAttachedHydrogens();
		if (n > 0)
		{
			pango_layout_set_text(pl, "H", 1);
			PangoRectangle rect;
			pango_layout_get_extents(pl, NULL, &rect);
			hw = rect.width / PANGO_SCALE;
			index = (m_HPos)? strlen(symbol): 0;
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index);
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, "H", 1,pData->View->GetFontName(), NULL);
		}
		g_object_unref(pl);
		if (n > 1)
		{
			pango_context_set_font_description(pc, pData->View->GetPangoSmallFontDesc());
			pl = pango_layout_new(pc);
			hs = g_strdup_printf("%d", n);
			pango_layout_set_text(pl, hs, strlen(hs));
			pango_layout_get_extents(pl, NULL, &rect);
			nw = rect.width / PANGO_SCALE;
			g_object_unref(pl);
			pango_context_set_font_description(pc, pData->View->GetPangoFontDesc());
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index + 1);
			gtk_text_buffer_insert_with_tags_by_name(m_buf, &start, hs, strlen(hs),pData->View->GetSmallFontName(), "subscript", NULL);
			gtk_text_buffer_get_iter_at_offset(m_buf, &start, index + 1);
			gtk_text_buffer_get_iter_at_offset(m_buf, &end, index + 1 + strlen (hs));
			gtk_text_buffer_remove_tag_by_name(m_buf, pData->View->GetFontName(), &start, &end);
			m_text_height += 2;
		}
		m_length = sw + hw + nw;
		m_lbearing = (m_HPos)? sw / 2: sw / 2 + hw + nw;

		item = g_object_get_data (G_OBJECT (group), "rect");
		g_object_set (item,
							"x1", x - m_lbearing - pData->Padding,
							"y1", y  - m_ascent + m_CHeight - pData->Padding,
							"x2", x - m_lbearing + m_length + pData->Padding,
							"y2", y  - m_ascent + m_CHeight + m_height + pData->Padding,
							NULL);
		
		item = g_object_get_data (G_OBJECT (group), "symbol");
		if (item)
			g_object_set (item,
							"x", x - m_lbearing,
							"y", y - m_ascent + m_CHeight,
							"width", m_length,
							"height", m_height,
							NULL);
		else {
			item = gnome_canvas_item_new (
							group,
							gnome_canvas_rich_text_ext_get_type(),
							"x", x - m_lbearing,
							"y", y - m_ascent + m_CHeight,
							"width", m_length,
							"height", m_height,
							"grow_height", false,
							"editable", false,
							"cursor_visible", false,
							"text_direction", GTK_TEXT_DIR_LTR,
							NULL);
			gnome_canvas_rich_text_ext_set_buffer(GNOME_CANVAS_RICH_TEXT_EXT(item), m_buf);
			g_object_set_data(G_OBJECT(group), "symbol", item);
			g_object_set_data(G_OBJECT(item), "object", this);
			g_signal_connect(G_OBJECT(item), "event", G_CALLBACK(on_event), pData->Canvas);
			gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(group));
		}
	}
	else
	{
		item = g_object_get_data (G_OBJECT (group), "rect");
		g_object_set (item,
							"x1", x - 3,
							"y1", y - 3,
							"x2", x + 3,
							"y2", y + 3,
							NULL);
		item = g_object_get_data (G_OBJECT (group), "symbol");
		if (item){
			gtk_object_destroy (GTK_OBJECT (item));
			g_object_set_data (G_OBJECT (group), "symbol", NULL);
		}
		m_length = m_text_height = 0;
	}
	m_width /= pData->ZoomFactor;
	m_height /= pData->ZoomFactor;
	if (m_Changed > 0) m_Changed--;
}
	
double gcpAtom::GetYAlign ()
{
	return m_y;
}

bool gcpAtom::HasImplicitElectronPairs ()
{
	map<string, Object*>::iterator i;
	gcpElectron* electron = (gcpElectron*) GetFirstChild (i);
	if (m_Valence > 0) {
		int nexplp = 0; //nexplp is the number of explicit lone pairs
		while (electron){ 
			if (electron->IsPair ())
				nexplp++;
			electron = (gcpElectron*) GetNextChild (i);
		}
		return (m_nlp > nexplp);
	}
	int nel = 0;
	while (electron){ 
		if (electron->IsPair ())
			nel += 2;
		else
			nel++;
		electron = (gcpElectron*) GetNextChild (i);
	}
	nel += GetTotalBondsNumber ();
	int nocc = GetChildrenNumber () + GetTotalBondsNumber ();
	return (nocc < m_ValenceOrbitals) && (((m_Element->GetValenceElectrons() - m_Charge) > nel + 1) || m_ChargeAuto);
}

bool gcpAtom::MayHaveImplicitUnpairedElectrons ()
{
	map<string, Object*>::iterator i;
	gcpElectron* electron = (gcpElectron*) GetFirstChild (i);
	int nel = 0;
	while (electron){ 
		if (electron->IsPair ())
			nel += 2;
		else
			nel++;
		electron = (gcpElectron*) GetNextChild (i);
	}
	nel += GetTotalBondsNumber ();
	return (m_ValenceOrbitals - GetTotalBondsNumber () - GetChildrenNumber () > 0)
			&& (((m_Element->GetValenceElectrons() - m_Charge) > nel) || m_ChargeAuto);
}

bool gcpAtom::GetPosition(double angle, double& x, double& y)
{
	if (angle > 360.)
		angle -= 360;
	else if (angle < 0.)
		angle += 360;
	if (((GetZ() == 6) && (m_Bonds.size() != 0)) ||
		 !m_nH || ((!m_HPos && (angle < 135. || angle > 225.)) ||
			(m_HPos && (angle > 45. && angle < 315.)))) {
		double t = tan (angle / 180. * M_PI);
		double limit = atan (m_height / m_width) * 180. / M_PI;
		if (angle < limit) {
			x = m_x +  12. + m_width / 2.;
			y = m_y - m_width / 2. * t;
		} else if (angle < 180. - limit) {
			if (!isnan (t))
				x = m_x + m_height / 2. / t;
			else
				x = m_x;
			y = m_y - m_height / 2.;
		} else if (angle < 180. + limit) {
			x = m_x - 12. - m_width / 2.;
			y = m_y + m_width / 2. * t;
		} else if (angle < 360. - limit) {
			if (!isnan (t))
				x = m_x - m_height / 2. / t;
			else
				x = m_x;
			y = m_y + m_height / 2.;
		} else {
			x = m_x +  12. + m_width / 2.;
			y = m_y - m_width / 2. * t;
		}			
		return true;
	}
	return false;
}

void gcpAtom::AddElectron (gcpElectron* electron)
{
	AddChild (electron);
	Update ();
}

void gcpAtom::NotifyPositionOccupation (unsigned char pos, bool occupied)
{
	if (occupied)
		m_OccupiedPos |= pos;
	else
		m_OccupiedPos &= ~pos;
}
	
xmlNodePtr gcpAtom::Save (xmlDocPtr xml)
{
	xmlNodePtr node = Atom::Save (xml), child;
	if (node) {
	// Save electrons
		map<string, Object*>::iterator i;
		gcpElectron* electron = (gcpElectron*) GetFirstChild (i);
		while (electron){ 
			child = electron->Save (xml);
			if (child)
				xmlAddChild (node, child);
			electron = (gcpElectron*) GetNextChild (i);
		}
	}
	if (m_Charge && !m_ChargeAutoPos) {
		char *buf;
		if (m_ChargePos) {
			switch (m_ChargePos) {
			case CHARGE_NE:
				buf = "ne";
				break;
			case CHARGE_NW:
				buf = "nw";
				break;
			case CHARGE_N:
				buf = "n";
				break;
			case CHARGE_SE:
				buf = "se";
				break;
			case CHARGE_SW:
				buf = "sw";
				break;
			case CHARGE_S:
				buf = "s";
				break;
			case CHARGE_E:
				buf = "e";
				break;
			case CHARGE_W:
				buf = "w";
				break;
			default:
				buf = "def"; // should not occur
			}
			xmlNewProp (node, (xmlChar*) "charge-position", (xmlChar*) buf);
		} else {
			buf = g_strdup_printf ("%g", m_ChargeAngle * 180. / M_PI);
			xmlNewProp (node, (xmlChar*) "charge-angle", (xmlChar*) buf);
			g_free (buf);
		}
		if (m_ChargeDist != 0.) {
			buf = g_strdup_printf ("%g", m_ChargeDist);
			xmlNewProp (node, (xmlChar*) "charge-dist", (xmlChar*) buf);
			g_free (buf);
		}
	}
	return node;
}
	
bool gcpAtom::Load (xmlNodePtr node)
{
	if (!Atom::Load (node))
		return false;
	//Load electrons
	xmlNodePtr child = node->children;
	gcpElectron *electron;
	while (child) {
		electron = NULL;
		if (!strcmp ((char*) child->name, "electron"))
			electron = new gcpElectron (this, false);
		else if (!strcmp ((char*) child->name, "electron-pair"))
			electron = new gcpElectron (this, true);
		if (electron && !electron->Load (child))
			return false;
		child = child->next;
	}
	char *buf = (char*) xmlGetProp (node, (xmlChar*) "charge-position");
	m_ChargePos = 0xff;
	if (buf) {
		if (! strcmp (buf, "ne")) {
			m_ChargePos = CHARGE_NE;
			m_ChargeAngle = M_PI / 4.;
		} else if (! strcmp (buf, "nw")) {
			m_ChargePos = CHARGE_NW;
			m_ChargeAngle = 3. * M_PI / 4.;
		} else if (! strcmp (buf, "n")) {
			m_ChargePos = CHARGE_N;
			m_ChargeAngle = M_PI / 2.;
		} else if (! strcmp (buf, "se")) {
			m_ChargePos = CHARGE_SE;
			m_ChargeAngle = 7. * M_PI / 4;
		} else if (! strcmp (buf, "sw")) {
			m_ChargePos = CHARGE_SW;
			m_ChargeAngle = 5. * M_PI / 4;
		} else if (! strcmp (buf, "s")) {
			m_ChargePos = CHARGE_S;
			m_ChargeAngle = 3 * M_PI / 2.;
		} else if (! strcmp (buf, "e")) {
			m_ChargePos = CHARGE_E;
			m_ChargeAngle = 0.;
		} else if (! strcmp (buf, "w")) {
			m_ChargePos = CHARGE_W;
			m_ChargeAngle = M_PI;
		}
		m_ChargeAutoPos = false;
		xmlFree (buf);
	} else {
		buf = (char*) xmlGetProp(node, (xmlChar*)"charge-angle");
		if (buf) {
			sscanf(buf, "%lg", &m_ChargeAngle);
			m_ChargeAngle *= M_PI / 180.;
			xmlFree (buf);
			m_ChargePos = 0;
			m_ChargeAutoPos = false;
		}
	}
	buf = (char*) xmlGetProp(node, (xmlChar*)"charge-dist");
	if (buf) {
		sscanf(buf, "%lg", &m_ChargeDist);
		xmlFree (buf);
		m_ChargeAutoPos = false;
	} else
		m_ChargeDist = 0.;
	return true;
}
	
bool gcpAtom::AcceptCharge (int charge) {
	int nb = GetTotalBondsNumber (), ne = 0;
	map<string, Object*>::iterator i;
	gcpElectron* electron = (gcpElectron*) GetFirstChild (i);
	while (electron){ 
		if (electron->IsPair ())
			ne += 2;
		else
			ne++;
		electron = (gcpElectron*) GetNextChild (i);
	}
	if (charge < 0)
		return (m_Element->GetTotalValenceElectrons () <= m_Element->GetMaxValenceElectrons () + charge - nb - 2 * GetChildrenNumber () + ne);
	if (nb)
		return (m_Element->GetValenceElectrons () >= charge + nb + ne);
	return (charge <= GetZ ());
}

void gcpAtom::SetChargePosition (unsigned char Pos, bool def, double angle, double distance)
{
	if (Pos != m_ChargePos) {
		m_ChargeAutoPos = def;
		if (m_ChargePos > 0)
			NotifyPositionOccupation (m_ChargePos, false);
		m_ChargePos = Pos;
		if (m_ChargePos > 0)
			NotifyPositionOccupation (m_ChargePos, true);
	}
	m_ChargeAngle = angle;
	m_ChargeDist = distance;
	m_AvailPosCached = false;
}

char gcpAtom::GetChargePosition (double *Angle, double *Dist)
{
	if (Angle)
		*Angle = m_ChargeAngle;
	if (Dist)
		*Dist = m_ChargeDist;
	return (m_ChargeAutoPos)? -1: m_ChargePos;
}

void gcpAtom::SetCharge (int charge)
{
	gcu::Atom::SetCharge (charge);
	m_ChargeAuto = false;
	Update ();
}
