// -*- C++ -*-

/* 
 * GChemPaint library
 * document.h 
 *
 * Copyright (C) 2001-2005
 *
 * Developed by Jean Bréfort <jean.brefort@normalesup.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#ifndef GCHEMPAINT_DOCUMENT_H
#define GCHEMPAINT_DOCUMENT_H

#include <list>
#include <map>
#include <set>
#include <string>
#include <libxml/tree.h>
#include <gtk/gtk.h>
#include <libgnomeprint/gnome-print.h>
#include <libbonobo.h>
#include <libbonoboui.h>
#include <gcu/document.h>
#include "atom.h"
#include "fragment.h"
#include "bond.h"
#include "molecule.h"
#include "operation.h"

enum gcpFileTypes
{
	gcp_ft_Normal = 1,
	gcp_ft_CML,
	gcp_ft_SDF,
	gcp_ft_PDB,
	gcp_ft_ChemDraw,
	gcp_ft_max
};

namespace OpenBabel
{
	class OBMol;
}

using namespace OpenBabel;
using namespace std;

extern SignalId OnChangedSignal;
extern SignalId OnDeleteSignal;

class gcpView;
class gcpApplication;
class gcpDocPropDlg;
class gcpDocument: public Document
{
	//Constructor and destructor
public:
	gcpDocument(gcpApplication *App, bool StandAlone);
	virtual ~gcpDocument();
	
	//Interface
public:
	GtkWidget* GetWidget();
	gcpView* GetView() {return m_pView;}
	void BuildBondList(list<gcpBond*>& BondList, Object* obj);
	bool ImportOB(OBMol& Mol);
	void ExportOB();
	void BuildAtomTable(map<string, unsigned>& AtomTable, Object* obj, unsigned& index);
	void SetType(int Type) {m_FileType = Type; m_bWriteable = true;}
	void Save();
	virtual bool Load(xmlNodePtr);
	const gchar* GetTitle();
	void SetTitle(const gchar* title);
	const gchar* GetLabel();
	void SetFileName(const gchar* Name, const gchar* ext);
	const gchar* GetFileName() {return m_filename;}
	void Print(GnomePrintContext *pc, gdouble width, gdouble height);
	void AddObject(Object* pObject);
	void AddAtom(gcpAtom* pAtom);
	void AddFragment(gcpFragment* pFragment);
	void AddBond(gcpBond* pBond);
	void ParseXMLTree(xmlDocPtr xml);
	void LoadObjects(xmlNodePtr node);
	xmlDocPtr BuildXMLTree();
	void NotifyDirty (gcpBond* pBond) {m_DirtyObjects.insert (pBond);}
	void Update();
	void Remove(Object*);
	void Remove(const char* Id);
	void OnProperties();
	void OnCloseProperties() {m_DocPropDlg = NULL;}
	void OnUndo();
	void OnRedo();
	const GDate* GetCreationDate() {return &CreationDate;}
	const GDate* GetRevisionDate() {return &RevisionDate;}
	const gchar* GetAuthor() {return m_author;}
	const gchar* GetMail() {return m_mail;}
	const gchar* GetComment() {return m_comment;}
	void SetAuthor(const gchar* author);
	void SetMail(const gchar* mail);
	void SetComment(const gchar* comment);
	void SetBonoboPersist(BonoboPersist* ps) {m_ps = ps;}
	void FinishOperation();
	void AbortOperation();
	void PopOperation();
	void PushOperation(gcpOperation* operation, bool undo = true);
	void SetActive();
	gcpOperation* GetNewOperation(gcpOperationType type);
	gcpOperation* GetCurrentOperation () {return m_pCurOp;}
	void AddData(xmlNodePtr node);
	bool CanUndo() {return m_UndoList.size() > 0;}
	void SetBonoboUI(BonoboUIComponent *uic) {m_pUic = uic;}
	BonoboUIComponent * GetBonoboUI() {return m_pUic;}
	void SetEditable(bool editable) {m_bWriteable = editable; m_bUndoRedo = true;}
	bool GetEditable() {return m_bWriteable;}
	gcpApplication* GetApplication() {return m_pApp;}
	bool IsDirty () {return m_bDirty;}
	void ExportImage (const char* filename, const char* type);
	void SetReadOnly (bool ro);
	virtual double GetYAlign ();

private:
	void RemoveAtom(gcpAtom* pAtom);
	void RemoveBond(gcpBond* pBond);
	void RemoveFragment(gcpFragment* pFragment);

	//Implementation
private:
	gcpView * m_pView;
	gchar* m_filename;
	gchar *m_title;
	gchar *m_label;
	gchar *m_comment, *m_author, *m_mail;
	set<Object*>m_DirtyObjects;
	BonoboPersist* m_ps;
	bool m_bDirty, m_bIsLoading, m_bUndoRedo, m_bReadOnly;
	int m_FileType;
	bool m_bWriteable;
	gcpDocPropDlg* m_DocPropDlg;
	GDate CreationDate, RevisionDate;
	list<gcpOperation*> m_UndoList, m_RedoList;
	gcpOperation* m_pCurOp;
	BonoboUIComponent* m_pUic;
	gcpApplication* m_pApp;
};

extern list<gcpDocument*> Docs;
extern bool bCloseAll;

#endif // GCHEMPAINT_DOCUMENT_H
