/* AppletViewer.java -- a standalone viewer for Java applets
   Copyright (C) 2003, 2004  Thomas Fitzsimmons <fitzsim@redhat.com>

   This file is part of GCJ Applet Viewer.

   GCJ Applet Viewer is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   GCJ Applet Viewer is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GCJ Applet Viewer; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package gnu.gcjwebplugin;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.applet.Applet;
import java.awt.Dimension;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.MissingResourceException;
import java.util.LinkedList;
import java.util.List;
import java.util.ResourceBundle;
import java.util.Vector;


class AppletViewer
{
  /**
   * The localized strings are kept in a separate file
   */
  public static ResourceBundle messages = ResourceBundle.getBundle
    ("gnu.gcjwebplugin.locale.MessagesBundle");

  private static HashMap classLoaderCache = new HashMap();
  
  private static ClassLoader getClassLoader(URL codebase, ArrayList archives)
  {
    // Ask cache for existing class loader.
    AppletClassLoader loader = (AppletClassLoader) classLoaderCache.get(codebase);

    if (loader == null)
      {
	// FIXME: why does URLClassLoader throw a NoClassDefFoundError
	// when the html file we pass it is in the current working
	// directory?  (e.g. the Slime Volleyball applet).  I suspect
	// a URLClassLoader bug in libgcj.

	// XXX: Add archives only the first time we see a codebase ...
	loader = new AppletClassLoader(codebase, archives);

	// Add new class loader to cache.
	classLoaderCache.put(codebase, loader);
      }
    
    return loader;
  }

  static Applet createApplet(AppletTag tag)
  {
    Applet applet = null;

    try
      {
        ClassLoader loader = getClassLoader(tag.prependCodebase (""),
						  tag.archives);
	String code = tag.code;

	if (code.endsWith(".class"))
	  code = code.substring(0, code.length() - 6);

	Class c = loader.loadClass(code);
	applet = (Applet) c.newInstance();
      }
    catch (Exception e)
      {
	e.printStackTrace();
      }
    
    return applet;
  }

  static void printVersion()
  {
    System.out.println("gcjappletviewer (GCJ Applet Viewer) "
                       + Configuration.VERSION);
    System.exit(0);
  }

  static void printHelpMessage(int status)
  {
    try
      {
	for (int count = 0; ; count++)
	  System.out.println
	    (AppletViewer.messages.getString("gcjwebplugin.help." + count));
      }
    catch (MissingResourceException e)
      {
	// Do nothing here.
      }
    
    System.exit(status);
  }

  /**
   * The main method starting the applet viewer.
   *
   * @param args the arguments given on the command line.
   *
   * @exception IOException if an error occurs.
   */
  public static void main(String[] args) throws IOException
  {
    String encoding = "UTF8";

    if (args.length == 0)
      {
	System.err.println(AppletViewer.messages.getString
			   ("gcjwebplugin.no_input_files"));
	printHelpMessage(1);
      }

    List argsList = new LinkedList(Arrays.asList(args));

    // Check for the -debug option, for appletviewer compatibility.
    // The -debug option can be given multiple times on the command
    // line.

    // FIXME: The debug option will be unsupported until we have the
    // ability to debug bytecode.  For now, just strip it out of the
    // argument list.
    Iterator argsIter = argsList.iterator();
    while (argsIter.hasNext())
      {
	String currentArg = (String) argsIter.next();
	if (currentArg.equals("--"))
	  break;
	else if (currentArg.equals("-debug"))
	  argsIter.remove();
      }

    // Check for the -encoding option, for appletviewer compatibility.
    // The -encoding option can be given multiple times on the command
    // line.  The last occurence wins.

    // FIXME: We should probably be using
    // java.nio.charset.CharsetDecoder to handle the encoding.  What
    // is the status of Classpath's implementation?
    argsIter = argsList.iterator();
    while (argsIter.hasNext())
      {
	String currentArg = (String) argsIter.next();
	if (currentArg.equals("--"))
	  break;
	else if (currentArg.equals("-encoding"))
	  {
	    if (! argsIter.hasNext())
	      {
		System.err.println("gcjappletviewer: option '-encoding' requires an argument");
		printHelpMessage(1);
	      }

	    // Remove the -encoding option.
	    argsIter.remove();
	    encoding = (String) argsIter.next();
	    // Remove the argument to -encoding.
	    argsIter.remove();
	  }
      }

    // Check for -J options, for appletviewer compatibility.

    // FIXME: currently we ignore these options.  What are their gcj
    // equivalents and how do we pass them to the runtime?
    argsIter = argsList.iterator();
    while (argsIter.hasNext())
      {
	String currentArg = (String) argsIter.next();

	if (currentArg.length() == 2)
	  {
	    if (currentArg.equals("--"))
	      break;
	    else if (currentArg.equals("-J"))
	      {
		System.err.println("gcjappletviewer: the -J option should not be followed by a space.");
		printHelpMessage(1);
	      }
	  }
	else if (currentArg.length() > 2)
	  {
	    if (currentArg.substring(0, 2).equals("-J"))
	      argsIter.remove();
	  }
      }

    args = (String[]) argsList.toArray(new String[argsList.size()]);

    LongOpt[] longOptions =
      new LongOpt[]
      {
        new LongOpt("code", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("codebase", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("archive", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("param", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("width", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("height", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("plugin", LongOpt.REQUIRED_ARGUMENT, null, 0),
        new LongOpt("version", LongOpt.NO_ARGUMENT, null, 0),
        new LongOpt("help", LongOpt.NO_ARGUMENT, null, 0),
      };

    Getopt opts = new Getopt("gcjappletviewer", args, "", longOptions);

    int optionValue;
    int optionIndex;

    String code = "";
    String codebase = "";
    String archives = "";
    List parameters = new ArrayList();
    Dimension dimensions = new Dimension(-1, -1);
    String pipeInName = "";
    String pipeOutName = "";

    boolean codeGiven = false;
    boolean widthGiven = false;
    boolean heightGiven = false;
    boolean pluginMode = false;

    // This loop parses the command line, placing all option arguments
    // at the start of args, all non-options at the end, and setting
    // optind to the index of the first non-option argument.
    while ((optionValue = opts.getopt()) != -1)
      {
	if (optionValue == 0)
	  {
	    optionIndex = opts.getLongind();

	    if (optionIndex == 0) // --code
	      {
		codeGiven = true;
		code = opts.getOptarg();
	      }
	    if (optionIndex == 1) // --codebase
	      codebase = opts.getOptarg();
	    if (optionIndex == 2) // --archive
	      archives = opts.getOptarg();
	    else if (optionIndex == 3) // --param
	      parameters.add(opts.getOptarg());
	    else if (optionIndex == 4) // --width
	      {
		widthGiven = true;
		dimensions.width =
		  Math.max(1, Integer.parseInt(opts.getOptarg()));
	      }
	    else if (optionIndex == 5) // --height
	      {
		heightGiven = true;
		dimensions.height =
		  Math.max(1, Integer.parseInt(opts.getOptarg()));
	      }
	    else if (optionIndex == 6) // --plugin
	      {
		pluginMode = true;
		String tmp = opts.getOptarg();
		int pos = tmp.indexOf(',');
		pipeInName = tmp.substring(0, pos);
		pipeOutName = tmp.substring(pos + 1);
		break;
	      }
	    else if (optionIndex == 7) // --version
	      printVersion();
	    else if (optionIndex == 8) // --help
	      printHelpMessage(0);
	  }
      }

    if (dimensions.height < 0)
      dimensions.height = 200;

    if (dimensions.width < 0)
      dimensions.width = (int) (1.6 * dimensions.height);

    System.setSecurityManager(new AppletSecurityManager(pluginMode));
    
    if (pluginMode)
      {
	InputStream in;
	OutputStream out;

	if (pipeInName.equals("stdin"))
	  in = System.in;
	else
	  in = new FileInputStream(pipeInName);

	if (pipeOutName.equals("stdout"))
	  out = System.out;
	else
	  out = new FileOutputStream(pipeOutName);

	PluginAppletViewer.start(in, out);
      }
    else
      new StandaloneAppletViewer(code, codebase, archives, parameters,
                                 dimensions, widthGiven, heightGiven,
                                 codeGiven, args, opts, opts.getOptind(),
                                 encoding);
  }
}
