// -*- C++ -*-

/* 
 * Gnome Crystal
 * atom.cc 
 *
 * Copyright (C) 2000-2002
 *
 * Developed by Jean Brfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gcrystal.h"
#include "atom.h"
#include "element.h"
#include <math.h>
#include <glib.h>
#include <GL/gl.h>
#include <GL/glu.h>

gcAtom::gcAtom()
{
	m_Z = 0;
	m_Symbol[0] = 0;
	m_dx = m_dy = m_dz = m_dr = 0.0 ;
	m_fRed = m_fBlue = m_fGreen = 0 ;
	m_fAlpha = 1 ;
	m_nCleave = 0 ;
}

gcAtom::~gcAtom()
{
}

gcAtom::gcAtom(int Z, double x, double y, double z, double r, float red, float green, float blue, float alpha)
{
	gcElement *elt = Elt[Z];
	if (elt)
	{
		m_Z = (unsigned char)Z;
		strncpy(m_Symbol, elt->Symbol(), 4);
	}
	else
	{
		m_Z = 0;
		m_Symbol[0] = 0;
	}
	m_dx = x;
	m_dy = y;
	m_dz = z;
	m_dr = r;
	m_fRed = red;
	m_fGreen = green;
	m_fBlue = blue;
	m_fAlpha = alpha;
	m_nCleave = 0;
}

gcAtom::gcAtom(gcAtom& caAtom)
{
	m_Z = caAtom.m_Z;
	strncpy(m_Symbol, caAtom.m_Symbol, 4);
	m_dx = caAtom.m_dx ;
	m_dy = caAtom.m_dy ;
	m_dz = caAtom.m_dz ;
	m_dr = caAtom.m_dr ;
	m_fRed = caAtom.m_fRed ;
	m_fGreen = caAtom.m_fGreen ;
	m_fBlue = caAtom.m_fBlue ;
	m_fAlpha = caAtom.m_fAlpha ;
	m_nCleave = 0 ;
}

gcAtom& gcAtom::operator=(gcAtom& caAtom)
{
	m_Z = caAtom.m_Z;
	strncpy(m_Symbol, caAtom.m_Symbol, 4);
	m_dx = caAtom.m_dx ;
	m_dy = caAtom.m_dy ;
	m_dz = caAtom.m_dz ;
	m_dr = caAtom.m_dr ;
	m_fRed = caAtom.m_fRed ;
	m_fGreen = caAtom.m_fGreen ;
	m_fBlue = caAtom.m_fBlue ;
	m_fAlpha = caAtom.m_fAlpha ;
	m_nCleave = 0 ;
	return *this ;
}

void gcAtom::Move(double x, double y, double z)
{
	m_dx += x ;
	m_dy += y ;
	m_dz += z ;
}

void gcAtom::Draw()
{
	if (m_nCleave) return ;
	GLUquadricObj *quadObj ;
	glPushMatrix() ;
	glTranslated(m_dy, m_dz, m_dx) ;
//	glColor4ub((unsigned char)m_fRed, (unsigned char)m_fGreen, (unsigned char)m_fBlue, (unsigned char)m_fAlpha) ;
	glColor4f(m_fRed, m_fGreen, m_fBlue, m_fAlpha) ;
	quadObj = gluNewQuadric() ;
    gluQuadricDrawStyle(quadObj, GL_FILL);
	gluQuadricNormals(quadObj, GL_SMOOTH) ;
	gluSphere(quadObj, m_dr, 20, 10) ;
	gluDeleteQuadric(quadObj) ;
	glPopMatrix() ;
}

void gcAtom::SetPosition(double x, double y, double z)
{
	m_dx = x ;
	m_dy = y ;
	m_dz = z ;
}

void gcAtom::GetPosition(double *x, double *y, double *z)
{
	*x = m_dx ;
	*y = m_dy ;
	*z = m_dz ;
}

void gcAtom::SetColor(float red, float green, float blue, float alpha)
{
	m_fRed = red ;
	m_fGreen = green ;
	m_fBlue = blue ;
	m_fAlpha = alpha ;
}

void gcAtom::GetColor(double *red, double *green, double *blue, double *alpha)
{
	*red = m_fRed ;
	*green = m_fGreen ;
	*blue = m_fBlue ;
	*alpha = m_fAlpha ;
}

void gcAtom::SetSize(double r)
{
	m_dr = r ;
}


double gcAtom::GetSize()
{
	return m_dr ;
}

bool gcAtom::operator==(gcAtom& caAtom)
{
	return (m_dx == caAtom.m_dx) &&
			(m_dy == caAtom.m_dy) &&
			(m_dz == caAtom.m_dz) ;
}

double gcAtom::ScalProd(int h, int k, int l)
{
	return m_dx * h + m_dy * k + m_dz * l ;
}

double gcAtom::Distance(double x, double y, double z, bool bFixed)
{
	if ((m_nCleave > 0) && ! bFixed) return 0 ;
	x -= m_dx ;
	y -= m_dy ;
	z -= m_dz ;
	return sqrt(x * x + y * y + z * z) + m_dr ;
}

void gcAtom::NetToCartesian(double a, double b, double c, double alpha, double beta, double gamma)
{
	double x = m_dx * a ;
	double y = m_dy * b ;
	double z = m_dz * c ;
	m_dx = x * sqrt(1-square(cos(beta)) - square((cos(gamma) - cos(beta)*cos(alpha))/sin(alpha))) ;
	m_dy = x * (cos(gamma) - cos(beta)*cos(alpha))/sin(alpha) + y * sin(alpha) ;
	m_dz = (x * cos(beta) + y * cos(alpha) + z) ;
}

xmlNodePtr gcAtom::Save(xmlDocPtr xml)
{
	xmlNodePtr parent, child;
	gchar buf[256];
	parent = xmlNewDocNode(xml, NULL, (xmlChar*)"atom", NULL);
	if (!parent) return NULL;
	
	if (strlen(m_Symbol))
	{
		child = xmlNewDocNode(xml, NULL, (xmlChar*)"element", (xmlChar*)m_Symbol);
		if (child) xmlAddChild(parent, child);
		else {xmlFreeNode(parent); return NULL;}
	}

	g_snprintf(buf, sizeof(buf) - 1, "%g %g %g", m_dx, m_dy, m_dz);
	child = xmlNewDocNode(xml, NULL, (xmlChar*)"position", (xmlChar*)buf);
	if (child) xmlAddChild(parent, child);
	else {xmlFreeNode(parent); return NULL;}
	
	g_snprintf(buf, sizeof(buf) - 1, "%g", m_dr);
	child = xmlNewDocNode(xml, NULL, (xmlChar*)"radius", (xmlChar*)buf);
	if (child) xmlAddChild(parent, child);
	else {xmlFreeNode(parent); return NULL;}
	
	g_snprintf(buf, sizeof(buf) - 1, "%g %g %g %g", m_fRed, m_fGreen, m_fBlue, m_fAlpha);
	child = xmlNewDocNode(xml, NULL, (xmlChar*)"color", (xmlChar*)buf);
	if (child) xmlAddChild(parent, child);
	else {xmlFreeNode(parent); return NULL;}
	
	return parent;
}

bool gcAtom::Load(xmlNodePtr node, unsigned version)
{
	char *txt;
	xmlNodePtr child = node->childs;
	while(child)
	{
		if (!strcmp((gchar*)child->name, "element"))
		{
			txt = (char*)xmlNodeGetContent(child);
			m_Z = EltsMap[txt];
			if (m_Z) strncpy(m_Symbol, txt, 4);
		}
		else if (!strcmp((gchar*)child->name, "position"))
		{
			txt = (char*)xmlNodeGetContent(child);
			sscanf(txt, "%lg %lg %lg", &m_dx, &m_dy, &m_dz);
		}
		else if (!strcmp((gchar*)child->name, "color"))
		{
			txt = (char*)xmlNodeGetContent(child);
			if (version < 0x200) sscanf(txt, "%g %g %g %g", &m_fBlue, &m_fRed, &m_fGreen, &m_fAlpha);
			else sscanf(txt, "%g %g %g %g", &m_fRed, &m_fGreen, &m_fBlue, &m_fAlpha);
		}
		else if (!strcmp((gchar*)child->name, "radius"))
		{
			txt = (char*)xmlNodeGetContent(child);
			sscanf(txt, "%lg", &m_dr);
		}
		child = child->next;
	}
	if (m_dr == 0) return false;
	return true;
}
