// -*- C++ -*-

/* 
 * Gnome Crystal
 * document.cc 
 *
 * Copyright (C) 2000-2002
 *
 * Developed by Jean Brfort <jean.brefort@ac-dijon.fr>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "../config.h"
#include "gcrystal.h"
#include <unistd.h>
#include <locale.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include "document.h"
#include "view.h"
#include "latticedlg.h"
#include "globals.h"
#include "filesel.h"
#include <gnome.h>
#include <glade/glade.h>
#include <math.h>
#include <gnome-xml/parserInternals.h>
#include <gnome-xml/xmlmemory.h>
#include <vector>
#ifdef HAVE_FSTREAM
#	include <fstream>
#else
#	include <fstream.h>
#endif
#ifdef HAVE_OSTREAM
#	include <ostream>
#else
#	include <ostream.h>
#endif

#define SAVE	1
#define LOAD	2
#define XML		3
#define FORMAT	4

#define PREC 1e-3

using namespace std;

char *LatticeName[] = {"simple cubic",
	"body-centered cubic",
	"face-centered cubic",
	"hexagonal",
	"tetragonal",
	"body-centered tetragonal",
	"orthorhombic",
	"base-centered orthorhombic",
	"body-centered orthorhombic",
	"face-centered orthorhombic",
	"rhombohedral",
	"monoclinic",
	"base-centered monoclinic",
	"triclinic"};

static void do_save_as(const gchar* filename, gcView* pView)
{
	gcDocument *pDoc = pView->GetDocument();
	pDoc->SetFileName(filename);
}

gcDocument::gcDocument()
{
	Init();
	m_filename = NULL;
	m_title = NULL;
	gcView* pView = new gcView(this);
	m_Views.push_back(pView);
	m_bEmpty = true;
	m_bDirty = false;
	m_bClosing = false;
#ifdef USING_BONOBO
	m_ps = NULL;
#endif
}

/*gcDocument::gcDocument(bool create_view)
{
	Init();
	if (create_view) pView = m_pView = new gcView(this);
}*/

gcDocument::~gcDocument()
{
	while (!m_Views.empty())
	{
		m_Views.pop_back();
	}
	if (m_filename != NULL) g_free(m_filename);
	if (m_title) g_free(m_title);
	Reinit();
	gcDialog *dialog;
	while (!m_Dialogs.empty())
	{
		dialog = m_Dialogs.front();
		m_Dialogs.pop_front();
		dialog->Destroy();
	}
}

void gcDocument::Reinit()
{
	//destruction of lists
	while (!AtomDef.empty())
	{
		delete AtomDef.front();
		AtomDef.pop_front();
	}
	while (!Atoms.empty())
	{
		delete Atoms.front();
		Atoms.pop_front();
	}
	while (!BondDef.empty())
	{
		delete BondDef.front();
		BondDef.pop_front();
	}
	while (!Bonds.empty())
	{
		delete Bonds.front();
		Bonds.pop_front();
	}
	while (!Cleavages.empty())
	{
		delete Cleavages.front();
		Cleavages.pop_front();
	}
	Init();
}

void gcDocument::Init()
{
	m_a = m_b = m_c = 100;
	m_alpha = m_beta = m_gamma = 90;
	m_lattice = cubic;
	m_xmin = m_ymin = m_zmin = 0;
	m_xmax = m_ymax = m_zmax = 1;
	m_bFixedSize = false;
	m_dDist = 0;
}

void gcDocument::Define(unsigned nPage)
{
	gcLatticeDlg* pBox = new gcLatticeDlg(this, nPage);
}

void gcDocument::Update()
{
	m_bEmpty = (AtomDef.empty() && BondDef.empty()) ? true : false;
	gcAtom Atom;
	gcBond Bond;
	gdouble alpha = m_alpha * 1.570796326794897 / 90;
	gdouble beta = m_beta * 1.570796326794897 / 90;
	gdouble gamma = m_gamma * 1.570796326794897 / 90;
	// Remove all atoms
	while (!Atoms.empty())
	{
		delete Atoms.front();
		Atoms.pop_front();
	}
	// Remove all bonds
	while (!Bonds.empty())
	{
		delete Bonds.front();
		Bonds.pop_front();
	}

	////////////////////////////////////////////////////////////
	//Establish list of atoms
	gcAtomList::iterator i;
	
	for (i = AtomDef.begin(); i != AtomDef.end(); i++)
	{
		Duplicate(**i);
		switch (m_lattice)
		{
		case body_centered_cubic:
		case body_centered_tetragonal:
		case body_centered_orthorhombic:
			Atom = **i;
			Atom.Move(0.5, 0.5, 0.5);
			Duplicate(Atom);
			break;
		case face_centered_cubic:
		case face_centered_orthorhombic:
			Atom = **i;
			Atom.Move(0.5, 0, 0.5);
			Duplicate(Atom);
			Atom = **i;
			Atom.Move(0, 0.5, 0.5);
			Duplicate(Atom);
		case base_centered_orthorhombic:
		case base_centered_monoclinic:
			Atom = **i;
			Atom.Move(0.5, 0.5, 0);
			Duplicate(Atom);
		}
	}
	
	////////////////////////////////////////////////////////////
	//Establish list of atoms
	gcBondList::iterator j;
	for (j = BondDef.begin() ; j != BondDef.end() ; j++)
	{
		switch ((*j)->Type())
		{
		case edges:
			Bond = **j;
			Bond.SetPosition(0 ,0, 0, 1, 0, 0);
			Duplicate(Bond);
			Bond.SetPosition(0 ,0, 0, 0, 1, 0);
			Duplicate(Bond);
			Bond.SetPosition(0 ,0, 0, 0, 0, 1);
			Duplicate(Bond);
			break ;
		case diagonals:
			Bond = **j;
			Bond.SetPosition(0 ,0, 0, 1, 1, 1);
			Duplicate(Bond);
			Bond.SetPosition(1 ,0, 0, 0, 1, 1);
			Duplicate(Bond);
			Bond.SetPosition(0 ,1, 0, 1, 0, 1);
			Duplicate(Bond);
			Bond.SetPosition(1 ,1, 0, 0, 0, 1);
			Duplicate(Bond);
			break ;
		case medians:
			Bond = **j;
			Bond.SetPosition(.5, .5, 0, .5, .5, 1);
			Duplicate(Bond);
			Bond.SetPosition(0, .5, .5, 1, .5, .5);
			Duplicate(Bond);
			Bond.SetPosition(.5, 0, .5, .5, 1, .5);
			Duplicate(Bond);
			break ;
		case bond:
			Duplicate(**j) ;
			switch (m_lattice)
			{
			case body_centered_cubic:
			case body_centered_tetragonal:
			case body_centered_orthorhombic:
				Bond = **j;
				Bond.Move(0.5, 0.5, 0.5);
				Duplicate(Bond);
				break;
			case face_centered_cubic:
			case face_centered_orthorhombic:
				Bond = **j;
				Bond.Move(0.5, 0, 0.5);
				Duplicate(Bond);
				Bond = **j;
				Bond.Move(0, 0.5, 0.5);
				Duplicate(Bond);
			case base_centered_orthorhombic:
			case base_centered_monoclinic:
				Bond = **j;
				Bond.Move(0.5, 0.5, 0);
				Duplicate(Bond);
			}
			break ;
		case ::unique:
			if (((*j)->Xmin() >= m_xmin) && ((*j)->Xmax() <= m_xmax)
				&& ((*j)->Ymin() >= m_ymin) && ((*j)->Ymax() <= m_ymax)
				&& ((*j)->Zmin() >= m_zmin) && ((*j)->Zmax() <= m_zmax))
					Bonds.push_back(new gcBond(**j)) ;
		}
	}

	//Searching the center of the crystal
	Atom.SetPosition((m_xmax +m_xmin) / 2, (m_ymax + m_ymin) / 2, (m_zmax + m_zmin) / 2);
	Atom.NetToCartesian(m_a, m_b, m_c, alpha, beta, gamma);
	
	//Manage cleavages
	gcCleavageList::iterator k;
	for (k = Cleavages.begin(); k != Cleavages.end(); k++)
	{
		double x;
		vector<double> ScalarProducts;
		vector<double>::iterator m;
		int n, _h, _k, _l;

		//scalar products calculus and storing
		for (i = Atoms.begin(); i != Atoms.end(); i++)
		{
			x = (*i)->ScalProd((*k)->h(), (*k)->k(), (*k)->l());
			for (m = ScalarProducts.begin(); m != (ScalarProducts.end()) && ((*m) > (x + PREC)); m++) ;
			if ((m == ScalarProducts.end()) || (fabs(*m - x) > PREC)) ScalarProducts.insert(m, x);
		}

		//cleave atoms and bonds
		if ((n = (*k)->Planes()) >= ScalarProducts.size())
		{
			GtkWidget* message = gnome_warning_dialog(_("Everything has been cleaved"));
			gtk_widget_show(message);
			for (i = Atoms.begin(); i != Atoms.end(); i++) (*i)->Cleave();
			for (j = Bonds.begin(); j != Bonds.end(); j++) (*j)->Cleave();
		}
		else
		{
			x = ScalarProducts[n - 1];
			for (i = Atoms.begin(); i != Atoms.end(); i++)
				if (x < (*i)->ScalProd((*k)->h(), (*k)->k(), (*k)->l())+ PREC)
					(*i)->Cleave() ;

			//bonds cleavage
			for (j = Bonds.begin(); j != Bonds.end(); j++)
				if (x < (*j)->ScalProd((*k)->h(), (*k)->k(), (*k)->l())+ PREC)
					(*j)->Cleave();
		}

		ScalarProducts.clear() ;
	}
	
	//Transform coordinates to Cartesians and find maximum distance from center
	gdouble x, y, z, d;
	Atom.GetPosition(&x, &y, &z);
	m_dDist = 0;
	for (i = Atoms.begin(); i != Atoms.end(); i++)
	{
		(*i)->NetToCartesian(m_a, m_b, m_c, alpha, beta, gamma);
		d =  (*i)->Distance(x, y, z, m_bFixedSize);
		m_dDist = __max(m_dDist, d);
		(*i)->Move(- x, - y, - z);
	}

	for (j = Bonds.begin(); j != Bonds.end(); j++)
	{
		(*j)->NetToCartesian(m_a, m_b, m_c, alpha, beta, gamma);
		d =  (*j)->Distance(x, y, z, m_bFixedSize);
		m_dDist = __max(m_dDist, d);
		(*j)->Move(- x, - y, - z);
	}
	UpdateAllViews();
}

void gcDocument::UpdateAllViews()
{
	list<gcView*>::iterator i;
	for (i = m_Views.begin(); i != m_Views.end(); i++) (*i)->Update();
}

void gcDocument::Duplicate(gcAtom& Atom)
{
	gcAtom AtomX, AtomY, AtomZ ;
	AtomX = Atom ;
	AtomX.Move(- floor(AtomX.x()-m_xmin), - floor(AtomX.y()-m_ymin), - floor(AtomX.z()-m_zmin)) ;
	while (AtomX.x() <= m_xmax)
	{
		AtomY = AtomX ;
		while (AtomY.y() <= m_ymax)
		{
			AtomZ = AtomY ;
			while (AtomZ.z() <= m_zmax)
			{
				Atoms.push_back(new gcAtom(AtomZ)) ;
				AtomZ.Move(0,0,1) ;
			}
			AtomY.Move(0,1,0) ;
		}
		AtomX.Move(1,0,0) ;
	}
}

void gcDocument::Duplicate(gcBond& Bond)
{
	gcBond BondX, BondY, BondZ ;
	BondX = Bond ;
	BondX.Move(- floor(BondX.Xmin()-m_xmin), - floor(BondX.Ymin()-m_ymin), - floor(BondX.Zmin()-m_zmin)) ;
	while (BondX.Xmax() <= m_xmax)
	{
		BondY = BondX ;
		while (BondY.Ymax() <= m_ymax)
		{
			BondZ = BondY ;
			while (BondZ.Zmax() <= m_zmax)
			{
				Bonds.push_back(new gcBond(BondZ)) ;
				BondZ.Move(0,0,1) ;
			}
			BondY.Move(0,1,0) ;
		}
		BondX.Move(1,0,0) ;
	}
}

void gcDocument::Draw()
{
	gcAtomList::iterator i;
	for (i = Atoms.begin(); i != Atoms.end(); i++) (*i)->Draw();
	gcBondList::iterator j;
	for (j = Bonds.begin(); j != Bonds.end(); j++) (*j)->Draw();
}

void gcDocument::GetSize(gdouble* xmin, gdouble* xmax, gdouble* ymin, gdouble* ymax, gdouble* zmin, gdouble* zmax)
{
	*xmin = m_xmin;
	*xmax = m_xmax;
	*ymin = m_ymin;
	*ymax = m_ymax;
	*zmin = m_zmin;
	*zmax = m_zmax;
}

void gcDocument::SetSize(gdouble xmin, gdouble xmax, gdouble ymin, gdouble ymax, gdouble zmin, gdouble zmax)
{
	m_xmin = xmin;
	m_xmax = xmax;
	m_ymin = ymin;
	m_ymax = ymax;
	m_zmin = zmin;
	m_zmax = zmax;
}

void gcDocument::GetCell(gcLattices *lattice, gdouble *a, gdouble *b, gdouble *c, gdouble *alpha, gdouble *beta, gdouble *gamma)
{
	*lattice = m_lattice;
	*a = m_a;
	*b = m_b;
	*c = m_c;
	*alpha = m_alpha;
	*beta = m_beta;
	*gamma = m_gamma;
}

void gcDocument::SetCell(gcLattices lattice, gdouble a, gdouble b, gdouble c, gdouble alpha, gdouble beta, gdouble gamma)
{
	m_lattice = lattice;
	m_a = a;
	m_b = b;
	m_c = c;
	m_alpha = alpha;
	m_beta = beta;
	m_gamma = gamma;
}

void gcDocument::SetFileName(const gchar* filename)
{
	if (m_filename) g_free(m_filename);
	m_filename = g_strdup(filename);
	int i = strlen(filename) - 1;
	while ((m_filename[i] != '/') && (i >= 0)) i--;
	if (i >=0) 
	{
		m_filename[i] = 0;
		chdir(m_filename);
		m_filename[i] = '/';
	}
	i++;
	int j = strlen(filename) - 1;
	while ((i < j) && (m_filename[j] != '.')) j--;
	if (m_title) g_free(m_title);
	if (strcmp(m_filename + j, ".gcrystal")) m_title = g_strdup(m_filename + i);
	else m_title = g_strndup(m_filename + i, j - i);
}

void gcDocument::SetTitle(const gchar* title)
{
	if (m_title) g_free(m_title);
	m_title = g_strdup(title);
}

void gcDocument::Save()
{
	if (!m_filename) return;
	gchar buf[256];
	xmlDocPtr xml;
	xmlNodePtr node;
	char *old_num_locale;

	xml = xmlNewDoc((xmlChar*)"1.0");
	if (xml == NULL) {Error(SAVE); return;}
	
	old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
	setlocale(LC_NUMERIC, "C");
	
	xml->root =  xmlNewDocNode(xml, NULL, (xmlChar*)"crystal", NULL);
	
	try
	{
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"generator", (xmlChar*)"Gnome Crystal "VERSION);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"lattice", (xmlChar*)LatticeName[m_lattice]);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
	
		g_snprintf(buf, sizeof(buf) - 1, "%g %g %g %g %g %g", m_a, m_b, m_c, m_alpha, m_beta, m_gamma);
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"cell", (xmlChar*)buf);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
	
		g_snprintf(buf, sizeof(buf) - 1, "%g %g %g %g %g %g", m_xmin, m_ymin, m_zmin, m_xmax, m_ymax, m_zmax);
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"size", (xmlChar*)buf);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		if (m_bFixedSize) xmlNewProp(node, (xmlChar*)"fixed", (xmlChar*)"true");
		
		gcAtomList::iterator i;
		for (i = AtomDef.begin(); i != AtomDef.end(); i++)
		{
			node = (*i)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
	
		gcBondList::iterator j;
		for (j = BondDef.begin(); j != BondDef.end(); j++)
		{
			node = (*j)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
	
		gcCleavageList::iterator k;
		for (k = Cleavages.begin(); k != Cleavages.end(); k++)
		{
			node = (*k)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
		
		list<gcView*>::iterator view;
		for (view = m_Views.begin(); view != m_Views.end(); view++)
		{
			node = (*view)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
	
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
	
		if (xmlSaveFile(m_filename, xml) < 0) Error(SAVE);
			
		xmlFreeDoc(xml);
		m_bDirty = false;
	}
	catch (int num)
	{
		xmlFreeDoc(xml);
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		Error(SAVE);
	}
}

xmlDocPtr gcDocument::BuildXMLTree()
{
	gchar buf[256];
	xmlDocPtr xml;
	xmlNodePtr node;
	char *old_num_locale;

	xml = xmlNewDoc((xmlChar*)"1.0");
	if (xml == NULL) {Error(SAVE); return NULL;}
	
	old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
	setlocale(LC_NUMERIC, "C");
	
	xml->root =  xmlNewDocNode(xml, NULL, (xmlChar*)"crystal", NULL);
	
	try
	{
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"generator", (xmlChar*)"Gnome Crystal "VERSION);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"lattice", (xmlChar*)LatticeName[m_lattice]);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
	
		g_snprintf(buf, sizeof(buf) - 1, "%g %g %g %g %g %g", m_a, m_b, m_c, m_alpha, m_beta, m_gamma);
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"cell", (xmlChar*)buf);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
	
		g_snprintf(buf, sizeof(buf) - 1, "%g %g %g %g %g %g", m_xmin, m_ymin, m_zmin, m_xmax, m_ymax, m_zmax);
		node = xmlNewDocNode(xml, NULL, (xmlChar*)"size", (xmlChar*)buf);
		if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		
		gcAtomList::iterator i;
		for (i = AtomDef.begin(); i != AtomDef.end(); i++)
		{
			node = (*i)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
	
		gcBondList::iterator j;
		for (j = BondDef.begin(); j != BondDef.end(); j++)
		{
			node = (*j)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
	
		gcCleavageList::iterator k;
		for (k = Cleavages.begin(); k != Cleavages.end(); k++)
		{
			node = (*k)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
		
		list<gcView*>::iterator view;
		for (view = m_Views.begin(); view != m_Views.end(); view++)
		{
			node = (*view)->Save(xml);
			if (node) xmlAddChild(xml->root, node); else throw (int) 0;
		}
	
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		
		return xml;
	}
	catch (int num)
	{
		xmlFreeDoc(xml);
		setlocale(LC_NUMERIC, old_num_locale);
		g_free(old_num_locale);
		return NULL;
	}
}

void gcDocument::Error(int num)
{
	gchar *mess;
	GtkWidget* message;
	switch (num)
	{
	case SAVE:
		mess = g_strdup_printf(_("Could not save file\n%s"),m_filename);
		break;
	case LOAD:
		mess = g_strdup_printf(_("Could not load file\n%s"),m_filename);
		break;
	case XML:
		mess = g_strdup_printf(_("%s: invalid xml file.\nTree is empty?"),m_filename);
		break;
	case FORMAT:
		mess = g_strdup_printf(_("%s: invalid file format."),m_filename);
		break;
	}
	message = gnome_error_dialog(mess);
	gtk_widget_show(message);
	g_free(mess);
}

bool gcDocument::Load(const gchar* filename)
{
	if (!IsEmpty()) Reinit();
	xmlDocPtr xml;
	gchar *oldfilename, *oldtitle;
	if (m_filename) oldfilename = g_strdup(m_filename);
	else oldfilename = NULL;
	oldtitle = g_strdup(m_title);
	try
	{
		if (SetFileName(filename),!m_filename || !m_title) throw (int) 0;
		if (!(xml = xmlParseFile(filename))) throw (int) 1;
		if (xml->root == NULL) throw (int) 2;
		if (strcmp((char*)xml->root->name, "crystal")) throw (int) 3;
		if (oldfilename) g_free(oldfilename);
		g_free(oldtitle);
		ParseXMLTree(xml);
		m_bDirty = false;
		return true;
	}
	catch (int num)
	{
		switch (num)
		{
			case 2: Error(XML); break;
			case 3: Error(FORMAT); break;
			default: Error(LOAD);
		}
		if (num > 0)
		{
			if (oldfilename) 
			{
				SetFileName(oldfilename);
				g_free(oldfilename);
			}
			else
			{
				g_free(m_filename);
				m_filename = NULL;
			}
			SetTitle(oldtitle);
			g_free(oldtitle);
		}
		if (num > 1) xmlFreeDoc(xml);
		return false;
	}
}

void gcDocument::ParseXMLTree(xmlDocPtr xml)
{
	char *old_num_locale, *txt;
	xmlNodePtr node;
	bool bViewLoaded = false;

	Reinit();
	old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
	setlocale(LC_NUMERIC, "C");
	
	//look for generator node
	unsigned version = 0xffffff , major, minor, micro;
	node = xml->root->childs;
	while (node)
	{
		if (!strcmp ((const char*)(node->name), "generator")) break;
		node = node->next;
	}
	if (node)
	{
		txt = (char*)xmlNodeGetContent(node);
		if (sscanf(txt, "Gnome Crystal %d.%d.%d", &major, &minor, &micro) == 3)
			version = micro + minor * 0x100 + major * 0x10000;
	}
	node = xml->root->childs;
	while(node)
	{
		if (!strcmp((gchar*)node->name, "lattice"))
		{
			txt = (char*)xmlNodeGetContent(node);
			int i = 0;
			while (strcmp(txt, LatticeName[i]) && (i < 14)) i++;
			if (i < 14) m_lattice = (gcLattices)i;
		}
		else if (!strcmp((gchar*)node->name, "cell"))
		{
			txt = (char*)xmlNodeGetContent(node);
			sscanf(txt, "%lg %lg %lg %lg %lg %lg", &m_a, &m_b, &m_c, &m_alpha, &m_beta, &m_gamma);
		}
		else if (!strcmp((gchar*)node->name, "size"))
		{
			txt = (char*)xmlNodeGetContent(node);
			sscanf(txt, "%lg %lg %lg %lg %lg %lg", &m_xmin, &m_ymin, &m_zmin, &m_xmax, &m_ymax, &m_zmax);
			txt = (char*)xmlGetProp(node, (xmlChar*)"fixed");
			if (txt && !strcmp(txt, "true")) m_bFixedSize = true;
		}
		else if (!strcmp((gchar*)node->name, "atom"))
		{
			gcAtom *pAtom = new gcAtom();
			if (pAtom->Load(node, version)) AtomDef.push_back(pAtom);
			else delete pAtom;
		}
		else if (!strcmp((gchar*)node->name, "line"))
		{
			gcBond *pBond = new gcBond();
			if (pBond->Load(node, version)) BondDef.push_back(pBond);
			else delete pBond;
		}
		else if (!strcmp((gchar*)node->name, "cleavage"))
		{
			gcCleavage *pCleavage = new gcCleavage();
			if (pCleavage->Load(node)) Cleavages.push_back(pCleavage);
			else delete pCleavage;
		}
		else if (!strcmp((gchar*)node->name, "view"))
		{
			if (bViewLoaded && !IsEmbedded())
			{
				gcView* pView = new gcView(this);
				pView->Load(node);
				m_Views.push_back(pView);
				if (!RequestApp(pView))
				{
					delete pView;
				}
			}
			else
			{
				m_Views.front()->Load(node); //the first view is created with the document
				bViewLoaded = true;
			}
		}
		node = node->next;
	}
	setlocale(LC_NUMERIC, old_num_locale);
	g_free(old_num_locale);
	xmlFreeDoc(xml);
	Update();
}

void gcDocument::OnNewDocument()
{
	Reinit();
	UpdateAllViews();
}

typedef struct {int n; list<gcAtom*> l;} sAtom;
typedef struct {int n; list<gcBond*> l;} sBond;

void gcDocument::OnExportVRML(const gchar* FileName, gcView* pView)
{
	char *old_num_locale, tmp[128];
	double x0, x1, x2, x3, x4, x5;
	int n = 0;
	ofstream file(FileName);
	map<string, sAtom /*list<gcAtom*>*/ >AtomsMap;
	map<string, sBond>BondsMap;
	if (!file)
	{
		gchar* mess = g_strdup_printf(_("Can't create file %s"), FileName);
		GtkWidget* message = gnome_error_dialog(mess);
		gtk_widget_show(message);
		g_free(mess);
	}
	old_num_locale = g_strdup(setlocale(LC_NUMERIC, NULL));
	setlocale(LC_NUMERIC, "C");
	file << "#VRML V2.0 utf8" << endl;
	
	//Create prototypes for atoms
	gcAtomList::iterator i;
	for (i = Atoms.begin(); i != Atoms.end(); i++)
	{
		(*i)->GetColor(&x0, &x1, &x2, &x3);
		snprintf(tmp, sizeof(tmp), "%g %g %g %g %g", (*i)->GetSize(), x0, x1, x2, x3);
		if (AtomsMap[tmp].l.empty())
		{
			AtomsMap[tmp].n = n;
			file << "PROTO Atom" << n++ << " [] {Shape {" << endl << "\tgeometry Sphere {radius " << (*i)->GetSize() / 100 << "}" << endl;
			file << "\tappearance Appearance {" << endl << "\t\tmaterial Material {" << endl << "\t\t\tdiffuseColor " << x0 << " " << x1 << " " << x2 << endl;
			if (x3 < 1) file << "\t\t\ttransparency " << (1 - x3) << endl;
			file << "\t\t\tspecularColor 1 1 1" << endl << "\t\t\tshininess 0.9" << endl << "\t\t}" << endl << "\t}\r\n}}" << endl;
		}
		AtomsMap[tmp].l.push_back(*i);
	}

	//Create prototypes for bonds
	gcBondList::iterator j;
	n = 0;
	for (j = Bonds.begin(); j != Bonds.end(); j++)
	{
		(*j)->GetColor(&x0, &x1, &x2, &x3);
		snprintf(tmp, sizeof(tmp), "%g %g %g %g %g %g", (*j)->Long(), (*j)->GetRadius(), x0, x1, x2, x3);
		if (BondsMap[tmp].l.empty())
		{
			BondsMap[tmp].n = n;
			file << "PROTO Bond" << n++ << " [] {Shape {" << endl << "\tgeometry Cylinder {radius " << (*j)->GetRadius() / 100 << "\theight " << (*j)->Long() / 100 << "}" << endl;
			file << "\tappearance Appearance {" << endl << "\t\tmaterial Material {" << endl << "\t\t\tdiffuseColor " << x0 << " " << x1 << " " << x2 << endl;
			if (x3 < 1) file << "\t\t\ttransparency " << (1 - x3) << endl;
			file << "\t\t\tspecularColor 1 1 1" << endl << "\t\t\tshininess 0.9" << endl << "\t\t}" << endl << "\t}\r\n}}" << endl;
		}
		BondsMap[tmp].l.push_back(*j);
	}
	
	//world begin
	pView->GetBackgroundColor(&x0, &x1, &x2, &x3);
	file << "Background{skyColor " << x0 << " " << x1 << " " << x2 << "}" << endl;
	file << "Viewpoint {fieldOfView " << pView->GetFoV()/90*1.570796326794897 <<"\tposition 0 0 " << pView->GetPos() / 100 << "}" << endl;
	pView->GetRotation(&x0, &x1, &x2);
	gcMatrix m(x0/90*1.570796326794897, x1/90*1.570796326794897, x2/90*1.570796326794897, euler);
	file << "Transform {" << endl << "\tchildren [" << endl;

	map<string, sAtom>::iterator k;
	for (k = AtomsMap.begin(); k != AtomsMap.end(); k++)
	{
		for (i = (*k).second.l.begin(); i != (*k).second.l.end(); i++)
		{
			if (!(*i)->IsCleaved())
			{
				x0 = (*i)->x();
				x1 = (*i)->y();
				x2 = (*i)->z();
				m.Transform(x0, x1, x2);
				file << "\t\tTransform {translation " << x1 / 100 << " " << x2 / 100 << " " << x0 / 100\
					<<  " children [Atom" << (*k).second.n << " {}]}" << endl;
			}
		}
		(*k).second.l.clear();
	}
	AtomsMap.clear();
	
	map<string, sBond>::iterator l;
	n = 0;
	for (l = BondsMap.begin(); l != BondsMap.end(); l++)
	{
		for (j = (*l).second.l.begin(); j != (*l).second.l.end(); j++)
		{
			if (!(*j)->IsCleaved())
			{
				x0 = (*j)->X1();
				x1 = (*j)->Y1();
				x2 = (*j)->Z1();
				m.Transform(x0, x1, x2);
				x3 = (*j)->X2();
				x4 = (*j)->Y2();
				x5 = (*j)->Z2();
				m.Transform(x3, x4, x5);
				gcBond bond(::unique, x0, x1, x2, x3, x4, x5, 0.0, 0.0, 0.0, 0.0, 0.0);
				bond.GetRotation(x0, x1, x2, x3);
				file << "\t\tTransform {" << endl << "\t\t\trotation " << x1 << " " << x2 << " " << x0 << " " << x3 << endl;
				x0 = (bond.X1() + bond.X2()) / 200;
				x1 = (bond.Y1() + bond.Y2()) / 200;
				x2 = (bond.Z1() + bond.Z2()) / 200;
				file << "\t\t\ttranslation " << x1 << " " << x2  << " " << x0 <<  endl\
						<< "\t\t\tchildren [Bond" << (*l).second.n << " {}]}" << endl;
			}
		}
		n++;
		(*l).second.l.clear();
	}
	BondsMap.clear();
	
	//end of the world
	file << "\t]" << endl << "}" << endl;

	file.close();
	setlocale(LC_NUMERIC, old_num_locale);
	g_free(old_num_locale);
}

gcView *gcDocument::GetView()
{
	return m_Views.front();
}

gcView *gcDocument::GetNewView()
{
}

void gcDocument::SetDirty()
{
	m_bDirty = true;
#ifdef USING_BONOBO
	if (m_ps) bonobo_persist_stream_set_dirty(m_ps, m_bDirty);
#endif
}

void gcDocument::AddView(gcView* pView)
{
	m_Views.push_back(pView);
	if (!m_bEmpty) m_bDirty = true;
}

bool gcDocument::RemoveView(gcView* pView)
{
	if (pView->IsLocked()) return false;
	if (m_Views.size() > 1)
	{
		m_Views.remove(pView);
		if (!m_bClosing && !m_bEmpty) m_bDirty = true;
		return true;
	}
	if (IsDirty())
	{
		if (!VerifySaved()) return false;
	}
	RemoveDocument(this);
	return true;
}

bool gcDocument::VerifySaved()
{
	m_bClosing = true;
	if (!m_bDirty) return true;
	gchar* str = g_strdup_printf(_("``%s'' has been modified.  Do you wish to save it?"), m_title);
	GtkWidget* mbox;
	int res;
	do
	{
		mbox = gnome_message_box_new(str, GNOME_MESSAGE_BOX_QUESTION, GNOME_STOCK_BUTTON_YES,
											GNOME_STOCK_BUTTON_NO, GNOME_STOCK_BUTTON_CANCEL, NULL);
		res = gnome_dialog_run(GNOME_DIALOG(mbox));
		if (res == 0)
		{
			if (m_filename == NULL)
			{
				gcFileSel* FileSel = new gcFileSel(_("Save model as..."), do_save_as, true, ".gcrystal", m_Views.front(), true);
				while (m_Views.front()->IsLocked())
					if (gtk_events_pending()) gtk_main_iteration();
			}
			if (m_filename) Save();
		}
	}
	while ((res == 0) && (m_filename == NULL));
	if (res == 1) m_bDirty = false;
	else if (res == 2) m_bClosing = false;
	g_free(str);
	return (res != 2);
}

void gcDocument::NotifyDialog(gcDialog* dialog)
{
	m_Dialogs.push_front(dialog);
}

void gcDocument::RemoveDialog(gcDialog* dialog)
{
	m_Dialogs.remove(dialog);
}
