/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

#include "genezip/gzfile_compress.h"

void genezip_utils::gzfile_compress::emit_result
(const vectorlist_search_result &res, bool show_value = false) {
  if (res.get_result_type() != vectorlist_search_result::NONE) {
    _generator.add_token(res);
  } else throw std::domain_error("genezip_utils::gzfile_compress::emit_result:"
				 " null result provided to stream");
  if (show_value) {
    switch (res.get_result_type()) {
    case vectorlist_search_result::NONE:
      std::cout << "FOUND NULL RESULT" << std::endl;
    case vectorlist_search_result::MATCH:
      std::cout << "FOUND MATCH: " << res.get_match().first << "/" 
		<< res.get_match().second << std::endl;
      return;
    case vectorlist_search_result::LITERAL:
      std::cout << "FOUND LITERAL: " << res.get_literal() << std::endl;
      return;
    case vectorlist_search_result::ENDOFBLOCK:
      std::cout << "FOUND END OF BLOCK" << std::endl;
      return;
    default:
      throw std::domain_error("genezip_utils::gzfile_compress::emit_result:"
			      " result type not recognized");
    }
  }
}

void genezip_utils::gzfile_compress::compress_current_value
(unsigned value,
 std::vector<bool> &intermediate_data_holder,
 huffman_code &litlen_code,
 huffman_code &distance_code) {
  //std::cout << "compressing by value" << std::endl;
  _cstate.buffer.add(value);
  vectorlist_search_result res;
  unsigned match_start = 0, match_length = 0;
  //now handle searching
  if (_cstate.skip_counter) {
    --_cstate.skip_counter;
  } else {
    //search
    unsigned start_index = _cstate.tree_vec_counter,
      search_forward_bound = _cstate.tree_vec_counter + 1;
    unsigned match_start = 0, match_length = 0;
    if (_cstate.tree_vec_counter > 1) {
      _cstate.stree->search(start_index, search_forward_bound, match_start,
			    match_length);
      if (match_length >= 2) {
	res.set_match(_cstate.tree_vec_counter - match_start, match_length);
      } else {
	res.set_literal(value);
      }
    } else {
      res.set_literal(value);
    }
    
    //if both the previous and current results are matches,
    //check for lazy matching
    if (res.get_result_type() == vectorlist_search_result::MATCH &&
	_cstate.previous_res.get_result_type() == 
	vectorlist_search_result::MATCH)
      {
	//if the current match length is better than the last match, keep 
	//this one!
	if (_cstate.previous_res.get_match().second < res.get_match().second) {
	  //emit the single literal at the beginning of the last match
	  _cstate.previous_res.set_literal(_cstate.previous_literal);
	  emit_result(_cstate.previous_res, _cstate.waiting);
	  _cstate.previous_res = res;
	} else {
	  //emit the previous match and skip for a while
	  _cstate.skip_counter = _cstate.previous_res.get_match().second - 2;
	  emit_result(_cstate.previous_res, _cstate.waiting);
	  _cstate.previous_res.nullify();
	  _cstate.pending_match = false;
	}
      } else
      {//otherwise emit.  don't skip under any circumstances
	if (_cstate.pending_match) {
	  _cstate.skip_counter = _cstate.previous_res.get_match().second - 2;
	  emit_result(_cstate.previous_res, _cstate.waiting);
	  _cstate.previous_res.nullify();
	  _cstate.pending_match = false;
	} else {
	  if (res.get_result_type() == vectorlist_search_result::MATCH) {
	    //set this as the previous result.  may be used for 
	    //lazy matching next round
	    _cstate.previous_res = res;
	    _cstate.pending_match = true;
	  } else {
	    _cstate.previous_res.nullify();
	    emit_result(res, _cstate.waiting);
	  }
	}
      }
  }
  if (_cstate.tree_vec_counter < GENEZIP_MAX_OFFSET_POINTER) {
    _cstate.stree->add_node_to_root(value, _cstate.tree_vec_counter);
  } else {
    _cstate.stree->add_and_delete(value, _cstate.tree_vec_counter);
  }
  
  //clean the tree periodically
  if (_cstate.tree_vec_counter && 
      (_cstate.tree_vec_counter % (32768<<3) == 0)) {
    //std::cout << "cleaning" << std::endl;
    _cstate.stree->dfs(NULL, true, _cstate.tree_vec_counter-65536);
  }
  _cstate.previous_literal = value;
  ++_cstate.tree_vec_counter;
}

void genezip_utils::gzfile_compress::finish_value_stream
(std::vector<bool> &
 intermediate_data_holder,
 huffman_code &
 litlen_code,
 huffman_code &
 distance_code) {
  //chomp the remaining values?
  if (_cstate.pending_match) {
    emit_result(_cstate.previous_res, _cstate.waiting);
    _cstate.previous_res.nullify();
    _cstate.pending_match = false;
  }
  //have to flush an end of block signal
  vectorlist_search_result res;
  res.set_end_of_block();
  emit_result(res, _cstate.waiting);
  //flush the generator
  _generator.flush_data(intermediate_data_holder,
			litlen_code,
			distance_code);
  //clean out the compression state
  _cstate.clear();
}

void genezip_utils::gzfile_compress::compress_from_vector
(uncompressed_buffer &buffer,
 std::vector<bool> & intermediate_data_holder,
 huffman_code & litlen_code,
 huffman_code & distance_code) {
  vectorlist_search_result res, previous_res;
  bool pending_match = false;
  
  suffix_tree stree(static_cast<unsigned>(pow(2, buffer.bits_per_element())
					  + 0.5));
  stree.set_raw_data_buffer(&buffer);
  
  bool waiting = false;
  std::string temp = "";

  unsigned skip_counter = 0;

    for (unsigned tree_vec_counter = 0; tree_vec_counter < buffer.size(); 
	 ++tree_vec_counter) {
      //now handle searching
      if (skip_counter) {
	--skip_counter;
      } else {
	//search
	unsigned start_index = tree_vec_counter,
	  search_forward_bound = tree_vec_counter + 1;
	unsigned match_start = 0, match_length = 0;
	if (tree_vec_counter > 1 && tree_vec_counter < (buffer.size() - 1)) {
	  stree.search(start_index, search_forward_bound, match_start, 
		       match_length);
	  if (match_length >= 2) {
	    res.set_match(tree_vec_counter - match_start, match_length);
	  } else {
	    res.set_literal(buffer.at(tree_vec_counter));
	  }
	} else {
	  res.set_literal(buffer.at(tree_vec_counter));
	}
	
	//if both the previous and current results are matches,
	//check for lazy matching
	if (res.get_result_type() == vectorlist_search_result::MATCH &&
	    previous_res.get_result_type() == vectorlist_search_result::MATCH)
	  {
	    //if the current match length is better than the last match, keep 
	    //this one!
	    if (previous_res.get_match().second < res.get_match().second) {
	      //emit the single literal at the beginning of the last match
	      previous_res.set_literal(buffer.at(tree_vec_counter - 1));
	      emit_result(previous_res, waiting);
	      previous_res = res;
	    } else {
	      //emit the previous match and skip for a while
	      skip_counter = previous_res.get_match().second - 2;
	      emit_result(previous_res, waiting);
	      previous_res.nullify();
	      pending_match = false;
	    }
	  } else
	  {//otherwise emit.  don't skip under any circumstances
	    if (pending_match) {
	      skip_counter = previous_res.get_match().second - 2;
	      emit_result(previous_res, waiting);
	      previous_res.nullify();
	      pending_match = false;
	    } else {
	      if (res.get_result_type() == vectorlist_search_result::MATCH) {
		//set this as the previous result.  may be used for 
		//lazy matching next round
		previous_res = res;
		pending_match = true;
	      } else {
		previous_res.nullify();
		emit_result(res, waiting);
	      }
	    }
	  }
      }
      if (tree_vec_counter < GENEZIP_MAX_OFFSET_POINTER) {
	stree.add_node_to_root(buffer.at(tree_vec_counter), tree_vec_counter);
      } else {
	stree.add_and_delete(buffer.at(tree_vec_counter), tree_vec_counter);
      }

      //clean the tree periodically
      if (tree_vec_counter && (tree_vec_counter % (32768<<3) == 0)) {
	//std::cout << "cleaning" << std::endl;
	stree.dfs(NULL, true, tree_vec_counter-65536);
      }
    }
    //chomp the remaining values?
    if (pending_match) {
      emit_result(previous_res, waiting);
      previous_res.nullify();
      pending_match = false;
    }
    //have to flush an end of block signal
    res.set_end_of_block();
    emit_result(res, waiting);
    //flush the generator
    _generator.flush_data(intermediate_data_holder,
			  litlen_code,
			  distance_code);
}
