/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file helper_functions.h
 */

#ifndef __GENEZIP__HELPER_FUNCTIONS_H__
#define __GENEZIP__HELPER_FUNCTIONS_H__
#include "genezip/config.h"
#include <string>
#include <sstream>
#include <stdexcept>
#include <iostream>
#include <fstream>
#include <vector>
#include <cstdlib>
#include "genezip/defines.h"

namespace genezip_utils {
  /*!
    \brief stringify an object
    \tparam value_type class of object to be converted
    \param obj object to be converted
    \return string representation of object
    \warning uses sstream object conversion rules
  */
  template <class value_type>
    std::string to_string(const value_type &obj) {
    std::ostringstream o;
    if (!(o << obj))
      throw std::domain_error("cannot convert object to string");
    return o.str();
  }
  /*!
    \brief generate an object from its string representation
    \tparam value_type class of object to be converted
    \param str string representation of desired object
    \return object specified by string representation
    \warning uses sstream string conversion rules
   */
  template <class value_type>
    value_type from_string(const std::string &str) {
    value_type res;
    std::istringstream strm1(str);
    if (!(strm1 >> res))
      throw std::domain_error("cannot convert string to object: \"" 
			      + str + "\"");
    return res;
  }
  //! sort std::pair<unsigned, unsigned> by first unsigned value
  //! \param p1 first pair in comparison
  //! \param p2 second pair in comparison
  //! \return whether p1.first < p2.first
  inline bool first_in_pair_sort(const std::pair<unsigned, unsigned> &p1,
				 const std::pair<unsigned, unsigned> &p2) {
    return p1.first < p2.first;
  }
  //! test string equality, ignoring case
  //! \param s1 first string to be compared
  //! \param s2 second string to be compared
  //! \return whether the strings are the same, regardless of case
  inline bool cicompare(const std::string &s1,
			const std::string &s2) {
    if (s1.size() == s2.size()) {
      for (unsigned i = 0; i < s1.size(); ++i) {
	if (toupper(s1.at(i)) != toupper(s2.at(i))) {
	  return false;
	}
      }
      return true;
    } else return false;
  }
  //! convert allele numbers to allele letters, using standard coding
  //! \param c allele number to be converted
  //! \return conversion of input into [ACGT]
  //! \warning will handle missingness as '0'; throw for not in [0,4]
  inline char allele_numbers_to_letters(char c) {
    if (c == '1') return 'A';
    if (c == '2') return 'G';
    if (c == '3') return 'C';
    if (c == '4') return 'T';
    if (c == '0') return c;
    throw std::domain_error("allele_number_to_letters: unrecognized numeric "
			    "code: " + to_string<char>(c));
  }
}

#endif //__HELPER_FUNCTIONS_H__
