/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file imputed_data_file_format.cc
 */

#include "genezip/imputed_data_file_format.h"

void genezip::imputed_data_file_format::set_defaults() {
  _format_identifier = "";
  _sample_file_suffix = "";
  _snp_file_suffix = "";
  _types_file_suffix = "";
  
  _has_sample_file = false;
  _sample_file_has_types = false;
  _sample_file_has_header = false;
  _sample_file_has_subheader = false;
  _has_snp_file = false;
  _snp_file_has_types = false;
  _snp_file_has_header = false;
  _has_types_file = false;
  _types_file_has_header = false;

  _snp_file_rsid_col = 0;
  _snp_file_chromosome_col = 0;
  _snp_file_position_col = 0;
  _snp_file_gposition_col = 0;
  _snp_file_missing_col = 0;
  _snp_file_frequency_col = 0;
  _snp_file_quality_col = 0;
  _snp_file_start_types_col = 0;

  _sample_file_fid_col = 0;
  _sample_file_iid_col = 0;
  _sample_file_pat_col = 0;
  _sample_file_mat_col = 0;
  _sample_file_sex_col = 0;
  _sample_file_pheno_col = 0;
  _sample_file_missing_col = 0;
  _sample_file_start_types_col = 0;

  _types_file_orientation = UNKNOWN_TYPE_ORIENTATION;
  _types_file_start_types_col = 0;
  _types_format = UNKNOWN_DATA_REPRESENTATION;

  _allele_source = UNKNOWN_ALLELE_SOURCE;
}

unsigned genezip::imputed_data_file_format::allele_col(unsigned i) const {
  if (i > _allele_cols.size())
    throw std::domain_error("imputed_data_file_format::allele_col:"
			    " invalid index (nalleles=" +
			    genezip_utils::to_string<unsigned>
			    (_allele_cols.size()) +
			    ", index_provided=" +
			    genezip_utils::to_string<unsigned>(i) + ")");
  if (!i)
    throw std::domain_error("imputed_data_file_format::snp_file_allele_col:"
			    " null index provided, should be interpreted"
			    " as no-allele-in-format");
  return _allele_cols.at(i - 1);
}

void genezip::imputed_data_file_format::allele_col(unsigned i, 
						   unsigned v) {
  if (!i)
    throw std::domain_error("imputed_data_file_format::allele_col ("
			    "set): null index provided, should be interpreted"
			    " as no-allele-in-format");
  if (i > _allele_cols.size())
    _allele_cols.resize(i, 0);
  _allele_cols.at(i - 1) = v;
}

void genezip::imputed_data_file_format::set_as_impute_v2() {
  format_identifier("IMPUTEV2");
  //sample_file_suffix(".sample");
  snp_file_suffix("");
  //types_file_suffix("");

  has_sample_file(false);
  /*
  sample_file_has_types(false);
  sample_file_has_header(true);
  sample_file_has_subheader(true);
  */
  has_snp_file(true);
  snp_file_has_types(true);
  snp_file_has_header(false);
  has_types_file(false);
  
  snp_file_chromosome_col(1);
  snp_file_rsid_col(2);
  snp_file_position_col(3);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(0);
  snp_file_quality_col(0);
  snp_file_start_types_col(6);
  /*
  sample_file_fid_col(2);
  sample_file_iid_col(1);
  sample_file_pat_col(0);
  sample_file_mat_col(0);
  sample_file_sex_col(0);
  sample_file_pheno_col(0);
  sample_file_missing_col(3);
  sample_file_start_types_col(0);
  */
  types_format(PROBABILITIES_FULLPOSTERIOR);

  clear_stored_alleles();
  allele_col(1, 4);
  allele_col(2, 5);
  allele_source(SNP_FILE);
}

void genezip::imputed_data_file_format::set_as_mach_mlgeno() {
  format_identifier("MACHMLGENO");
  sample_file_suffix(".mlprob");
  snp_file_suffix(".mlinfo");

  has_sample_file(true);
  sample_file_has_types(true);
  sample_file_has_header(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(true);
  has_types_file(false);

  snp_file_rsid_col(1);
  snp_file_chromosome_col(0);
  snp_file_position_col(0);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(4);
  snp_file_quality_col(7);
  snp_file_start_types_col(0);

  sample_file_fid_col(0);
  sample_file_iid_col(1);
  sample_file_pat_col(0);
  sample_file_mat_col(0);
  sample_file_sex_col(0);
  sample_file_pheno_col(0);
  sample_file_missing_col(0);
  sample_file_start_types_col(3);

  types_format(GENOTYPES_AS_LETTERS);

  clear_stored_alleles();
  allele_col(1, 2);
  allele_col(2, 3);
  allele_source(SNP_FILE);
}

void genezip::imputed_data_file_format::set_as_mach_mldose() {
  format_identifier("MACHMLDOSE");
  sample_file_suffix(".mldose");
  snp_file_suffix(".mlinfo");

  has_sample_file(true);
  sample_file_has_types(true);
  sample_file_has_header(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(true);
  has_types_file(false);

  snp_file_rsid_col(1);
  snp_file_chromosome_col(0);
  snp_file_position_col(0);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(4);
  snp_file_quality_col(7);
  snp_file_start_types_col(0);

  sample_file_fid_col(0);
  sample_file_iid_col(1);
  sample_file_pat_col(0);
  sample_file_mat_col(0);
  sample_file_sex_col(0);
  sample_file_pheno_col(0);
  sample_file_missing_col(0);
  sample_file_start_types_col(3);

  types_format(DOSAGES_FIRSTALLELE);

  clear_stored_alleles();
  allele_col(1, 2);
  allele_col(2, 3);
  allele_source(SNP_FILE);
}

void genezip::imputed_data_file_format::set_as_mach_mlprob() {
  format_identifier("MACHMLPROB");
  sample_file_suffix(".mlprob");
  snp_file_suffix(".mlinfo");

  has_sample_file(true);
  sample_file_has_types(true);
  sample_file_has_header(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(true);
  has_types_file(false);

  snp_file_rsid_col(1);
  snp_file_chromosome_col(0);
  snp_file_position_col(0);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(4);
  snp_file_quality_col(7);
  snp_file_start_types_col(0);

  sample_file_fid_col(0);
  sample_file_iid_col(1);
  sample_file_pat_col(0);
  sample_file_mat_col(0);
  sample_file_sex_col(0);
  sample_file_pheno_col(0);
  sample_file_missing_col(0);
  sample_file_start_types_col(3);

  types_format(PROBABILITIES_TWOPOSTERIOR);

  clear_stored_alleles();
  allele_col(1, 2);
  allele_col(2, 3);
  allele_source(SNP_FILE);
}

void genezip::imputed_data_file_format::set_as_beagle_phased() {
  format_identifier("BEAGLEPHASED");
  snp_file_suffix(".r2");
  types_file_suffix(".phased");

  has_sample_file(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(false);
  has_types_file(true);
  types_file_has_header(true);

  snp_file_rsid_col(1);
  snp_file_chromosome_col(0);
  snp_file_position_col(0);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(0);
  snp_file_quality_col(2);
  snp_file_start_types_col(0);

  types_file_orientation(SNP_MAJOR);
  types_file_start_types_col(3);
  types_format(GENOTYPES_PHASED);

  clear_stored_alleles();
}

void genezip::imputed_data_file_format::set_as_beagle_dose() {
  format_identifier("BEAGLEDOSE");
  snp_file_suffix(".r2");
  types_file_suffix(".dose");

  has_sample_file(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(false);
  has_types_file(true);
  types_file_has_header(true);

  snp_file_rsid_col(1);
  snp_file_chromosome_col(0);
  snp_file_position_col(0);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(0);
  snp_file_quality_col(2);
  snp_file_start_types_col(0);

  types_file_orientation(SNP_MAJOR);
  types_file_start_types_col(4);
  types_format(DOSAGES_FIRSTALLELE);

  clear_stored_alleles();
  allele_col(1, 2);
  allele_col(2, 3);
  allele_source(TYPES_FILE);
}

void genezip::imputed_data_file_format::set_as_beagle_gprobs() {
  format_identifier("BEAGLEGPROBS");
  snp_file_suffix(".r2");
  types_file_suffix(".gprobs");

  has_sample_file(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(false);
  has_types_file(true);
  types_file_has_header(true);

  snp_file_rsid_col(1);
  snp_file_chromosome_col(0);
  snp_file_position_col(0);
  snp_file_gposition_col(0);
  snp_file_missing_col(0);
  snp_file_frequency_col(0);
  snp_file_quality_col(2);
  snp_file_start_types_col(0);

  types_file_orientation(SNP_MAJOR);
  types_file_start_types_col(4);
  types_format(PROBABILITIES_FULLPOSTERIOR);

  clear_stored_alleles();
  allele_col(1, 2);
  allele_col(2, 3);
  allele_source(TYPES_FILE);
}

void genezip::imputed_data_file_format::set_as_plink_ped() {
  format_identifier("PLINKPED");
  sample_file_suffix(".ped");
  snp_file_suffix(".map");

  has_sample_file(true);
  sample_file_has_types(true);
  sample_file_has_header(false);
  sample_file_has_subheader(false);
  has_snp_file(true);
  snp_file_has_types(false);
  snp_file_has_header(false);
  has_types_file(false);

  snp_file_rsid_col(2);
  snp_file_chromosome_col(1);
  snp_file_position_col(4);
  snp_file_gposition_col(3);
  snp_file_missing_col(0);
  snp_file_frequency_col(0);
  snp_file_quality_col(0);
  snp_file_start_types_col(0);

  sample_file_fid_col(1);
  sample_file_iid_col(2);
  sample_file_pat_col(3);
  sample_file_mat_col(4);
  sample_file_sex_col(5);
  sample_file_pheno_col(6);
  sample_file_missing_col(0);
  sample_file_start_types_col(7);
  
  clear_stored_alleles();
}


genezip::imputed_data_file_format 
genezip::parse_imputed_data_format_from_file(const std::string &filename) {
  imputed_data_file_format custom_format;
  genezip_utils::file_io_handler input(filename,
				       genezip_utils::file_io_handler::READ);
  std::string line = "", token = "", value = "";
  unsigned value1 = 0, value2 = 0;
  std::string::size_type pos = 0;

  typedef void (imputed_data_file_format::*us_ptr)(unsigned u);
  typedef std::pair<std::string, us_ptr> p_us;
  typedef void (imputed_data_file_format::*bo_ptr)(bool b);
  typedef std::pair<std::string, bo_ptr> p_bo;
  typedef void (imputed_data_file_format::*st_ptr)(const std::string &s);
  typedef std::pair<std::string, st_ptr> p_st;

  std::vector<p_us> u_tokens;
  std::vector<p_us>::const_iterator u_iter;
  std::vector<p_bo> b_tokens;
  std::vector<p_bo>::const_iterator b_iter;
  std::vector<p_st> s_tokens;
  std::vector<p_st>::const_iterator s_iter;

  b_tokens.push_back(p_bo("has_sample_file",
			  &imputed_data_file_format::has_sample_file));
  b_tokens.push_back(p_bo("sample_file_has_types",
			  &imputed_data_file_format::sample_file_has_types));
  b_tokens.push_back(p_bo("sample_file_has_header",
			  &imputed_data_file_format::sample_file_has_header));
  b_tokens.push_back(p_bo("sample_file_has_subheader",
			  &imputed_data_file_format::sample_file_has_subheader)
		     );
  b_tokens.push_back(p_bo("has_snp_file",
			  &imputed_data_file_format::has_snp_file));
  b_tokens.push_back(p_bo("snp_file_has_types",
			  &imputed_data_file_format::snp_file_has_types));
  b_tokens.push_back(p_bo("snp_file_has_header",
			  &imputed_data_file_format::snp_file_has_header));
  b_tokens.push_back(p_bo("has_types_file",
			  &imputed_data_file_format::has_types_file));
  b_tokens.push_back(p_bo("types_file_has_header",
			  &imputed_data_file_format::types_file_has_header));
  u_tokens.push_back(p_us("snp_file_rsid_col",
			  &imputed_data_file_format::snp_file_rsid_col));
  u_tokens.push_back(p_us("snp_file_chromosome_col",
			  &imputed_data_file_format::snp_file_chromosome_col));
  u_tokens.push_back(p_us("snp_file_position_col",
			  &imputed_data_file_format::snp_file_position_col));
  u_tokens.push_back(p_us("snp_file_gposition_col",
			  &imputed_data_file_format::snp_file_gposition_col));
  u_tokens.push_back(p_us("snp_file_missing_col",
			  &imputed_data_file_format::snp_file_missing_col));
  u_tokens.push_back(p_us("snp_file_frequency_col",
			  &imputed_data_file_format::snp_file_frequency_col));
  u_tokens.push_back(p_us("snp_file_quality_col",
			  &imputed_data_file_format::snp_file_quality_col));
  u_tokens.push_back(p_us("snp_file_start_types_col",
			  &imputed_data_file_format::snp_file_start_types_col)
		     );
  u_tokens.push_back(p_us("sample_file_fid_col",
			  &imputed_data_file_format::sample_file_fid_col));
  u_tokens.push_back(p_us("sample_file_iid_col",
			  &imputed_data_file_format::sample_file_iid_col));
  u_tokens.push_back(p_us("sample_file_pat_col",
			  &imputed_data_file_format::sample_file_pat_col));
  u_tokens.push_back(p_us("sample_file_mat_col",
			  &imputed_data_file_format::sample_file_mat_col));
  u_tokens.push_back(p_us("sample_file_sex_col",
			  &imputed_data_file_format::sample_file_sex_col));
  u_tokens.push_back(p_us("sample_file_pheno_col",
			  &imputed_data_file_format::sample_file_pheno_col));
  u_tokens.push_back(p_us("sample_file_missing_col",
			  &imputed_data_file_format::sample_file_missing_col));
  u_tokens.push_back(p_us("sample_file_start_types_col",
			  &imputed_data_file_format::sample_file_start_types_col));
  u_tokens.push_back(p_us("types_file_start_types_col",
			  &imputed_data_file_format::types_file_start_types_col));

  s_tokens.push_back(p_st("format_identifier",
			  &imputed_data_file_format::format_identifier));
  s_tokens.push_back(p_st("sample_file_suffix",
			  &imputed_data_file_format::sample_file_suffix));
  s_tokens.push_back(p_st("snp_file_suffix",
			  &imputed_data_file_format::snp_file_suffix));
  s_tokens.push_back(p_st("types_file_suffix",
			  &imputed_data_file_format::types_file_suffix));
  
  while (input.readline(line)) {
    if ((pos = line.find_first_not_of(" \t")) != std::string::npos) {
      if (line.at(pos) == '#') continue;
      //parse the token
      std::istringstream strm1(line);
      if (!(strm1 >> token))
	throw std::domain_error("parse_imputed_data_format_from_file: "
				"cannot read token from line: \"" + line +
				"\" in file \"" +
				filename + "\"");
      bool found_token_match = false;
      for (b_iter = b_tokens.begin(); 
	   !found_token_match && b_iter != b_tokens.end();
	   ++b_iter) {
	if (genezip_utils::cicompare(b_iter->first, token)) {
	  //allow any number of boolean responses to this
	  if (!(strm1 >> value))
	    throw std::domain_error("parse_imputed_data_format_from_file: "
				    "cannot read value for token \"" +
				    token + "\" in file \"" +
				    filename + "\"");
	  bool is_false = genezip_utils::cicompare(value, "f") ||
	    genezip_utils::cicompare(value, "false") ||
	    genezip_utils::cicompare(value, "n") ||
	    genezip_utils::cicompare(value, "no") ||
	    genezip_utils::cicompare(value, "0");
	  ((custom_format).*(b_iter->second))(!is_false);
	  found_token_match = true;
	}
      }
      for (u_iter = u_tokens.begin();
	   !found_token_match && u_iter != u_tokens.end();
	   ++u_iter) {
	if (genezip_utils::cicompare(u_iter->first, token)) {
	  if (!(strm1 >> value))
	    throw std::domain_error("parse_imputed_data_format_from_file: "
				    "cannot read value for token \"" +
				    token + "\" in file \"" +
				    filename + "\"");
	  ((custom_format).*(u_iter->second))
	    (genezip_utils::from_string<unsigned>(value));
	  found_token_match = true;
	}
      }
      for (s_iter = s_tokens.begin();
	   !found_token_match && s_iter != s_tokens.end();
	   ++s_iter) {
	if (genezip_utils::cicompare(s_iter->first, token)) {
	  if (!(strm1 >> value))
	    throw std::domain_error("parse_imputed_data_format_from_file: "
				    "cannot read value for token \"" +
				    token + "\" in file \"" +
				    filename + "\"");
	  ((custom_format).*(s_iter->second))(value);
	  found_token_match = true;
	}
      }
      if (!found_token_match &&
	  genezip_utils::cicompare(token, "allele_source")) {
	found_token_match = true;
	if (!(strm1 >> value))
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot read value for token \"" +
				  token + "\" in file \"" +
				  filename + "\"");
	if (genezip_utils::cicompare(value, "snp_file")) {
	  custom_format.allele_source(SNP_FILE);
	} else if (genezip_utils::cicompare(value, "types_file")) {
	  custom_format.allele_source(TYPES_FILE);
	} else {
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot reconcile enumerated type from "
				  "provided string for " + token + " flag "
				  "(expected snp_file or types_file, "
				  "encountered \"" + value + "\") in file " +
				  filename + "\"");
	}
      }
      if (!found_token_match &&
	  genezip_utils::cicompare(token, "allele_col")) {
	found_token_match = true;
	if (!(strm1 >> value1 >> value2))
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot read values for token \"" +
				  token + "\" (expected two unsigneds)"
				  " in file \"" + filename + "\"");
	custom_format.allele_col(value1, value2);
      }
      if (!found_token_match &&
	  genezip_utils::cicompare(token, "types_format")) {
	found_token_match = true;
	if (!(strm1 >> value))
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot read value for token \"" +
				  token + "\" in file \"" +
				  filename + "\"");
	if (genezip_utils::cicompare(value, "genotypes_as_letters")) {
	  custom_format.types_format(GENOTYPES_AS_LETTERS);
	} else if (genezip_utils::cicompare(value, "genotypes_as_numbers")) {
	  custom_format.types_format(GENOTYPES_AS_NUMBERS);
	} else if (genezip_utils::cicompare(value, "genotypes_phased")) {
	  custom_format.types_format(GENOTYPES_PHASED);
	} else if (genezip_utils::cicompare(value, "dosages_firstallele")) {
	  custom_format.types_format(DOSAGES_FIRSTALLELE);
	} else if (genezip_utils::cicompare(value, "dosages_bothalleles")) {
	  custom_format.types_format(DOSAGES_BOTHALLELES);
	} else if (genezip_utils::cicompare(value,
					    "probabilities_twoposterior")) {
	  custom_format.types_format(PROBABILITIES_TWOPOSTERIOR);
	} else if (genezip_utils::cicompare(value,
					    "probabilities_fullposterior")) {
	  custom_format.types_format(GENOTYPES_AS_NUMBERS);
	} else {
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot reconcile enumerated type from "
				  "provided string for " + token + " flag "
				  "(expected one of: {genotypes_as_letters, "
				  "genotypes_as_numbers, genotypes_phased, "
				  "dosages_firstallele, dosages_bothalleles, "
				  "probabilities_twoposterior, "
				  "probabilities_fullposterior}, "
				  "encountered \"" + value + "\") in file " +
				  filename + "\"");
	}
      }
      if (!found_token_match &&
	  genezip_utils::cicompare(token, "types_file_orientation")) {
	found_token_match = true;
	if (!(strm1 >> value))
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot read value for token \"" +
				  token + "\" in file \"" +
				  filename + "\"");
	if (genezip_utils::cicompare(value, "snp_major")) {
	  custom_format.types_file_orientation(SNP_MAJOR);
	} else if (genezip_utils::cicompare(value, "individual_major")) {
	  custom_format.types_file_orientation(INDIVIDUAL_MAJOR);
	} else {
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot reconcile enumerated type from "
				  "provided string for " + token + " flag "
				  "(expected snp_major or individual_major, "
				  "encountered \"" + value + "\") in file " +
				  filename + "\"");
	}
      }
      if (!found_token_match &&
	  genezip_utils::cicompare(token, "format_identifier")) {
	found_token_match = true;
	if (!(strm1 >> value))
	  throw std::domain_error("parse_imputed_data_format_from_file: "
				  "cannot read value for token \"" +
				  token + "\" in file \"" +
				  filename + "\"");
	custom_format.format_identifier(value);
      }
      if (!found_token_match) {
	throw std::domain_error("parse_imputed_data_format_from_file: "
				"cannot find token match for \"" +
				token + "\" in file \"" +
				filename + "\"");
      }
    }
  }
  input.close();
  return custom_format;
}

void genezip::imputed_data_file_format::report(std::ostream &out) const {
  out << "format identifier: " << format_identifier() << std::endl
      << "sample file suffix: " << sample_file_suffix() << std::endl
      << "snp file suffix: " << snp_file_suffix() << std::endl
      << "types file suffix: " << types_file_suffix() << std::endl
      << "has sample file: " << has_sample_file() << std::endl
      << "sample file has types: " << sample_file_has_types() << std::endl
      << "sample file has header: " << sample_file_has_header() << std::endl
      << "sample file has subheader: " << sample_file_has_subheader() << std::endl
      << "has snp file: " << has_snp_file() << std::endl
      << "snp file has types: " << snp_file_has_types() << std::endl
      << "snp file has header: " << snp_file_has_header() << std::endl
      << "has types file: " << has_types_file() << std::endl
      << "types file has header: " << types_file_has_header() << std::endl
      << "snp file rsid col: " << snp_file_rsid_col() << std::endl
      << "snp file chromosome col: " << snp_file_chromosome_col() << std::endl
      << "snp file position col: " << snp_file_position_col() << std::endl
      << "snp file gposition col: " << snp_file_gposition_col() << std::endl
      << "snp file missing col: " << snp_file_missing_col() << std::endl
      << "snp file frequency col: " << snp_file_frequency_col() << std::endl
      << "snp file quality col: " << snp_file_quality_col() << std::endl
      << "snp file start types col: " << snp_file_start_types_col() << std::endl
      << "sample file fid col: " << sample_file_fid_col() << std::endl
      << "sample file iid col: " << sample_file_iid_col() << std::endl
      << "sample file pat col: " << sample_file_pat_col() << std::endl
      << "sample file mat col: " << sample_file_mat_col() << std::endl
      << "sample file sex col: " << sample_file_sex_col() << std::endl
      << "sample file pheno col: " << sample_file_pheno_col() << std::endl
      << "sample file missing col: " << sample_file_missing_col() << std::endl
      << "sample file start types col: " << sample_file_start_types_col() << std::endl
      << "types file orientation: " << types_file_orientation() << std::endl
      << "types file start types col: " << types_file_start_types_col() << std::endl
      << "types format: " << types_format() << std::endl
      << "allele cols:";
  for (unsigned i = 0; i < _allele_cols.size(); ++i)
    out << ' ' << _allele_cols.at(i);
  out << std::endl
      << "allele source: " << allele_source() << std::endl;
}
