/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file multilevel_huffman_hash.h
  \brief decode a huffman code bit sequence

  The description of this hash is in http://www.gzip.org/algorithm.txt (again,
  implementation is mine).  Briefly, for a code of X bits, the first Y bits 
  are encoded in one table; if X>Y, the remaining X-Y bits are encoded in a 
  second level of table at the pointer in position Y in this hash.  The number
  of bits handled in a level of this hash is specified by gzip (9 for 
  literal/match length code; 6 for match offset), but should probably be
  optimized for this usage.  If the first level of the hash handles B bits,
  and the search code is fewer bits than that, the value is padded with 0s 
  on the LSB side.  When generating the hash, for codes with fewer bits 
  than B, all entries with the given bit prefix are filled with the same 
  decoded value.
 */
#ifndef __GENEZIP__MULTILEVEL_HUFFMAN_HASH_H__
#define __GENEZIP__MULTILEVEL_HUFFMAN_HASH_H__

#include <vector>
#include <iostream>
#include <string>
#include <stdexcept>
#include <utility>
#include <map>
#include "genezip/helper_functions.h"
namespace genezip_utils {
  /*!
    \class multilevel_huffman_hash
    \brief decode a huffman code in a multilevel hash
  
    See description of file for more information
  */
  class multilevel_huffman_hash {
  public:
    /*!
      \brief constructor
    */
    multilevel_huffman_hash() : _tranche_one_threshold(0) {}
    /*!
      \brief destructor
    */
    ~multilevel_huffman_hash() throw() {clear_table();}
    /*!
      \brief fill the hash with huffman codes
      @param huffman_data all huffman codes and associated values
      @param tranche_one_threshold the number of bits to be handled at the 
      first level of hash
      @param max_bit_length number of bits in the longest huffman code
    */
    void populate(const std::vector<std::pair<unsigned, 
		  std::pair<unsigned, unsigned> > > &huffman_data,
		  unsigned tranche_one_threshold,
		  unsigned max_bit_length);
    /*!
      \brief decode a huffman code using a prepopulated hash
      @param huffcode encoded huffman code
      @param length number of bits in the current code
      \return decoded value
    */  
    unsigned translate(unsigned huffcode, unsigned length) const;
    //! get the number of bytes used to store this object
    //! \return the number of bytes used to store this object
    inline unsigned bytes_of_compressed_memory() const {
      unsigned res = sizeof(unsigned);
      for (std::vector<std::pair<unsigned, multilevel_huffman_hash *> >::const_iterator iter
	     = _lookup_table.begin(); iter != _lookup_table.end(); ++iter) {
	res += sizeof(unsigned) + iter->second ? sizeof(multilevel_huffman_hash *) : sizeof(unsigned);
      }
      return res;
    }
  private:
    /*!
      \brief clear the contents of the table
    */
    void clear_table();
    //! recursive hash data structure
    std::vector<std::pair<unsigned, multilevel_huffman_hash *> > _lookup_table;
    //! bit length threshold for current level of recursive table
    unsigned _tranche_one_threshold;
  };
}
#endif //__MULTILEVEL_HUFFMAN_HASH_H__
