/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file output_buffer.h
  \brief during decoding, buffer decompressed values to allow
  reverse references
 */
#ifndef __GENEZIP__OUTPUT_BUFFER_H__
#define __GENEZIP__OUTPUT_BUFFER_H__

#include <vector>
#include <stdexcept>
#include <fstream>
#include <iostream>
#include "genezip/helper_functions.h"
#include "genezip/prob_vector.h"
namespace genezip_utils {
  /*!
    \class output_buffer
    \brief during decoding, buffer decompressed values to allow
    reverse references
  */
  class output_buffer {
  public:
    /*!
      \brief constructor
    */
  output_buffer() : _outvec(0) {}
    /*!
      \brief constructor with preallocation
      \param out pointer to storage vector used by this output buffer
      \param initial_reserve number of literals for which memory
      should be preallocated (to reduce push_back copy/alloc operations)
    */
  output_buffer(prob_vector *out,
		unsigned initial_reserve) : _outvec(out) {
      if (_outvec) _outvec->reserve(initial_reserve);
    }
    /*!
      \brief copy constructor
      @param other existing object
    */
    output_buffer(const output_buffer &other) :  _outvec(other._outvec) {}
    /*!
      \brief destructor
    */
    ~output_buffer() throw() {}
    /*!
      \brief set the pointer to the output "stream" used by this buffer
      @param out pointer to vectur output "stream"
     */
    inline void set_stream(prob_vector *out) {_outvec = out;}
    /*!
      \brief add a literal to the buffer
      @param u new literal for the buffer
    
      basically a deprecated wrapper, sorry.
    */
    inline void add_literal(unsigned u) {add_safely_to_buffer(u);}
    /*!
      \brief add a reverse reference (expanded) to the output buffer
      @param distance_from_leading_edge offset from current location
      @param number_to_repeat length of match
    */
    void repeat_sequence(unsigned distance_from_leading_edge,
			 unsigned number_to_repeat) {
      if (_outvec->size() < distance_from_leading_edge)
	throw std::domain_error("genezip_utils::output_buffer::repeat_"
				"sequence: invalid distance offset: \"" 
				+ to_string<unsigned>
				(distance_from_leading_edge)
				+ "\" relative to buffer size="
				+ to_string<unsigned>(_outvec->size()));
      unsigned start_index = _outvec->size() - distance_from_leading_edge;
      unsigned u = 0;
      for (unsigned num_added = 0; 
	   num_added < number_to_repeat;
	   ++num_added, ++start_index) {
	_outvec->at(start_index, u);
	add_safely_to_buffer(u);
      }
    }
    /*!
      \brief get the current insertion index for the buffer
      \return the current insertion index for the buffer
    */
    inline unsigned leading_edge() const {
      return _outvec ? _outvec->size() : 0;
    }
  private:
    /*!
      \brief add a literal to the buffer
      @param u literal to add to buffer
    */
    inline void add_safely_to_buffer(unsigned u) {
      if (_outvec) _outvec->push_back(u);
    }
    /*!
      \brief get the size of the buffer
      \return the size of the buffer
    */
    inline std::vector<unsigned>::size_type size() {
      return _outvec ? _outvec->size() : 0;
    }
    prob_vector *_outvec;//!< output target, to write to vector
  };
}
#endif //__OUTPUT_BUFFER_H__
