/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
 * \file prob_vector.h
 * \brief stored imputed data in compressed formate
 */
#ifndef __GENEZIP__PROB_VECTOR__
#define __GENEZIP__PROB_VECTOR__
#include <vector>
#include <stdexcept>
#include <cmath>
namespace genezip_utils {
  /*!
   * \brief use stl vector to store data, but store it compressed to save 
   lots of space
  */
  class prob_vector {
  public:
    /*!
     * \brief stl-like typedef for internal storage type
     */
    typedef unsigned value_type;
    /*!
     * \brief stl-like typedef for internal storage vector
     */
    typedef std::vector<value_type> vector;
    /*!
     * \brief stl-like typedef for internal storage size
     */
    typedef vector::size_type size_type;
    /*!
     * \brief constructor: initialize vector
     * @param init optional initial vector allocation size
     * @param n_bits the number of bits you want to use per value
     * \warning do NOT change n_bits unless you know what you're doing
     *
     * Setting the initial size can dramatically improve data insertion time
     */
    prob_vector(const std::vector<value_type>::size_type &init = 0,
		const size_type &n_bits = 10)
      : _used_bound(0), _scaling(n_bits), _max_prec(1) {
      reserve(init);
      _max_prec = (_max_prec << _scaling) - 1;
    }
    /*!
     * \brief copy constructor
     * @param other preexisting prob_vector
     */
    prob_vector(const prob_vector &other)
      : _data(other._data),
      _used_bound(other._used_bound),
      _scaling(other._scaling), 
      _max_prec(other._max_prec) {}
    /*!
     * \brief destructor: all memory is managed
     */
    ~prob_vector() throw() {}
    /*!
      \brief clear out stored data, preparing the vector for reuse
     */
    void clear() {
      _data.clear();
      _used_bound = 0;
    }

    /*!
     * \brief append a value to the end of the vector (double)
     * @param value new double to be added to vector
     * \warning any precision beyond [01].\\d\\d\\d will be ignored
     */
    void push_back(const double &value);
    /*!
     * \brief append a value to the end of the vector (unsigned)
     * @param value new unsigned to be added to vector
     * \warning adds the value raw, without the conversion applied
     * to doubles
     */
    void push_back(unsigned value);
    //! \brief append a value to the end of the vector (int)
    //! @param value new int to be added to the vector
    //! \warning casts int to unsigned, so negative values will underflow
    void push_back(int value) {push_back(static_cast<unsigned>(value));}
    /*!
     * \brief set a (double) value at a given index of the vector
     * @param index location in vector for insertion
     * @param value datum to be inserted
     */
    void set(const size_type &index, const double &value);
    /*!
     * \brief set a (unsigned) value at a given index of the vector
     * @param index location in vector for insertion
     * @param value datum to be inserted
     * \warning return value is returned raw from storage
     */
    void set(const size_type &index, unsigned value);
    /*!
     * \brief access a value at a given location in the vector
     * @param index location in the vector
     * @param value destination for stored value (as double)
     * \warning returns a value converted from the stored unsigned
     */
    void at(const size_type &index, double &value) const;
    /*!
     * \brief access a value at a given location in the vector
     * @param index location in the vector
     * @param value destination for stored value (as unsigned)
     * \warning returns raw value stored in vector
     */
    void at(const size_type &index, unsigned &value) const;
    /*!
     * \brief determine how many values are stored
     * \return the number of stored values
     */
    inline size_type size() const {return _used_bound;}
    /*!
     * \brief determine if the vector is empty
     * \return whether the vector is empty
     */
    inline bool empty() const {return size() == 0;}
    /*!
     * \brief allocate but do not fill some data
     * @param amt amount of space to allocate
     *
     * this will not change data currently stored in the vector,
     * but may require a copy
     */
    inline void reserve(const size_type &amt) {
      _data.reserve(static_cast<size_type>(ceil(amt *
						static_cast<double>(_scaling) 
						/ 8.0 / sizeof(value_type))));
    }
    //! get the number of bits used per value
    //! \return the number of bits used per value
    inline unsigned bits_per_element() const {return _scaling;}
    //inline const vector &internal() const throw() {return _data;}
  private:
    /*!
     * \brief data vector
     */
    vector _data;
    /*!
     * \brief internal max_loc+1 of stored data
     */
    size_type _used_bound;
    /*!
     * \brief scaling factor for storage
     *
     * Basically, the number of bits you want
     */
    unsigned _scaling;
    /*!
     * \brief calculated maximum precision
     */
    unsigned _max_prec;
  };
}
#endif //__PROB_VECTOR__
