/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file smart_pointer.h
  \brief a small implementation of a smart pointer

  Note that this class is for pointers allocated with new, not new[].  Some
  compilers accept both delete syntaxes with all allocated pointers, but others
  don't.  Normal pointer operations are supported.
 */
#ifndef __GENEZIP__SMART_POINTER_H__
#define __GENEZIP__SMART_POINTER_H__

#include <stdexcept>
namespace genezip_utils {
  /*!
    \class smart_pointer
    \brief a wrapper class that automatically deletes a pointer when the
    last reference is removed
    @tparam value_type the class to which the pointer... points?
  */
  template <class value_type>
    class smart_pointer {
  public:
    /*!
      \brief constructor
    */
    smart_pointer() : _ptr(0), _count(0) {}
    /*!
      \brief constructor: provide a pointer that this class will manage
      @param ptr pointer to be managed
      \warning this constructor should only be used for nonmanaged pointers.
      This WILL delete the pointer
    */
    smart_pointer(value_type *ptr);
    /*!
      \brief copy constructor
      @param other existing smart pointer
    */
    smart_pointer(const smart_pointer &other);
    /*!
      \brief destructor
    */
    ~smart_pointer() throw();
    /*!
      \brief get the stored pointer.  Pointer will still be managed
      \return the managed pointer.  Still considered managed by this object
    */
    value_type *get() const {return _ptr;}
    /*!
      \brief set the stored pointer.  Will manage the pointer/remove 
      any currently managed pointer
      @param ptr new pointer to be managed
    */
    void set(value_type *ptr);
    /*!
      \brief dereferencing operator.  will throw if there is no managed pointer
      \return reference to object at stored address
    */
    value_type &operator*() const;
    /*!
      \brief dereferencing operator.  will throw if there is no managed pointer
      \return managed pointer (for recursive call)
    */
    value_type *operator->() const;
    //! assignment operator: copy over existing object
    //! @param other existing object
    //! \return reference to this object
    smart_pointer &operator=(const smart_pointer &other);
    /*!
      \brief determine how many copies of this pointer exist
      \return number of copies of the managed pointer (including *this)
    */
    unsigned get_count() const {return _count ? (*_count) : 0;}
    //! get the pointer to the count variable for this pointer
    //! \return pointer to the count variable for this pointer
    unsigned *get_count_pointer() const {return _count;}
  private:
    /*!
      \brief allocate the count variable
      @param initial_count first value to store in count (presumably always 1)
    */
    void allocate_count(unsigned initial_count);
    /*!
      \brief remove the internal pointers; delete them if count==1
    */
    void clear_internals();
    //! the managed pointer
    value_type *_ptr;
    //! a shared counter of the number of instances of this pointer
    unsigned *_count;
  };



  template <class value_type>
    smart_pointer<value_type>::smart_pointer(value_type *ptr) 
    : _ptr(ptr), 
    _count(0) {
    if (_ptr) allocate_count(1);
  }

  template <class value_type>
    smart_pointer<value_type>::smart_pointer(const smart_pointer &other) 
    : _ptr(other._ptr), 
    _count(other._count) {
    if (_count) ++(*_count);
  }

  template <class value_type>
    smart_pointer<value_type> &smart_pointer<value_type>::operator=
    (const smart_pointer &other) {
    if (this != &other) {
      clear_internals();
      _ptr = other.get();
      _count = other.get_count_pointer();
      if (_count) ++(*_count);
    }
    //set(other.get());
  }

  template <class value_type>
    smart_pointer<value_type>::~smart_pointer() throw() {
    clear_internals();
  }

  template <class value_type>
    void smart_pointer<value_type>::set(value_type *ptr) {
    if (_ptr) clear_internals();
    if (_ptr = ptr) allocate_count(1);
  }

  template <class value_type>
    value_type &smart_pointer<value_type>::operator*() const {
    if (_ptr) return *_ptr;
    throw std::domain_error("smart_pointer::operator*: dereferenced "
			    "nonexistent pointer");
  }

  template <class value_type>
    value_type *smart_pointer<value_type>::operator->() const {
    if (_ptr) return _ptr;
    throw std::domain_error("smart_pointer::operator->: dereferenced "
			    "nonexistent pointer");
  }

  template <class value_type>
    void smart_pointer<value_type>::allocate_count(unsigned initial_count) {
    if (_count) throw std::domain_error("smart_pointer::allocate_count: "
					"count exists");
    _count = new unsigned;
    *_count = initial_count;
  }

  template <class value_type>
    void smart_pointer<value_type>::clear_internals() {
    if (_ptr) {
      if (_count && (*_count) == 1) {
	delete _ptr;
	delete _count;
      } else if (_count) {
	--(*_count);
      }
      _ptr = 0;
      _count = 0;
    }
  }
}
#endif //__SMART_POINTER_H__
