/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file uncompressed_buffer.h
  \brief wrapper for input uncompressed data
 */
#ifndef __GENEZIP__UNCOMPRESSED_BUFFER_H__
#define __GENEZIP__UNCOMPRESSED_BUFFER_H__

#include <string>
#include <vector>
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdexcept>
#include "genezip/helper_functions.h"
#include "genezip/prob_vector.h"
namespace genezip_utils {
  /*!
    \class uncompressed_buffer
    \brief wrapper for input uncompressed data
  */
  class uncompressed_buffer {
  public:
    /*!
      \brief constructor
    */
    uncompressed_buffer()
      : _buffered_data(NULL),
      _current_buffer_read_index(0),
      _current_buffer_bound_index(0) {}
    /*!
      \brief constructor with data
      @param data data for which this object will be an interface
    */
    //uncompressed_buffer(std::vector<unsigned> &data) throw(std::bad_alloc) : _buffered_bad_data(data),
    //_current_buffer_read_index(0),
    //_current_buffer_bound_index(data.size()) {}
    uncompressed_buffer(prob_vector &data)
      : _buffered_data(&data),
      _current_buffer_read_index(0),
      _current_buffer_bound_index(data.size()) {}
    /*!
      \brief destructor
    */
    ~uncompressed_buffer() throw() {}
    /*!
      \brief read the next literal from the wrapped data vector
      \return the next literal from the wrapped data vector
    */
    unsigned read();
    /*!
      \brief read a literal without incrementing the read pointer, if a
      literal is available
      @param u1 target which will contain the next literal, if one is available
      \return the number of literals read (either 1 or 0)
    */
    unsigned read_nonincrement(unsigned *u1);
    /*!
      \brief read two literals, but only increment the read pointer once
      @param u1 target which will contain the next literal, if one is available
      @param u2 target which will contain the next next literal, if one is available
      \return the number of literals read (2/1/0; if 1, will be in c1)
    */
    unsigned read_two_characters(unsigned *u1, unsigned *u2);
    /*!
      \brief determine whether there are more literals to read
      \return whether there are more literals to read

      Nomenclature is left over from a previous version of the library.
    */
    inline bool eof() {
      return _current_buffer_read_index == _current_buffer_bound_index;
    }
    /*!
      \brief try to match the current read position with literals some 
      distance back
      @param reverse_offset number of literals behind the current 
      location from which the match should start
      \return the length of the observed match (must be at least 2, 
      otherwise the method returns 0)
    */
    unsigned match_with_current(unsigned reverse_offset);
    //! get value at a given position
    //! @param index location of requested value
    //! \return value at specified index
    unsigned at(unsigned index) const {
      if (_buffered_data &&
	  _buffered_data->size() > index) {
	unsigned res = 0;
	_buffered_data->at(index, res);
	return res;
      }
      if (!_buffered_data)
	throw std::domain_error("uncompressed_buffer::at: ptr null");
      else
	throw std::domain_error("uncompressed_buffer::at: index invalid"
				" (requested=" + to_string<unsigned>(index)
				+ ", size=" + 
				to_string<unsigned>(_buffered_data->size())
				+ ")");
    }
    //! get the size of the stored data
    //! \return the size of the stored data
    unsigned size() const {return _buffered_data ? _buffered_data->size() : 0;}
    //! get the number of bits required to store any given element in data
    //! \return number of bits required to store any given element in data
    unsigned bits_per_element() const {
      return _buffered_data ? _buffered_data->bits_per_element() : 0;
    }
  private:
    //! the buffer of data
    prob_vector *_buffered_data;
    //! the current index for read operations
    unsigned _current_buffer_read_index;
    //! the current index bounding valid data
    unsigned _current_buffer_bound_index;
  };
}
#endif //__UNCOMPRESSED_BUFFER_H__
