/*
Copyright 2013 Cameron Palmer

This file is a part of Genezip.

Genezip is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Genezip is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Genezip.  If not, see <http://www.gnu.org/licenses/>
*/

/*!
  \file vectorlist.h
  \brief data structure for hash chaining with linked lists, and for a 
  search result from the hash

  This is an odd data structure where the linked list is represented by 
  nodes, but the nodes are in a vector.  The nodes are singly linked, 
  and the pointers are actually array indices.  The list thus supports 
  random access.  Free and full nodes are maintained in two separate lists 
  (with pointers to both list head and tail), and additions to each list 
  from the other are made at the head of the lists; thus, addition of a 
  node and deletion of a node and all its descendants are constant time 
  operations.

  The list search result object is also herein defined.
 */
#ifndef __GENEZIP__VECTORLIST_H__
#define __GENEZIP__VECTORLIST_H__
#include <vector>
#include <string>
#include <stdexcept>
#include <utility>
namespace genezip_utils {
  /*!
    \class vectorlist_search_result
    \brief results of a search through a vectorlist object
  */
  class vectorlist_search_result {
  public:
    /*!
      \enum SEARCH_RESULT_TYPE
      \brief identifies the data contained within the search result
    */
    typedef enum {
      LITERAL, //!< search result is "literal"
      MATCH, //!< search result is "match"
      ENDOFBLOCK, //!< search result is end of block code
      NONE //!< undefined search result type
    } SEARCH_RESULT_TYPE;
    /*!
      \brief constructor
    */
    vectorlist_search_result()
      : _match(std::pair<unsigned, unsigned>(0,0)),
      _literal(0), 
      _type(NONE) {}
    /*!
      \brief copy constructor
      @param other existing object
    */
    vectorlist_search_result(const vectorlist_search_result &other) 
      : _match(other._match), 
      _literal(other._literal), 
      _type(other._type) {}
    /*!
      \brief destructor
    */
    ~vectorlist_search_result() throw() {}
    /*!
      \brief set the object as a match, with an offset and a length
      @param offset distance from compression front of match
      @param length length of match
    */
    inline void set_match(unsigned offset, unsigned length) {
      _match.first = offset; _match.second = length; _type = MATCH;
    }
    /*!
      \brief set the object as a literal
      @param val new literal for this object
    */
    inline void set_literal(unsigned val) {
      _literal = val; _type = LITERAL;
    }
    /*!
      \brief set the object as an end-of-block code
    */
    inline void set_end_of_block() {_type = ENDOFBLOCK;}
    /*!
      \brief set the object as empty
    */
    inline void nullify() {_type = NONE;}
    /*!
      \brief get the type of the current object
    */
    inline SEARCH_RESULT_TYPE get_result_type() const {return _type;}
    /*!
      \brief get the match offset and length (if valid operation for 
      this object)
      \return (match offset, match length)
    */
    inline const std::pair<unsigned, unsigned> &get_match() const {
      if (get_result_type() != MATCH)
	throw std::domain_error("vectorlist_search_result::get_match: "
				"stored datum is not a match");
      return _match;
    }
    /*!
      \brief get the literal (if valid operation for this object)
      \return literal for this object
    */
    inline unsigned get_literal() const  {
      if (get_result_type() != LITERAL)
	throw std::domain_error("vectorlist_search_result::get_literal: "
				"stored datum is not a literal");
      return _literal;
    }
  private:
    //! distance and length parameters of a "match" result
    std::pair<unsigned, unsigned> _match;
    //! ASCII code for a "literal" result
    unsigned _literal;
    //! type descriptor of search result
    SEARCH_RESULT_TYPE _type;
  };

  /*!
    \class vectorlist
    @tparam value_type storage data type for this class

    Basically a linked list structure.  use value_type=unsigned.
    All method definitions are included in the end of the file, 
    as this is a templated class.
  */
  template <class value_type>
    class vectorlist {
  public:
    /*!
      \typedef typename std::pair<value_type, unsigned> data_pair;
      \brief convenience typedef for a pair
    */
    typedef typename std::pair<value_type, unsigned> data_pair;
    /*!
      \brief constructor
    */
    vectorlist() 
      : _head_of_free_nodes(0), 
      _last_of_free_nodes(0), 
      _head_of_full_nodes(0), 
      _last_of_full_nodes(0) {
      get_more_nodes(1);
    }
    /*!
      \brief copy constructor
    */
  vectorlist(const vectorlist<value_type> &other)
    : _head_of_free_nodes(other._head_of_free_nodes),
      _last_of_free_nodes(other._last_of_free_nodes),
      _head_of_full_nodes(other._head_of_full_nodes), 
      _last_of_full_nodes(other._last_of_full_nodes),
      _data(other._data) {}
    /*!
      \brief destructor
    */
    ~vectorlist() throw() {}
    /*!
      \brief add an element to the list
      @param datum new value to be stored
    */
    void add_element(value_type datum);
    /*!
      \brief remove and return the first element in the list
      \return the value of the (now prior) first element in the list
    */
    value_type pop_element();
    /*!
      \brief determine whether the full node list has something in it
      \return whether the list is nonempty
    */
    inline bool empty() const {return !_head_of_full_nodes;}
    /*!
      \brief get a "pointer" to the beginning of the list.  The index 
      is on [0,size()), but stored as [1,size()], with 0 an error code
      \return pointer to the beginning of the list
    */
    inline unsigned first_node_index() {
      if (empty()) throw std::domain_error("vectorlist::first_node_index: "
					   "empty list!");
      return _head_of_full_nodes - 1;
    }
    /*!
      \brief get the value stored at a specified node
      @param index index of desired node
      \return value at the specified node
      \warning: this will return junk if called without tracing indices 
      through next_node_index/next_node_index_valid
    */
    value_type node_element(unsigned index) {
      if (index >= _data.size())
	throw std::domain_error("vectorlist::node_element: supplied index "
				"is out of bounds: " 
				+ to_string<unsigned>(index) + ", " 
				+ to_string<unsigned>(_data.size()));
      return _data.at(index).first;
    }
    /*!
      \brief determine whether a given node has a nonnull child
      @param index pointer to known node
      \return whether the node has a child
    */
    inline bool next_node_index_valid(unsigned index) const {
      if (index >= _data.size())
	throw std::domain_error("vectorlist::next_node_index_valid: "
				"supplied index is out of bounds");
      if (_data.at(index).second == index+1) 
	throw std::domain_error("vectorlist::next_node_index_valid: "
				"cycle detected (index=" 
				+ to_string<unsigned>(index) + ", size=" 
				+ to_string<unsigned>(_data.size()) + ")");
      return _data.at(index).second;
    }
    /*!
      \brief get the index of the child of a given node
      @param current_index pointer to current node
      \return index of child of given node
    */
    unsigned next_node_index(unsigned current_index) {
      if (current_index >= _data.size())
	throw std::domain_error("vectorlist::next_node_index: supplied "
				"index is out of bounds");
      if (next_node_index_valid(current_index))
	return _data.at(current_index).second - 1;
      throw std::domain_error("vectorlist::next_node_index: current "
			      "index has no child node");
    }
    /*!
      \brief starting at a supplied index, remove all current/descendant 
      nodes from the full list
      @param index current node (first to be removed)
      @param index of parent of current node NB: THIS IS ON [1,size()], 
      @param previous_index_incremented previous node in list, on [1,size()]
      0 MEANS NO PARENT
    */
    void remove_all(unsigned index, unsigned previous_index_incremented) {
      if (index >= _data.size())
	throw std::domain_error("vectorlist::remove_all: supplied index "
				"is out of bounds");
      if (previous_index_incremented > _data.size()) 
	throw std::domain_error("vectorlist::remove_all: supplied previous "
				"index is out of bounds");

      if (previous_index_incremented) {
	_data.at(previous_index_incremented - 1).second = 0;
      } else {
	_head_of_full_nodes = 0;
      }
      if (!_head_of_free_nodes) {
	_head_of_free_nodes = index + 1;
      } else {
	_data.at(_last_of_free_nodes - 1).second = index + 1;
      }
      _last_of_free_nodes = _last_of_full_nodes;
      _last_of_full_nodes = previous_index_incremented;
    }
    //! get the number of bytes used by this object
    //! \return number of bytes used by this object
    unsigned memory_used() const {
      return _data.size() * (sizeof(unsigned) + sizeof(value_type));
    }
  private:
    /*!
      \brief get more nodes!
      @param count the number of new nodes to get
    
      Pointers in this list are indices of an array; resizing the array 
      does not break the pointers
    */
    void get_more_nodes(unsigned count);
    unsigned _head_of_free_nodes; //!< beginning of free node sublist
    unsigned _last_of_free_nodes; //!< end of free node sublist
    unsigned _head_of_full_nodes; //!< beginning of used node sublist
    unsigned _last_of_full_nodes; //!< end of used node sublist
    std::vector<data_pair> _data; //!< actual stored "list"
  };

  template <class value_type>
    void vectorlist<value_type>::add_element(value_type datum) {
    if (!_head_of_free_nodes) {
      //get more nodes
      get_more_nodes(_data.size());
    }

    //new elements add to the beginning of the lists, so the tails 
    //only need to be modified if the lists are empty
    if (!_head_of_full_nodes) {
      _last_of_full_nodes = _head_of_free_nodes;
    }
    if (!_data.at(_head_of_free_nodes - 1).second) {
      _last_of_free_nodes = 0;
    }
  
    //place the value in the first empty node, and add it to the
    //list of full nodes
    unsigned temp = _data.at(_head_of_free_nodes-1).second;
    _data.at(_head_of_free_nodes-1).first = datum;
    _data.at(_head_of_free_nodes-1).second = _head_of_full_nodes;
    _head_of_full_nodes = _head_of_free_nodes;
    _head_of_free_nodes = temp;
  }
  template <class value_type>
    value_type vectorlist<value_type>::pop_element() {
    if (_head_of_full_nodes) {

      //removed element operates at the beginning of the lists,
      //so the tails only need to be modified if the lists are empty
      if (!_data.at(_head_of_full_nodes - 1).second) {
	_last_of_full_nodes = 0;
      }
      if (!_head_of_free_nodes) {
	_last_of_free_nodes = _head_of_full_nodes;
      }

      //get the value in the first full node, and move the node to the 
      //list of free nodes
      unsigned temp = _data.at(_head_of_full_nodes-1).second;
      _data.at(_head_of_full_nodes-1).second = _head_of_free_nodes;
      _head_of_free_nodes = _head_of_full_nodes;
      _head_of_full_nodes = temp;
      return _data.at(_head_of_free_nodes-1).first;
    } else {
      throw std::domain_error("vectorlist::pop_element: cannot pop "
			      "element at head of list, as there are "
			      "no values in the list");
    }
  }
  template <class value_type>
    void vectorlist<value_type>::get_more_nodes(unsigned count) {
    unsigned original_count = _data.size();
    _data.resize(_data.size() + count);
    for (unsigned i = original_count; i < _data.size(); ++i) {
      //the empty list of nodes will have 0 as a code, which is a valid
      //pointer value
      if (i+1 == _head_of_free_nodes) 
	throw std::domain_error("vectorlist::get_more_nodes: introducing "
				"cycle for index=" + to_string<unsigned>(i+1));
      _data.at(i).second = _head_of_free_nodes;
      //point the input ptr to the new node
      _head_of_free_nodes = i+1;
    }
    //only the tail of the free nodes has been modified
    _last_of_free_nodes = _data.size();
  }
}
#endif //__VECTORLIST_H__
