# -*- shell-script -*-
# genshiken_main.sh -- Parses the WWW for video/image links, and then sends them to either: wget; a media player; or an image viewer
# Copyright © 2015-2016 Michael Pagan
#
# Author: Michael Pagan
# E-Mail: michael.pagan@member.fsf.org
# Jabber: pegzmasta@member.fsf.org
#
# This file is part of Genshiken.
#
# Genshiken is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Genshiken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Genshiken. If not, see http://www.gnu.org/licenses/.
#===================================================================

###
### ERROR HANDLING
###

# Let's end Genshiken in a graceful manner
trap 'Early_Termination' SIGHUP SIGINT SIGTERM

# If there are any bugs in this source code, let's find out where they are as they occur (NOTE: loops and conditionals are out of the question)
#-trap 'err=$?; echo -e "ERROR:$(basename $0):line ${LINENO}:command($(eval echo $BASH_COMMAND)):error($err)"' ERR

###
### DEPENDENCIES
###

# The name of this program is ...
#-PROGNAME=$(basename $0 | sed 's_.*_GNU\ \u&_')
readonly PROGNAME=Genshiken
readonly VERSION=0.5.2

# Dependency 1: GNU Bash 4+ (There are certain features that I use that only exist in bash 4 and up)
[[ $BASH_VERSINFO -lt 4 ]] && { echo "[${FUNCNAME[0]}] You need to be running version 4+ of GNU Bash in order to run this script.  Sorry!" >&2; exit 3; }

# Dependency 2: getopt from `util-linux' 2.25.2 or higher
getopt -T && [[ $? -ne 4 ]] &&
echo "[${FUNCNAME[0]}] You need getopt from util-linux 2.25.2 or higher in order for ${PROGNAME,?} to parse parameters properly." >&2 && exit 4
unset GETOPT_COMPATIBLE # Ensure that this version of `getopt' does not behave like its predecessors

# Do not run if there are too little/many arguments
[[ $# -eq  0 ]] && exit 12 # No need to say anything here-- you ask for nothing; you get nothing...
[[ $# -gt 16 ]] && { echo "[${FUNCNAME[0]}] $PROGNAME has been given too many arguments (i.e. $((${#}-16)) too many)." >&2; exit 12; }

# Temporarily allow errors, so that genshiken won't quit pre-maturely before users can view messages (out of the 20+ checks, 2 or 3 may ERR)
set +e && UNSET=0

# If bash is searching for unbound variables, then it will surely encounter them before the first read command below.           |
# Because the Here String has yet to take effect when bash encounters these labels, bash will report them as unbound.           |
# The variables $INFOPATH, $USER, and $PAGER may also not be set, due to the user/administrator not setting them up in advance. |
[[ -o nounset ]] && set +o nounset && UNSET=1

# GNU Bash should allow us to use glob-patterns of all types, but we won't allow it to overwrite our files.
[[ -o noglob      ]] && set +o noglob
[[ ! -o noclobber ]] && set -o noclobber # Let's NOT overwrite files ...
[[ `shopt | gawk '$1 ~ "extglob" { print $2 }'` != 'on' ]] && shopt -s extglob

# Dependency 3 & 4: The GNU Core Utilities & GNU Find Utilities (I expect we're using free software; if not, results may be unpredictable). |
# x-----------> This dependency is a given, but you never know: Someone may have made proprietary versions!  What errors will we find?      |
[[ `  cp   --version | gawk '{ if (NR<2) { print $2, $3 }; }' | sed -e 's_^.__' -e 's_.$__'` != 'GNU coreutils' ]] ||
[[ `  find  -version | grep 'GNU' | sed -n '1 p' | sed 's_.*(\(.*\)).*_\1_'`                 != 'GNU findutils' ]] ||
[[ `  sed  --version | sed -n '1 p' | sed 's_[^GNU]*__g'`                                    != 'GNU' ]] ||
[[ `  gawk --version | gawk '{ if (NR<2) { print $1 }; }'`                                   != 'GNU' ]] ||
[[ ! `grep --version | grep -m 1                                                                 GNU` ]] &&
{
  echo "[${FUNCNAME[0]}] The location of $(type grep 2> /dev/null; [[ $? -ne 0 ]] && echo grep:)"
  echo "[${FUNCNAME[0]}] The location of $(type gawk 2> /dev/null; [[ $? -ne 0 ]] && echo gawk:)"
  echo "[${FUNCNAME[0]}] The location of $(type find 2> /dev/null; [[ $? -ne 0 ]] && echo find:)"
  echo "[${FUNCNAME[0]}] The location of $(type sed  2> /dev/null; [[ $? -ne 0 ]] && echo sed:)"
  echo "[${FUNCNAME[0]}] The location of $(type cp   2> /dev/null; [[ $? -ne 0 ]] && echo cp:)"
  echo
  echo "[${FUNCNAME[0]}] Free Software like GNU Grep/Awk/Sed, GNU Core Utilities, and GNU Find Utilities are required in order to run this script."
  echo "[${FUNCNAME[0]}] Ensure that your versions are free.  Go to <www.gnu.org> for more information.  These links will point you in the right direction:"
  echo
  echo "+  http://www.gnu.org/distros/free-distros.html  ---------------  Install a fully free operating system today!"
  echo "+  http://www.gnu.org/philosophy/categories.html#GNUsoftware  --  What kinds of software exist on the WWW?  What's the difference between them?"
  echo "+  http://www.gnu.org/software/software.html#allgnupkgs  -------  A list of all GNU packages available.  Most are copyleft."
  echo "+  http://directory.fsf.org/wiki/GNU/  -------------------------  A Wiki for said packages ..."
  echo "+  http://directory.fsf.org/wiki/All  --------------------------  The Free Software Directory lists all the free software available on the net."
} >&2 && exit 5 || GNU=0

# Dependency 5: Wget (GNU's Network Downloader)
[[ -z `which wget` ]] &&
{ echo "[${FUNCNAME[0]}] You need to install GNU Wget, in order to download files with $PROGNAME." >&2; GNU=1; }

# Dependency 6: cURL (Required in order make header requests, and to send confirmation data to a server containing our desired files)
[[ -z `which curl` ]] &&
{ echo "[${FUNCNAME[0]}] You need to install curl, in order for $PROGNAME to communicate to servers." >&2; GNU=1; }

# Dependency 7: lynx (Required in order to extract the source code for websites, since wget/curl sometimes fail)
[[ -z `which lynx` ]] &&
{ echo "[${FUNCNAME[0]}] You need to install lynx, in order for $PROGNAME to extract a website's source code." >&2; GNU=1; }

# Dependency 8: A video player (mplayer[2] || totem || vlc || xine || your-choice)
[[ -z ${PLAYER:-} && `which mplayer` ]] && PLAYER='mplayer -fs' # If mplayer2 is installed, an alias will have already been made ...
[[ -z ${PLAYER:-} && `which mpv`     ]] && PLAYER='mpv --fs'    # Recommended ...
[[ -z ${PLAYER:-} && `which totem`   ]] && PLAYER='totem --fullscreen'
[[ -z ${PLAYER:-} && `which vlc`     ]] && PLAYER='cvlc --play-and-exit -f'
[[ -z ${PLAYER:-} && `which xine`    ]] && PLAYER='xine -f'
[[ -z ${PLAYER:-} ]] &&
{
  echo "[${FUNCNAME[0]}] In order to watch videos with $PROGNAME, you need to install one of the following video players:"
  echo -e "mplayer\ntotem (A.K.A.-- GNOME Videos)\nvlc\nxine\n[${FUNCNAME[0]}] Edit the source code if this list is not to your liking."; GNU=1
} >&2

# Dependency 9: A picture viewer (geeqie || ristretto || eog || your-choice)
[[ -z ${IMAGER:-} && `which geeqie`    ]] && IMAGER='geeqie -t'
[[ -z ${IMAGER:-} && `which ristretto` ]] && IMAGER=ristretto
[[ -z ${IMAGER:-} && `which eog`       ]] && IMAGER=eog
[[ -z ${IMAGER:-} ]] &&
{
  echo "[${FUNCNAME[0]}] In order to view pictures with $PROGNAME, you need to install one of the following image viewers:"
  echo -e "geeqie\nristretto\neog\n[${FUNCNAME[0]}] Edit the source code if this list is not to your liking.\n"; GNU=1
} >&2

# Optional: A video address extractor, youtube-dl (for unethical video links that can't be parsed by genshiken)
if [[ -n `which youtube-dl` ]]; then
  extract='youtube-dl -g'
  extractors='youtube-dl --list-extractors && echo veoh | tr [[:upper:]] [[:lower:]]'
else
  echo "[${FUNCNAME[0]}] F.Y.I.-- Not all links can be extracted with $PROGNAME alone.  Video links that require proprietary software is ignored by default."
  echo "[${FUNCNAME[0]}] If you install 'youtube-dl', more video links will become readable by '${PROGNAME,?}'-- allowing you to bypass the non-free restriction."
  [[ ! -z ${WAIT:-} && $WAIT -le 1 ]] && : || sleep 1
fi

# Optional: A notification program (notify-send || gxmessage)
if [[ -n `which notify-send` ]]; then
  MSG=notify-send
elif [[ -n `which gxmessage` ]]; then
  MSG="gxmessage --timeout 3 -fn serif -fn bold -fn 24 -borderless -center -fg white -bg black -ontop"
else
  echo "[${FUNCNAME[0]}] F.Y.I.-- If you would like \"$PROGNAME\" to notify you on which series and which episode you're on ..."
  echo "[${FUNCNAME[0]}] Try installing either \"notify-send\" or \"gxmessage\".  These packages are optional."
  [[ ! -z ${WAIT:-} && $WAIT -le 1 ]] && : || sleep 1
fi

# If any of the above dependencies are missing/insufficient: Exit.
[[ $GNU = 1 ]] && { echo -e "\n[${FUNCNAME[0]}] ${A}Please install the above packages that $PROGNAME depends on first; afterwords, try again." >&2; exit 6; }

###
### ENVIRONMENT SETUP
###

# Ensure the $USER variable is set
[ -z ${USER:-} ] && USER=`id -un`

# Ensure the user sets his $HOME Variable, if not already
HOME=${HOME:?"[${FUNCNAME[0]}] $PROGNAME can't operate without a HOME variable.  Export \"${USER:=UNKNOWN USER}\`s\" HOME variable to the environment."}

# Ensure we can use a pager when needed (a data stream can't be piped to 'mcview', so I excluded it)
if [[ -z ${PAGER:-} || $PAGER = 'mcview' ]]; then
   [[ -z `which less` ]] && PAGE=more || PAGE=less
else
  PAGE=$PAGER
fi
if [[ $TERM = 'dumb' ]]; then
  PAGE=tee
fi
# These constants are command strings that will be processed by the `eval` command later on.
readonly TEST_CONNECTION="ping -c 1 www.duckduckgo.com |& grep '.*'"
readonly EPISODE_LINKS="sed -e 's_.*href=\"\(.*\)\"[ ]title=\".*>\(.*\)</a></li>_<\1>\2_' -e 's:[ ]:_:g' -e 's/English[ ]Dubbed//'"
readonly BULLETIN="sed 's_.*[^:]\$_ + &_g'"
readonly GET_JUNK="ls -la | egrep -v '_$|^d|total|series_|\.*flv|\.*mp4|\.*avi|\.*ogg|\.*webm|\.*jpg|\.*jpeg' | gawk '{ print \$9 }'"
readonly GET_DOMAIN="sed -e 's_<\(.*\)>.*_\1_' -e 's_www\.__' -e 's_.*//\([a-z].*\)\..*_\1_'"
readonly HOST_LOCATE="sed -e 's_.*src=.*//\(.*\)\.\(.*\)\.\(.*\)_\1.\2.\3_' -e 's_/.*__'"
readonly HTML_PARSER="sed -e 's_<li>__g' -e 's_</li>__g' -e 's_<p.*>__' -e 's_</p>__' -e 's_<ul>__' -e 's_</ul>__' \
                          -e 's_<a.name=...></a>__' -e 's_sep.>.__' | grep '[[:alnum:]]'"
readonly HUMAN_READABLE="sed -e 's:/:_:g' -e 's_\.__g' -e 's:[()]::g' -e 's:__:_:g' -e 's_/__g' -e 's_amp;__g' -e 's_#038;__g' -e 's_&#215;_x_g' \
                             -e 's_&#8211;_-_g' -e 's_\&#8216;_'\''_g' -e 's_\&#8217;_'\''_g' -e 's_&#8220;_\"_g' -e 's_&#8221;_\"_g' -e 's_&#8230;__g'"
readonly HUMAN_READABLE_2="sed -e 's_#038;__g' -e 's_&#215;_x_g' -e 's_&#8211;_-_g' -e 's_\&#8216;_'\''_g' -e 's_\&#8217;_'\''_g' -e 's_&#8220;_\"_g' \
                               -e 's_&#8221;_\"_g' -e 's_&#8230;__g'"
readonly BROWSER_READABLE="sed -e 's_%2F_/_g' -e 's_%3A_:_' -e 's_//_/_2g' -e 's_[ ]_%20_g' -e 's_(_%28_g' -e 's_)_%29_g' \
                               -e 's/%5F/_/g' -e 's_\[_%5B_g' -e 's_\]_%5D_g' -e 's_&#038;_%26_g'"

#  Ensure that we use the users preferred temporary directory if they have one            |
#+ NOTE: (TMPDIR is a directory that will store our files; they're not temporary, though) |
export TMPDIR=${TMPDIR:-$HOME/.config}

# Set-up some basic variables, that I know I'll need in advance, with a Here String
read count PAUSE DEBUG START_AT first processing cover_page manga_count ADD_PAGE ADD_JPEG ADD_MANGA ATTEMPT \
     RAND BASE_DIR DIR_DET DOMAIN1 DOMAIN2 \
     <<< "0 0 0 0 1 1 1 1 page_count=$((page_count+1)) jpeg_count=$((jpeg_count+1)) manga_count=$((manga_count+1)) re_try=$((re_try+1)) \
          $RANDOM $HOME $TMPDIR/${PROGNAME,?}-$USER/details NOT-DEFINED-YET watchcartoononline"

# Lets define all possible long parameters that genshiken will accept
read LONG_OPTS <<< 'version,help,usage,source-comments,source-code,configure:,copyright,license,warranty,status-report,\
                    alert,no-update,no-pager,no-sessions,limited-bandwidth,impatient:,start-at:,base-dir:,rgb:,colorize:,commands,verbose::,get:,view:,stream:'

# NO LONGER PARANOID!  Now that the read commands and all dependency checks are finished, I'll turn back on any previously set debug options
[[ $UNSET = 1 ]] && set -o nounset && unset UNSET

# Let's source genshiken's functions
doc=/usr/local/share/doc/genshiken
functions=/usr/local/share/genshiken/functions/*
for func in $functions; do source $func; done

###
### PARAMETER EVALUATION
###

PARAMS=`getopt                      \
        -o VhRECLWIafpm:s:d:vx:i:y: \
        --long ${LONG_OPTS//[ ]/}   \
        -n ${PROGNAME,?} -- "$@"`

# Faulty parameters will not be accepted ...
[[ $? -ne 0 ]] && echo "Terminating..." >&2 && exit 13

# Evaluate our parameters
eval set -- "$PARAMS"

###
### VERBOSITY SETTINGS
###

if [ "$1" = '-v' -o "$1" = '--commands' ]; then
  shift
  [[ ! -o xtrace ]] &&
  bash -x $0 $@ |& tee $DIR_DET/../debug.log && exit 0

elif [[ "$1" = '--verbose' ]]; then
  shift
  [[ ! -o xtrace ]] &&
  {
    # Replace the command expansion prompt with a debugging prompt (trailing newline exists to separate the prompt from the command expansion)
    [[ -f $TMPDIR/sleep.log ]] && rm $TMPDIR/sleep.log
    PAUSE="$1"; [[ -z ${PAUSE:-} ]] && echo 0 > $TMPDIR/sleep.log || echo $PAUSE > $TMPDIR/sleep.log # This hack will allow us to delay the prompt!
    PS4='($(basename ${BASH_SOURCE} && sleep $(< $TMPDIR/sleep.log))'
    export PS4=${PS4}':${LINENO}): ${FUNCNAME[0]} - [Lvl-${SHLVL}, Sub-${BASH_SUBSHELL}, Err-$?]
';  bash -x $0 $@ |& tee $DIR_DET/../debug.log && exit 0
  }
fi

###
### ACTIVATE COMMAND-LINE OPTIONS
###

while (($#))
do case "$1" in
    -V|--version) # What version is Genshiken on anyway?
      echo $PROGNAME $VERSION
      ${PROGNAME,?} --copyright | sed -n '1 p'
      echo -e 'License GPLv3+:\t\tGNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>'
      echo -e 'This is free software:\tyou are free to change and redistribute it.'
      echo -e 'There is NO WARRANTY, to the extent permitted by law.\nThanks to the following:'
      cat $doc/AUTHORS | gawk -F: '{ print $2,"\t",$3 }'
      exit
    ;;
    -h|--help|--usage) # Provide a set of known parameters that genshiken understands to the user
      echo "Usage: ${PROGNAME,?} [-V|-h|-M|-R|-C|-L|-W|-I]"
      echo "Usage: ${PROGNAME,?} [DEBUG] Session=Anime-Type [Options]..."
      echo
      echo -e '3 types of Sessions:\n\t-x Anime-Type;    --get=Anime-Type\n\t-y Anime-Type; --stream=Anime-Type\n\t-i Anime_Type;   --view=Anime-Type\n'
      echo '5 different Anime-Types:'
      echo -e '\tanime-sub\tEnglish Subtitles\n\tanime-dub\tEnglish Dubbed\n\tmovie\t\tDubbed Movies\n\tova\t\tSpecial Episodes\n\tcartoon\t\tCartoon Shows\n'
      echo 'Breaking Options:'
      echo -e "\t-V, --version\t\t\tPrint ${PROGNAME}'s version number and legal information to standard output"
      echo -e "\t-h, --help, --usage\t\tThis help-- a basic usage summary."
      echo -e "\t-R, --source-comments\t\tPrint ${PROGNAME}'s source code comments to your pager"
      echo -e "\t-E, --source-code\t\tPrint ${PROGNAME}'s source code to your pager"
      echo -e "\t--configure=long-opt\t\tMake a supplementary 'long-opt' a default parameter; remove with 'remove-long-opt'; remove config file with 'remove'."
      echo -e "\t-C, --copyright\t\t\tPrint ${PROGNAME}'s copyright information to standard output"
      echo -e "\t-L, --license\t\t\tDump and pipe the contents of the FSF's copy of the GPL to your pager (internet connection required)"
      echo -e "\t-W, --warranty\t\t\tPrint out a short notice on the \"warranty\" to standard output"
      echo -e "\t-I, --status-report\t\tPrint out a brief summary of how much space is being taken up by ${PROGNAME}'s Download directories\n"
      echo 'Supplementary Options:'
      echo -e "\t-a, --alert\t\t\tAlert (BEL) on errors and download completions (requires PC Speaker Module to be loaded into your kernel)"
      echo -e "\t-m level, --impatient=level\tA \"level\" of 2 ignores sleep(1) before a pager; of 1, prevents sleep(1); of 0, removes cache from streaming"
      echo -e "\t-f, --no-update\t\t\tSkips the Extract_Series_List procedure that creates an Anime-List, and proceeds to using your last downloaded copy"
      echo -e "\t--no-sessions\t\t\tPrevents ${PROGNAME} from creating/restoring previous session files (check the man page for more information)"
      echo -e "\t-p, --no-pager\t\t\tSends output that would normally go to a pager, to standard output instead"
      echo -e "\t--limited-bandwidth\t\tRestarts network connections whenever there are network time-outs, through the use of wget and mplayer"
      echo -e "\t--rgb=type, --colorize=type\tAdds color during a ${PROGNAME,?} session.  A value of \"dark\" provides subtle colors; of \"light\", bold colors"
      echo -e "\t-v, --commands\t\t\tOutput ${PROGNAME}'s command expansions to standard output, and to \"$TMPDIR/${PROGNAME,?}-$USER/debug.log\""
      echo -e "\t--verbose=[seconds]\t\tSimilar to \"--commands\", except a debug prompt is added and an optional value delays the prompt by \"seconds\""
      echo -e "\t-s #, --start-at=number\t\tStart at the video or image \"number\" of your choice.  A value of 'latest' will calculate the last episode number"
      echo -e "\t-d dir, --base-dir=dir\t\tSetup your ${PROGNAME} Downloads directory in \"dir\", instead of your HOME"
      echo
      echo "Submit bugs at: <https://savannah.nongnu.org/bugs/?func=additem&group=${PROGNAME,?}>"
      echo "Public mailing-list for user help and discussion: ${PROGNAME,?}-users@nongnu.org"
      echo "Genshiken Homepage: <http://www.nongnu.org/${PROGNAME,?}/>"
      #-echo "General help using GNU software: <http://www.gnu.org/gethelp/>"
      exit 0
    ;;
    -R|--source-comments) # Strip a majority of genshiken's commands inside the source code, in order to view the documentation within (for the Jedi)
      sed -e '/^#!.*/ d' -e '/^# -*-.*/ d' -e '/^# Copyright.*/,/#=.*/ d' $(which ${PROGNAME,?}) $(which $0) $functions | grep '#[ +-]' | $PAGE; exit 0
    ;;
    -E|--source-code) # This is free software ('free' as in freedom).  If you like what you see, then please contribute!
      sed -e '/^#!.*/ d' -e '/^# -*-.*/ d' -e '/^# Copyright.*/,/#=.*/ d' $(which ${PROGNAME,?}) $(which $0) $functions | $PAGE; exit 0
    ;;
    --configure) # Create a configuration file that will store default arguments for genshiken to run

      # Ensure we have a configuration directory-- just in case
      if [[ ! -d $HOME/.config/${PROGNAME,?}-$USER ]]
      then mkdir -p $HOME/.config/${PROGNAME,?}-$USER
      fi

      # Define config file location
      config=$HOME/.config/${PROGNAME,?}-$USER/config

      # Deal with specific arguments accordingly
      case $2 in

        alert)

          # Add new value or replace old value
          if [[ -f $config ]]; then
            if [[ -n $(grep $2 < $config | sed 's/--\(.*\)/\1/') ]]
            then { echo "[${FUNCNAME[0]}] This option has already been added to config file!" >&2; exit 1; }
            else echo "--${2}" >> $config
            fi
          else
            echo "--${2}" > $config
          fi
          echo "[${FUNCNAME[0]}] Alert switch on (ensure you're PC Module is loaded in the kernel if you can't hear it) ..."
        ;;
        base-dir)
          echo -n "[${FUNCNAME[0]}] Enter the base-directory: "; read directory

          # An error code of 14 should be expected, since no main session parameter is provided ...
          (${PROGNAME,?} --base-dir="$directory" &> /dev/null)

          # If the argument is valid, add the base-dir to the config file
          if [[ $? -ne 8 ]]; then

            # Add new value or replace old value
            if [[ -f $config ]]; then
              if [[ -n $(grep $2 < $config | sed 's/--\(.*\)=.*/\1/') ]]
              then sed -i "s/\(--${2}\)=.*/\1=$directory/" $config
              else echo "--${2}=$directory" >> $config
              fi
            else
              echo "--${2}=$directory" > $config
            fi
            echo "[${FUNCNAME[0]}] Base Directory, <$directory>, added to config file"
          else
            echo "[${FUNCNAME[0]}] ERROR.  Not able to add Base Directory, <$directory>.  You do not have the proper permissions." >&2
            exit 1
          fi
          unset directory
        ;;
        colorize|rgb)
          echo -n "[${FUNCNAME[0]}] \`dark' or \`light'? "; read color

          # If the argument is valid, add the color option to the config file
          if [[ $color = 'dark' || $color = 'light' ]]; then

            # Add new value or replace old value
           if [[ -f $config ]]; then
              if [[ -n $(grep $2 < $config | sed 's/--\(.*\)=.*/\1/') ]]
              then sed -i "s/\(--${2}\)=.*/\1=$color/" $config
              else echo "--${2}=$color" >> $config
              fi
            else
              echo "--${2}=$color" > $config
            fi
            echo "[${FUNCNAME[0]}] A color of \"$color\" has been added to the config file"
          else
            echo "[${FUNCNAME[0]}] ERROR.  \`$color\' is not a valid argument." >&2
            exit 1
          fi
          unset color
       ;;
        impatient)
          echo -n "[${FUNCNAME[0]}] Choose an impatient level from of 0, 1, or 2: "; read level

          # An error code of 14 should be expected, since no main session parameter is provided ...
          (${PROGNAME,?} --impatient=$level &> /dev/null)

          # If the argument is valid, add the color option to the config file
          if [[ $? -ne 1 ]]; then

            # Add new value or replace old value
            if [[ -f $config ]]; then
              if [[ -n $(grep $2 < $config | sed 's/--\(.*\)=.*/\1/') ]]
              then sed -i "s/\(--${2}\)=.*/\1=$level/" $config
              else echo "--${2}=$level" >> $config
              fi
            else
              echo "--${2}=$level" > $config
            fi
            echo "[${FUNCNAME[0]}] An impatience level of $level has been added to the config file."
          else
            echo "[${FUNCNAME[0]}] ERROR.  You must choose a number from 0-2." >&2
            exit 1
          fi
          unset level
        ;;
        limited-bandwidth)

          # An error code of 14 should be expected, since no main session parameter is provided ...
          (${PROGNAME,?} --limited-bandwidth &> /dev/null)

          # If the argument is valid, add the color option to the config file
          if [[ $? -ne 1 ]]; then

            # Add new value or replace old value
            if [[ -f $config ]]; then
              if [[ -n $(grep $2 < $config | sed 's/--\(.*\)/\1/') ]]
              then { echo "[${FUNCNAME[0]}] This option has already been added to config file."; exit 1; }
              else echo "--${2}" >> $config
              fi
            else
              echo "--${2}" > $config
            fi
            echo "[${FUNCNAME[0]}] The limited-bandwidth option has been added to the config file."
          else
            echo "[${FUNCNAME[0]}] ERROR.  You need MPlayer or its derivitives in order to add this option." >&2
            exit 1
          fi
        ;;
        no-pager)

          # Add new value or replace old value
          if [[ -f $config ]]; then
            if [[ -n $(grep $2 < $config | sed 's/--\(.*\)/\1/') ]]
            then { echo "[${FUNCNAME[0]}] This option has already been added to config file!" >&2; exit 1; }
            else echo "--${2}" >> $config
            fi
          else
            echo "--${2}" > $config
          fi
          echo "[${FUNCNAME[0]}] The no-pager option has been added to the config file."
        ;;
        no-sessions)

          # Add new value or replace old value
          if [[ -f $config ]]; then
            if [[ -n $(grep $2 < $config | sed 's/--\(.*\)/\1/') ]]
            then { echo "[${FUNCNAME[0]}] This option has already been added to config file!" >&2; exit 1; }
            else echo "--${2}" >> $config
            fi
          else
            echo "--${2}" > $config
          fi
          echo "[${FUNCNAME[0]}] The no-sessions option has been added to the config file."
        ;;
        no-update)

          # Add new value or replace old value
          if [[ -f $config ]]; then
            if [[ -n $(grep "$2" < $config | sed 's/--\(.*\)/\1/') ]]
            then { echo "[${FUNCNAME[0]}] This option has already been added to config file!" >&2; exit 1; }
            else echo "--${2}" >> $config
            fi
          else
            echo "--${2}" > $config
          fi
          echo "[${FUNCNAME[0]}] The no-update option has been added to the config file."
        ;;
        # For removing lines containing a key word (e.g. "remove-alert" will remove the alert line in the config file)
        remove-*)

          # Ensure the file and the option exists before we attempt to remove it
          [[ ! -f $config ]] && { echo "[${FUNCNAME[0]}] You do not have a configuration file for ${PROGNAME,?}!" >&2; exit 1; }
          [[ -z `grep $(gawk -F- '{ print $2 }' <<< $2) $config` ]] && { echo "[${FUNCNAME[0]}] This Option doesn't exist in the config file!" >&2; exit 1; }
          case $(gawk -F- '{ print $2 }' <<< $2) in
            alert)             grep -v 'alert'             $config | sort -o $config; echo "[${FUNCNAME[0]}] \"Alert\" option removed!";;
            base-dir)          grep -v 'base-dir'          $config | sort -o $config; echo "[${FUNCNAME[0]}] \"Base Directory\" option removed!";;
            colorize|rgb)     egrep -v 'colorize|rgb'      $config | sort -o $config; echo "[${FUNCNAME[0]}] \"Colorize\" option removed!";;
            impatient)         grep -v 'impatient'         $config | sort -o $config; echo "[${FUNCNAME[0]}] \"Impatient\" option removed!";;
            limited-bandwidth) grep -v 'limited-bandwidth' $config | sort -o $config; echo "[${FUNCNAME[0]}] \"Limited Bandwidth\" option removed!";;
            no-pager)          grep -v 'no-pager'          $config | sort -o $config; echo "[${FUNCNAME[0]}] \"No Pager\" option removed!";;
            no-sessions)       grep -v 'no-sessions'       $config | sort -o $config; echo "[${FUNCNAME[0]}] \"No Sessions\" option removed!";;
            no-update)         grep -v 'no-update'         $config | sort -o $config; echo "[${FUNCNAME[0]}] \"No Update\" option removed!";;
            start-at)          grep -v 'start-at'          $config | sort -o $config; echo "[${FUNCNAME[0]}] \"Start At\" option removed!";;
            *) echo "Internal error!"; exit 1;;
          esac
        ;;
        # For removing the configuration file
        remove)
          [[ -f $config ]] && rm $config || { echo "[${FUNCNAME[0]}] You do not have a configuration file for ${PROGNAME,?}!" >&2; exit 1; }
          echo "[${FUNCNAME[0]}] Configuration file removed ..."
        ;;
        start-at)
          echo -n "[${FUNCNAME[0]}] Enter the number you wish to start at: "; read number

          # An error code of 14 should be expected, since no main session parameter is provided ...
          (${PROGNAME,?} --start-at="$number" &> /dev/null)

          # If the argument is valid, add the base-dir to the config file
          if [[ $? -ne 1 ]]; then

            # Add new value or replace old value
            if [[ -f $config ]]; then
              if [[ -n $(grep $2 < $config | sed 's/--\(.*\)=.*/\1/') ]]
              then sed -i "s/\(--${2}\)=.*/\1=$directory/" $config
              else echo "--${2}=$directory" >> $config
              fi
            else
              echo "--${2}=$directory" > $config
            fi
            echo "[${FUNCNAME[0]}] A start value of \"$number\" will be added to the config file"
          else
            echo "[${FUNCNAME[0]}] ERROR.  You must choose either the string 'latest', or a number greater than 0." >&2
            exit 1
          fi
          unset number
        ;;
      esac
      exit 0
    ;;
    -C|--copyright) # The unsung hero, that allows programmers and hackers alike to freely create/share software, is... the GNU GPL!
      sed -n -e '3,4 p' -e '12,23 p' < $0 | gawk -F'#[ ]' '{ print $2 }' && exit 0
    ;;
    -L|--license) # View the current GNU GPL in it's entirety (requires an internet connection)
      connect=$(eval $TEST_CONNECTION)
      connect=${connect:6}

      # If the connection is down, then find a local copy of the license
      [[ -z ${connect##unknown*} ]] &&
      { less $(echo $(dirname $(dirname $(which ${PROGNAME,?})))/share/doc/${PROGNAME,?}/COPYING); } ||
      { lynx -dump 'http://www.gnu.org/licenses/gpl.txt' | $PAGE && exit 0; }
    ;;
    -W|--warranty) # This software comes "as is"
      ${PROGNAME,?} --copyright | sed -n '1 p'
      ${PROGNAME,?} --version   | sed -n '3,5 p'
      exit 0
    ;;
    -I|--status-report) # Provide some basic info and statistics, like: Important directories, # of anime, # of manga, etc.

      # Analyze genshiken and determine the make-up of the script
      CMT=$(gawk 'BEGIN { CMT='`$0 -R | wc -l`'; ALL='`$0 -E | wc -l`'; print CMT/ALL*100 }' | sed 's_\..*__') CMD=$[100 - $CMT]
      SIZE_WRAP=$(du -bL $(which ${PROGNAME,?})      | gawk '{ print $1 }' | sed -e 's_\..__' -e 's_\([0-9]*\)\(.*\)_\1_')
      SIZE_MAIN=$(du -bL $0                      | gawk '{ print $1 }' | sed -e 's_\..__' -e 's_\([0-9]*\)\(.*\)_\1_')
      SIZE_FUNC=$(for int in $(du -bL $functions | gawk '{ print $1 }' | sed -e 's_\..__' -e 's_\([0-9]*\)\(.*\)_\1_')
		  do total=${total:-0}; total=$(expr $total + $int); done; echo $total)
      SIZE=$(expr $SIZE_WRAP '+' $SIZE_MAIN '+' $SIZE_FUNC)
      SIZE=$(gawk 'BEGIN { print '$SIZE' / 1024 }' | gawk '{ printf "%.0f\n", $1 }' | sed 's_.*_& Kib_')
      gone='Does Not Exist, yet!' GET_LINES="cat $(which ${PROGNAME,?}) $0 $functions | wc -l"
      PART=$(lsblk -lo name,mountpoint | grep '/home' | gawk '{ print $1 }') # The user's home partition

      # Determine how much free space is available on the active partition
      [[ -z ${PART:-} ]] &&
      {
        PART=$(lsblk -lo name,mountpoint  | grep '/$' | gawk '{ print $1 }') # The user's root partition
        FREE_SPACE=$(df -h | grep '/$'    | gawk '{ print $4 }' | sed 's_\(.*\)\(.$\)_\1 \2ib_')
        FREE_VOLUME=$(gawk 'BEGIN { FREE='$(df | grep '/$'      | gawk "{ print \$4 }")';
                                    USED='$(df | grep '/home'   | gawk "{ print \$3 }")'; print FREE/(USED+FREE)*100
                                 }' | sed 's_\(.*\...\).*_\1 %_')
      } ||
      {
        FREE_SPACE=$(df -h | grep '/home' | gawk '{ print $4 }' | sed 's_\(.*\)\(.$\)_\1 \2ib_')
        FREE_VOLUME=$(gawk 'BEGIN { FREE='$(df | grep '/home'   | gawk "{ print \$4 }")';
                                    USED='$(df | grep '/home'   | gawk "{ print \$3 }")'; print FREE/(USED+FREE)*100
                                 }' | sed 's_\(.*\...\).*_\1_')
      }
      # Evaluate the percentage of anime contained within the user's active partition
      [[ -d "$HOME/Videos/${PROGNAME}_Downloads" ]] &&
      {
        DIR_ANIM="$HOME/Videos/${PROGNAME}_Downloads"
        ANIME_VOL=$(gawk "BEGIN { print $(du -c $DIR_ANIM |\
                    grep total | sed -e 's_\(\<.*\>\).*_\1_' -e 's_.$__') / $(lsblk -b | grep /home |\
                    gawk '{ print $4 }' | sed 's_.$__') * 100000 }" | sed "s_\(.*\)\.\(..\).*_\1\.\2_")
        DIR_ANIM="$HOME/Videos/${PROGNAME}_Downloads\t\t(${USER^?}\`s anime directory)"
      } || DIR_ANIM="$gone \t(${USER^?}\`s anime directory)"

      # Evaluate the percentage of manga contained within the user's active partition
      [[ -d "$HOME/Pictures/${PROGNAME}_Downloads" ]] &&
      {
        DIR_PICS="$HOME/Pictures/${PROGNAME}_Downloads"
        MANGA_VOL=$(gawk "BEGIN { print $(du -c $DIR_PICS |\
                     grep total | sed -e 's_\(\<.*\>\).*_\1_' -e 's_.$__') / $(lsblk -b | grep /home |\
                     gawk '{ print $4 }' | sed 's_.$__') * 100000 }" | sed "s_\(.*\)\.\(..\).*_\1\.\2_")
        DIR_PICS="$HOME/Pictures/${PROGNAME}_Downloads\t(${USER^?}\`s manga directory)"
      } || DIR_PICS="$gone \t(${USER^?}\`s manga directory)"

      # Ensure all percentages are 4 characters long (i.e. '0.06' gets converted to -> '00.06'; thus, uniformity)
      [[ $MANGA_VOL   < 1 ]] && MANGA_VOL=$(  sed 's_.*_0&_' <<< $MANGA_VOL)
      [[ $ANIME_VOL   < 1 ]] && ANIME_VOL=$(  sed 's_.*_0&_' <<< $ANIME_VOL)
      [[ $FREE_VOLUME < 1 ]] && FREE_VOLUME=$(sed 's_.*_0&_' <<< $FREE_VOLUME)
      [[ $MANGA_VOL  != 0 ]] && MANGA_VOL="($MANGA_VOL % of /dev/$PART)" || MANGA_VOL=
      [[ $ANIME_VOL  != 0 ]] && ANIME_VOL="($ANIME_VOL % of /dev/$PART)" || ANIME_VOL=

      # Report our findings...
      echo -e "# Line-Count:\t `eval $GET_LINES`\n# Comments %:\t $CMT %\n# Commands %:\t $CMD %\n# Script-Size:\t $SIZE\n# Current Dir:   $PWD"
      echo -e "# Pictures Dir:  $DIR_PICS\n# Videos Dir:    $DIR_ANIM"
      DIR_ANIM="$HOME/Videos/${PROGNAME}_Downloads" DIR_PICS="$HOME/Pictures/${PROGNAME}_Downloads"
      echo -e "# Manga Total:   `[[ -d $DIR_PICS/ ]] && expr $(ls $DIR_PICS/* | sed "s\.*[^:]$\ + &\g" | grep + | wc -l) '-' 1   || echo 0`"
      echo -e "# Manga Size:    `[[ -d $DIR_PICS/ ]] && du -ch $DIR_PICS | grep total | gawk '{ print $1 }' | sed 's_\(.*\)\(.$\)_\1 \2ib_' || :` $MANGA_VOL"
      echo -e "# Anime Total:   `[[ -d $DIR_ANIM/ ]] && expr $(ls $DIR_ANIM/*/? | sed "s\.*[^:]$\ + &\g" | grep + | wc -l) '-' 1 || echo 0`"
      echo -e "# Anime Size:    `[[ -d $DIR_ANIM/ ]] && du -ch $DIR_ANIM | grep total | gawk '{ print $1 }' | sed 's_\(.*\)\(.$\)_\1 \2ib_' || :` $ANIME_VOL"
      echo -e "# Free Space:    $FREE_SPACE (${FREE_VOLUME} % of /dev/$PART)"
      exit 0
    ;;

    # The rest of this `for` loop represents the supplementary options that are not stand-alone |
    # x---------------------------------------------------------------------------------------- |
    -a|--alert)     A=\\a;         shift;;
    -f|--no-update) SKIP=1;        shift;;
    -p|--no-pager)  PAGE=tee;      shift;;
    --no-sessions)  NO_SESSIONS=1; shift;;
    --limited-bandwidth) # Use this option if the network starts timing out on you...
      SLOW=1
      [[ -n `which mplayer` ]] && PLAYER='mplayer -fs' && shift ||
      [[ -n `which mpv`     ]] && PLAYER='mpv --fs'    && shift ||
      { echo -e "[${FUNCNAME[0]}] You need either MPlayer or its derivitives in order to use this Option!" >&2; exit 1; }
      ;;
    -m|--impatient) # How impatient are we?
      WAIT="$2"
      [[ "$2" -ge 0 && "$2" -le 2 && ! -z `grep '[[:digit:]]' <<< "$2"` ]] && shift 2 ||
      { echo "[${FUNCNAME[0]}] Impatience level must be a digit within the range of 0-2" >&2; exit 1; }
      ;;
    -s|--start-at) # Which file number should we <<start-on>>?
      START_AT="$2"
      [[ "$2" -gt 0 && ! -z `grep '[[:digit:]]' <<< "$2"` || ! -z `grep 'latest' <<< "$2"` ]] && shift 2 ||
      { echo "[${FUNCNAME[0]}] Start-at must be a digit greater than 0, or the string 'latest'" >&2; exit 1; }
      ;;
    -d|--base-dir) # What is our base directory?

      # Lets ensure that the user has permission to use this directory; if not, report the error to the user
      BASE_DIR="$2"; mkdir -p $BASE_DIR
      [[ $? -eq 0 ]] ||
      { echo -e "[${FUNCNAME[0]}] You don't have the proper permissions to use \"$BASE_DIR\", as your Base Directory."; exit 8; }
      shift 2
    ;;
    --rgb|--colorize) # I love syntax highlighting!  Let's add some color, and rid ourselves of boring prompts for good.                            |
                      # NOTE: I added some backspace sequences '\b' in order to prevent the `echo` command from printing extra characters when bold |

      # Let's ensure our Here-Doc gets it right, too
      BRIGHT="$2"
      [[ -f $DIR_DET/../.rgb && `< $DIR_DET/../.rgb` != $BRIGHT-$$ ]] && sed -i s_.*_"$BRIGHT-$$"_ $DIR_DET/../.rgb
      [[ ! -f $DIR_DET/../.rgb ]] && echo $BRIGHT > $DIR_DET/../.rgb

      # Let's determine the brightness level
      case $BRIGHT in

        dark) # The user wants the colors to be dark and subtle
          R='\e[31m'                 # Red Ascii Prefix ----------- Function Name
          G='\e[32m'                 # Green Asciii Prefix -------- Regular String
          Y='\e[33m'                 # Yellow Ascii Prefix -------- Comments, External Output, User Input
          B='\e[34m'                 # Blue Ascii Prefix ---------- $PROGNAME/Networks
          M='\e[35m'                 # Magenta Ascii Prefix ------- Anything that's variable
          C='\e[36m'                 # Cyan Ascii Prefix ---------- Files/Paths/URL's
          N='\e[m'                   # Ascii Color Terminate ------ Begins & Ends in string output to prevent debugger color interference
        ;;
        light) # The user wants the colors to be bold and bright
          R='\e[31;1m'               # Red Ascii Prefix ----------- Function Name
          G='\e[32;1m\]\b\b'         # Green Asciii Prefix -------- Regular String
          Y='\e[33;1m'               # Yellow Ascii Prefix -------- Comments, External Output, User Input
          B='\e[34;1m\]\b\b'         # Blue Ascii Prefix ---------- $PROGNAME/Networks
          M='\e[35;1m\]\b\b'         # Magenta Ascii Prefix ------- Anything that's variable
          C='\e[36;1m\]\b\b'         # Cyan Ascii Prefix ---------- Files/Paths/URL's
          N='\b[\b\e[0m \b]\b  \b\b' # Ascii Color Terminate ------ Begins & Ends in string output to prevent debugger color interference
        ;;
        *) : # The user doesn't know what they want ...
        ;;
      esac
      shift 2
    ;;
    # These main options won't be executed until we've entered the Main Options `for` loop.
    -x|--get)
      [[ ! -d $DIR_DET/  ]] && mkdir -p $DIR_DET/
      [[ -d $DIR_DET/../stream ]] && rm -r $DIR_DET/../stream
      [[ -f $HOME/Videos/${PROGNAME}_Downloads/missing_files.log ]] &&
      { rm  $HOME/Videos/${PROGNAME}_Downloads/missing_files.log; }
      MAIN='--get'; Anime_Type="$2"; shift 2
    ;;
    -i|--view)
      [[ -d $DIR_DET/../stream ]] && rm -r $DIR_DET/../stream; MAIN='--view'; Anime_Type="$2"; shift 2
    ;;
    -y|--stream)
      [[ ! -d $DIR_DET/ ]] && mkdir -p $DIR_DET
      [[ -d $DIR_DET/../stream ]] && rm -rf $DIR_DET/../stream; mkdir $DIR_DET/../stream
      stream_into="into $(info $(gawk '{ print $1 }' <<< $PLAYER) | sed -n '4 p' | gawk -F'(' '{ print $1 }')"
      [[ `sed 's_\<.*\>[ ]\(.*\)_\1_' <<< $stream_into | tr [:upper:] [:lower:]` != ${PLAYER//[ ]-?*/} ]] && stream_into=
      Streaming=Streaming DIR_ANIM=$TMPDIR/${PROGNAME,?}-$USER/stream
      MAIN='--stream'; Anime_Type="$2"; shift 2
    ;;
    --) shift; break;;
    *) echo "Internal error!"; exit 1;;
  esac
done

# DO NOT DETERMINE THAT WE'RE STREAMING, SIMPLY BASED ON THE EXISTENCE OF THE STREAM DIRECTORY!          |
#                                                                                                        |
# This is my way of informing other procedures if we're streaming or not.                                |
# (This should work when running multiple instances of genshiken, simultaneously with different options) |
#                                                                                                        |
# Old Method: [[ -d $DIR_DET/../stream ]] && Execute_Procedures_Relating_to_Streams                      |
# New Method: [[ ! -z $STREAM ]] && Execute_Procedures_Relating_to_Streams ==============================/
STREAM=$(ps x | grep $$ | egrep -e '-[y]|--[s]tream' || echo)
STREAM=${STREAM:-$([[ -f $DIR_DET/../.${PROGNAME,?}-here-doc.sh ]] && egrep -e '-[y]|--[s]tream' $DIR_DET/../.${PROGNAME,?}-here-doc.sh || echo)}
SEENIT=$(ps x | grep $$ | egrep -e '-[o]|--[v]iew'   || echo)
SEENIT=${SEENIT:-$([[ -f $DIR_DET/../.${PROGNAME,?}-here-doc.sh ]] && egrep -e '-[o]|--[v]iew'   $DIR_DET/../.${PROGNAME,?}-here-doc.sh || echo)}

###
### LEGAL/WARRANTY INFORMATION
###

# Sanity check-- if no session parameter has been issued, then exit this script
if [[ ! -z ${MAIN:-} ]]; then

  #  If we are planning on downloading/streaming videos, then test the internet connection by |
  #+ sending packets to a website, and evaluating the host response.                          |
  if [[ -z ${SEENIT:-} ]]; then
    connection=$(eval $TEST_CONNECTION)
    connection=${connection:6}

    # Evaluate packets received; if there is no connection: Then exit now
    if [[ -z ${connection##unknown*} ]]; then
      echo -e "${R}[${FUNCNAME[0]}] ${G}Sorry!  ${B}$PROGNAME ${G} can't continue, unless you're connected to the Internet. ${N}  \b" >&2
      echo -e "${R}[${FUNCNAME[0]}] ${G}Take a look at some of ${B}${PROGNAME,?}${G}'s functions for automating this process. ${N}  \b"   >&2
      exit 10
    fi
  fi
  # Let's remind the user that Genshiken comes with NO WARRANTY
  echo -e "$(sed -n "s_.*_${R:+$(echo \\$R)}# ${Y:+$(echo \\$Y)}&_p" <<< "$(${PROGNAME,?} --warranty)")"
  echo -e "${R}# ${N} \b"
  # Let the user know what they're getting into (for those "Saki Kasukabe" types)            |
  # NOTE: The description is variable, because Genshiken may do more later on in the future. |
  whatis ${PROGNAME,?} &> /dev/null
  [[ $? -ne 0 ]] && describe=$(man ${PROGNAME,?} | sed -n '6 s/[ ]*//p') || describe=$(whatis ${PROGNAME,?})
  echo -e "${R}# ${B}$PROGNAME ${Y} -$(gawk -F- '{ print $2 }' <<< $describe)${N}"
  echo -e "${R}# ${N} \b"
  # Let's be responsible: If we find evidence of copyright infringement, then report it. |
  # My simple function can't go that far though, so a simple reminder will do.           |
  if [[ -z ${SEENIT:-} ]]; then
    echo -e "${R}# ${Y}Before we continue-- Please take note of the following: ${N}"
    echo -e "${R}# ${N} \b"
    echo -e "${R}# ${Y}All strings tagged with \"${R}<${C}URL${R}>${Y}\" (enter '--rgb' mode if not visible) are website addresses."
    echo -e "${R}# ${Y}Contained inside of these webpages are links with videos, embedded from third-party sites."
    echo -e "${R}# ${Y}All videos streamed/downloaded from ${R}<${C}www.watchcartoononline.io${R}> ${Y}come from third-party sites."
    echo -e "${R}# ${B}$PROGNAME ${Y}will inform you where all videos are comming from, as you download/stream them."
    echo -e "${R}# ${B}$PROGNAME ${Y}is not responsible for any content that you may encounter via ${R}<${C}www.watchcartoononline.io${R}>${Y}. \b"
    echo -en "${R}# ${Y}Press ${M}[ENTER]${Y} to read the following Legal Disclaimer and DMCA from the \"${B}Watchcartoononline.COM Network${Y}\":  \b\b"
    read; echo -e "${R}# ${Y}  \b\b"; Legal_Disclaimer | $PAGE
    echo -en "${R}# ${Y}If you've read and comprehended the Legal Disclaimer and DMCA, then enter ${M}yes ${Y}to continue:${M}   \b\b"; read
    case $(sed y/yes/YES/ <<< $REPLY) in
      YES) : ;;
      *) echo -e "${R}[${FUNCNAME[0]}] ${G}Sorry!  ${B}$PROGNAME ${G}can't continue, until you've acknowledged the above Disclaimer." >&2
         exit 2;;
    esac
  fi
else
  echo -e "${R}[${FUNCNAME[0]}] No session parameter has been provided for this script!  Exitting now ..." >&2 && exit 14
fi
# Remind the user which arguments are accepted when the time comes
args="${M}anime-dub${G}|${M}anime-sub${G}|${M}cartoon${G}|${M}movie${G}|${M}ova ${N}"

# The download path is variable, and can be changed by the user at run-time (option: '--BASE-DIR').
[[ -z ${STREAM:-} ]] && DIR_ANIM="$BASE_DIR/Videos/${PROGNAME}_Downloads"
                        DIR_PICS="$BASE_DIR/Pictures/${PROGNAME}_Downloads"

###
###  THE MAIN PROCEDURE
###
case $MAIN in
  --stream) # Stream your Anime of choice directly off of a website via your media player, instead of the browser's (saves disk space, too)
    case $Anime_Type in
      anime-[ds]ub|cartoon|movie|ova) # Identical to the '--get' Option.  The difference lies on what we did before we entered this loop.

        # Determine if the user has an HMDI monitor connected, and if he wants to continue genshiken on the other monitor
        HDMI_Setup

        #  Before we select our anime of choice: Let's find out if the user wants to restore any             |
        #+ existing sessions of Genshiken, that was previously downloading/streaming/viewing an anime series |
        [[ -z ${NO_SESSIONS:-} ]] && Restore_Previous_Session $Anime_Type
        [[ ! -f $DIR_DET/../.last_anime_type ]] && echo $Anime_Type > $DIR_DET/../.last_anime_type
        [[ -f   $DIR_DET/../.last_anime_type && `< $DIR_DET/../.last_anime_type` != $Anime_Type ]] && sed -i s_.*_"$Anime_Type"_ $DIR_DET/../.last_anime_type

        # If we provided the '--no-update' option, then skip the first procedure and move on to our 'Anime_Selection' menu
        if [[ -z ${SKIP:-} ]]; then Extract_Series_List $Anime_Type; Clean_Up directory; Anime_Selection $Anime_Type
        else
          [[ ! -d $DIR_DET/$Anime_Type/ ]] && mkdir -p $DIR_DET/$Anime_Type/
          cd      $DIR_DET/$Anime_Type/; Clean_Up directory; Anime_Selection $Anime_Type
        fi
        # Restore the original computer display
        Restore_Monitor
      ;;
      *) # The wrong streaming argument was provided.
        echo -e "${N}${R}[${FUNCNAME[0]}] ${A}${G}Sorry, but the only streaming options available right now are: $args" >&2
        exit 17
      ;;
    esac
  ;;
  --get) # Allow the user the ability to download anime and manga
    case $Anime_Type in
      anime-[ds]ub|cartoon|movie|ova) # Enter a menu, where the user may select which anime they want to download

        #  Before we select our anime of choice: Let's find out if the user wants to restore any             |
        #+ existing sessions of Genshiken, that was previously downloading/streaming/viewing an anime series |
        [[ -z ${NO_SESSIONS:-} ]] && Restore_Previous_Session $Anime_Type
        [[ ! -f $DIR_DET/../.last_anime_type ]] && echo $Anime_Type > $DIR_DET/../.last_anime_type
        [[ -f   $DIR_DET/../.last_anime_type && `< $DIR_DET/../.last_anime_type` != $Anime_Type ]] && sed -i s_.*_"$Anime_Type"_ $DIR_DET/../.last_anime_type

        # If we provided the '--no-update' option, then skip the first procedure and move on to our 'Anime_Selection' menu
        if [[ -z ${SKIP:-} ]]; then Extract_Series_List $Anime_Type; Clean_Up directory; Anime_Selection $Anime_Type
        else
          [[ ! -d $DIR_DET/$Anime_Type/ ]] && mkdir -p $DIR_DET/$Anime_Type/
          cd      $DIR_DET/$Anime_Type/; Clean_Up directory; Anime_Selection $Anime_Type
        fi
      ;;
      manga) : # Process a text file containing a set of links
:<<    'MANGA-PROCEDURE'
--------if [[ ! -z ${3:-} ]]; then
----------if [[ `sed 's_.*\(.txt\)_\1_' <<< $3` != '.txt' ]]; then
------------{
--------------echo -e "\n${N}${R}[${FUNCNAME[0]}] ${A}${G}The second argument must be a text file containing title links from ${M}${2}${G}land!\n ${N}"
--------------echo -e "${N}${R}[${FUNCNAME[0]}] ${G}The title links are not located in the galleries-- they're on the main page. ${N}"
--------------echo -e "${N}${R}[${FUNCNAME[0]}] ${G}There's no need to enter the galleries, except to confirm if you want the ${M}$Anime_Type${G} or not. ${N}"
--------------echo -e "${N}${R}[${FUNCNAME[0]}] ${G}To capture these links: Find an image of a ${M}$Anime_Type${G}, & copy the title link above it. ${N}"
--------------echo -e "${N}${R}[${FUNCNAME[0]}] ${G}Place as many of these links as you want in a .txt file; utilize file as argument 2.\n ${N}"
------------} >&2; exit 1
----------fi
----------echo -e "\n${N}${R}[${FUNCNAME[0]}] ${G}Extracting all ${M}${2}${G}s in list: \"${M}$3${G}\". ${N}"; manga_total=$(grep http $3 | wc -l)

----------# Count how many mangas there are, and loop through the below script for each manga
----------for manga in `grep http $3`
----------do
------------# Announce which manga we're on
------------if [[ $manga_count = 1 ]]
------------then echo -e "${N}${R}[${FUNCNAME[0]}] ${G}There are a total of ${M}$manga_total ${2}${G}s in list: \"${M}$3${G}\" ${N}"
------------fi; echo -e "${N}${R}[${FUNCNAME[0]}] ${G}Processing ${M}$Anime_Type $manga_count ${G}of ${M}$manga_total${G}.\n"

------------# Call the 'Extract_Files' function, in order to extract all the pictures for the requested manga
------------Extract_Files $manga; echo -e "${N}${R}[${FUNCNAME[0]}] ${G}Finished extracting \"${M}$manga${G}\" ==FROM== \"${M}$3${G}\". ${N}"
------------eval $ADD_MANGA
----------done; echo -e "\n${N}${R}[${FUNCNAME[0]}] ${G}Finished extracting all ${M}${2}${G}s in list: \"${M}$3${G}\"\n"
--------else
----------echo -e "\n${N}${R}[${FUNCNAME[0]}] ${A}${G}This option has not been automated yet!  It requires an additional argument to work.\n ${N}" >&2
----------exit 1
--------fi
MANGA-PROCEDURE
      ;;
      manga) # NOT READY YET
        echo -e "\n${N}${R}[${FUNCNAME[0]}] ${A}${G}Sorry!  You can't use this option yet.  It will be ready in the next release.\n ${N}" >&2; exit 1
      ;;
      *) # The wrong download argument was provided.
        echo -e "${N}${R}[${FUNCNAME[0]}] ${A}${G}Sorry, but the only download options available right now are: $args ${N}" >&2
        exit 15
      ;;
    esac
  ;;
  --view) # Allow the user the ability to watch videos/images downloaded with Genshiken
    case $Anime_Type in
      anime-[ds]ub|cartoon|movie|ova) # View all the anime that has been downloaded with genshiken

        # Determine if the user has an HMDI monitor connected, and if he wants to continue genshiken on the other monitor
        HDMI_Setup

        #  Before we select our anime of choice: Let's find out if the user wants to restore any             |
        #+ existing sessions of Genshiken, that was previously downloading/streaming/viewing an anime series |
        [[ -z ${NO_SESSIONS:-} ]] && Restore_Previous_Session $Anime_Type
        [[ ! -f $DIR_DET/../.last_anime_type ]] && echo $Anime_Type > $DIR_DET/../.last_anime_type
        [[ -f   $DIR_DET/../.last_anime_type && `< $DIR_DET/../.last_anime_type` != $Anime_Type ]] && sed -i s_.*_"$Anime_Type"_ $DIR_DET/../.last_anime_type

        # As long as the user has already downloaded some anime with genshiken, allow them to play any anime they have with genshiken.
        [[ -d $DIR_ANIM/$Anime_Type ]] && cd $DIR_ANIM/$Anime_Type && Watch_Anime $Anime_Type ||
        {
          echo -e "${N}${R}[${FUNCNAME[0]}] ${A}${G}You've yet to download any ${M}${Anime_Type//-*/}${G}'s with \"${B}$PROGNAME ${G}\"!? ${N} \b"
          echo -e "${N}${R}[${FUNCNAME[0]}] ${G}You can't use this option, just yet. ${N} \b"
        } >&2 && exit 18

        # Restore the original computer display
        Restore_Monitor
      ;;
      manga) # View all the manga that has been downloaded with genshiken

        # As long as the user has already downloaded some manga with genshiken, allow them to view any manga they have with genshiken.
        [[ -d $DIR_PICS/ ]] && cd $DIR_PICS/ && View_Manga $Anime_Type ||
        {
          echo -e "${N}${R}[${FUNCNAME[0]}] ${A}${G}You've yet to download any ${M}${Anime_Type//-*/}${G}'s with \"${B}$PROGNAME ${G}\"!? ${N} \b"
          echo -e "${N}${R}[${FUNCNAME[0]}] ${G}You can't use this option, just yet. ${N} \b"
        } >&2 && exit 19
      ;;
      *) # The wrong viewing argument was provided.
        echo -e "${N}${R}[${FUNCNAME[0]}] ${A}${G}Sorry, but the only viewing options available right now are: $args ${N}" >&2; exit 16
      ;;
    esac
  ;;
esac

# End:
# genshiken-main.sh ends here
