/**************************************************************************
 * FileConf - Functions for the make of the options of the command line. 
 * Version:     0.3.8 14/10/2007
 * File:        FileConf.c
 * Description: Functions for the reading, verification, execution of the 
 *              options of the command line. 
 * Error:       ID file 4, last number 14
 * Platform(s):	GNU/Linux, All
 * Author(s):	Della Bianca Giuseppe <bepi@adria.it><bepii@libero.it>
 *
 * Copyright (C) 2002-6 Della Bianca Giuseppe <bepi@adria.it><bepii@libero.it>
 * This file is part of GesConf.
 *
 * GesConf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GesConf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GesConf; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 **************************************************************************/
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include "global.h"
#define  _EL_PRIVATE_H
#define  _FC_PRIVATE_H
#include "ErrLog.h"
#include "BaseString.h"
#include "file.h"
#include "FileConf.h"

// variable for the management of the configuration file
struct _ConfPar ConfPar;
static struct _FcRows FcRows;
// variable for the management of the actions on the configuration values
static int ReadEnab;
static int SaveEnab;
static int SortEnab;
// variable for the search of the group/key 
static int FindRowNum;

static char *StrNull= "";

/**************************************************************************
 * Function: FcListConf
 * Details:  List the values of the configuration demands. 
 *           Comes always made the scansion of all the values. 
 * Returns: 
 **************************************************************************/
int FcListConf (char *group, char *key, char *val, char *WStat){

  int result;
  char OneKeyType[LENSTR + 1];
  struct _StatRow RStat;

  strcpy (OneKeyType, (char*)(ConfPar.OneKeyEnab ? "LIST" : ""));
  WStatToRStat (WStat, &RStat);
  result= ScanRowConf (group, key, val, &RStat, OneKeyType);

  return (result);
} // FcListConf

/**************************************************************************
 * Function: FcGetConf
 * Details:  Return the value of the key. 
 * Returns: 
 **************************************************************************/
int FcGetConf (char *GrpIn, char *KeyIn, char *ValKeyIn, char *val, char *WStat){

  int result;
  char OneKeyType[LENSTR + 1];
  struct _StatRow RStat;
 
  WStatToRStat (WStat, &RStat);
  result= GetKeyVal (GrpIn, KeyIn, ValKeyIn, val, &RStat);
  if (result){
    if (ConfPar.OneKeyEnab){
      strcpy (OneKeyType, "GET");
      WStatToRStat ("K", &RStat);
      result= ScanRowConf (GrpIn, KeyIn, ValKeyIn, &RStat, OneKeyType);
    }
    else{
      if (ConfPar.pfExeConfVal != NULL)
	ConfPar.pfExeConfVal (GrpIn, KeyIn, val, &RStat, "");
    }
  }

  return (result);
} // FcGetConf

/**************************************************************************
 * Function: FcSetConf
 * Details:  Set the value of the key.
 * Returns: 
 **************************************************************************/
int FcSetConf (char *GrpIn, char *KeyIn, char *ValKeyIn, char *ValIn){

  int result;

  result= SetKeyVal (GrpIn, KeyIn, ValKeyIn, ValIn);

  return (result);
} // FcSetConf

/**************************************************************************
 * Function: FcDelConf
 * Details:  Delete the group/key.
 * Returns: 
 **************************************************************************/
int FcDelConf (char *GrpIn, char *KeyIn, char *ValKeyIn, char *DelType){

  int result;

  result= DelGrpKey (GrpIn, KeyIn, ValKeyIn, DelType);

  return (result);
} // FcDelConf

/**************************************************************************
 * Function: FcSetConfToComm
 * Details:  Set the key in to comment.
 * Returns: 
 **************************************************************************/
int FcConfToComm (char *GrpIn, char *KeyIn, char *ValKeyIn){
  int result;

  result= SetKeyToComm (GrpIn, KeyIn, ValKeyIn);

  return (result);
} // FcSetConfToComm

/**************************************************************************
 * Function: FcSetLockConf
 * Details:  Set the lock of the group/key. 
 * Returns: 
 **************************************************************************/
int FcSetLockConf (char *GrpIn, char *KeyIn, char *ValKeyIn, char *lock){
  int result;

  result= SetLockGrpKey (GrpIn,KeyIn, ValKeyIn, lock, true);

  return (result);
} // FcSetLockConf

/**************************************************************************
 * Function: FcReadFile
 * Details:  Reads the valuse from configuration file.
 * Returns: 
 **************************************************************************/
int FcReadFile (){
  int result;

  result= ReadFile (ConfPar.ConfFile);
  return (result);    
} //FcReadFile

/**************************************************************************
 * Function: FcWriteFile
 * Details:  Write the values in the configuration file.
 * Returns: 
 **************************************************************************/
int FcWriteFile (){
  int result;

  if (!ReadEnab && SaveEnab)
    result= WriteFile (ConfPar.ConfFile);

  return (result);    
} // FcWriteFile

/**************************************************************************
 * Function: FcStatFormRStat
 * Details:  Convert the row status in the work status.
 * Returns: 
 **************************************************************************/
void FcStatFromRStat (char *WStat, struct _StatRow *stat){

  WStatFromRStat (WStat, stat);

} // FcStatFromRStat

/**************************************************************************
 * Function: FcInit
 * Details:  Initialize the parameters of the configuration.  
 * Returns: 
 **************************************************************************/
void FcInit (){

  InitConf ("CONF");
  InitConf ("FIND");    
} //FcInit

/**************************************************************************
 * Function: OneValFromMultiKey
 * Details:  Make one value from multiple key for the ScanfRowConf
 * Returns: 
 **************************************************************************/
static int OneValFromMultiKey (struct _ConfRow *ActSRow, struct _ConfRow *OldSRow, char *OneKeyType){

  int result= true, PrintAct= true, PrintOld= false, NewKey= false;
  char ActVal[LLENGKV + 1];

  if (*OneKeyType != '\0'){
    if (!(PrintOld= ActSRow == 0)){
      if (!(PrintOld= ActSRow->stat.type != 'K')){
	NewKey= strcmp (ActSRow->WorkGrp, OldSRow->WorkGrp);
	NewKey= NewKey || strcmp (ActSRow->key, OldSRow->key);
	NewKey= NewKey || ActSRow->stat.type != OldSRow->stat.type;
	PrintOld= PrintOld || NewKey;
	PrintAct= false;
      }
    }
// if is almost one previus configuration
    if (PrintOld && OldSRow->RowIniGrp > 0){
      if (OldSRow->RowIniGrp > 1)
	OldSRow->stat.lock= OldSRow->stat.edit= '?';
      if (ConfPar.pfExeConfVal != NULL)
	ConfPar.pfExeConfVal (OldSRow->WorkGrp, OldSRow->key, OldSRow->val, &OldSRow->stat, OneKeyType);
      strcpy (OldSRow->val, "");
      OldSRow->RowIniGrp= 0;
    }
    if (NewKey){
      OldSRow->WorkGrp= ActSRow->WorkGrp;
      OldSRow->key= ActSRow->key;
      OldSRow->stat= ActSRow->stat;
    } // NewKey
    if (!PrintAct){
      result= AddTxtWithSep (ActVal, OldSRow->val, ActSRow->val, ConfPar.ValSep, LLENGKV);
      strcpy (OldSRow->val, ActVal);
      OldSRow->RowIniGrp++;
    }
  } // if (*OneKeyType != '\0')

  if (PrintAct && ActSRow != 0){
    if (ConfPar.pfExeConfVal != NULL)
	ConfPar.pfExeConfVal (ActSRow->WorkGrp, ActSRow->key, ActSRow->val, &ActSRow->stat, OneKeyType);
  }

  return (result);
} // OneValFromMultiKey

/**************************************************************************
 * Function: ScanRowConf
 * Details:  List the values of the configuration demands. 
 *           Comes always made the scansion of all the values. 
 * Returns: 
 **************************************************************************/
static int ScanRowConf (char *group, char *key, char *val, struct _StatRow *stat, char *OneKeyType){

  int result= true, RowNum, GrpFind, GrpOk, KeyFind, CommFind, LenGrp;
  char OldVal[LLENGKV + 1];
  struct _ConfRow *ActCRow, ActSRow, OldSRow;
  struct _StatRow RStat;

// if is demande one value for multy key
  if (*OneKeyType != '\0'){
// Sort configuration rows in alphanumeric order
    SortConf ();
    *OldVal= '\0';
    OldSRow.WorkGrp= OldSRow.key= StrNull;
    OldSRow.val= OldVal;
    WStatToRStat ("???", &OldSRow.stat);
// it is, only for ScanRowConf, the number of previus configuration
    OldSRow.RowIniGrp= 0;
  }

  for (RowNum= 0; result && RowNum < FcRows.NumConf; RowNum++){
// if is demande one value for multy key the key must be sequential
    if (*OneKeyType != '\0')
      ActCRow= RowSortFc (RowNum);
    else
      ActCRow= RowFc (RowNum);
// if is demanded group
    ActSRow= *ActCRow;
    WStatToRStat ("  *", &RStat);
    ActSRow.stat.type= 'G';
    GrpFind= IsFindGroup (group, &RStat, &ActSRow, "GKOPT");
// if is one row 'group'
    if (ISGRP(ActCRow->stat.type)){
// if is demanded to only print one type of group
      GrpOk= *key == '\0' && IsFindGroup (group, stat, ActCRow, "GKOPT");
      if (GrpOk){
	ActSRow= *ActCRow;
	ActSRow.key= StrNull;
	ActSRow.val= StrNull;
	OneValFromMultiKey (&ActSRow, &OldSRow, OneKeyType);
      }
    } // if (ISGRP(ActCRow->stat.type))
    else if (GrpFind){
// if is demanded to only print one key or type ok key
      KeyFind= IsFindKey (group, key, val, stat, ActCRow, "GKOPT");
// if is comment of the key
      RStat= *stat;
      RStat.type= ' ';
      CommFind= IsFindKey (group, key, val, &RStat, ActCRow, "GKOPT");
      CommFind= CommFind && (OKSTYPE('I', stat) && *ActCRow->comm != '\0');
// if the print of the value is demanded
      if (CommFind){
	ActSRow= *ActCRow;
	ActSRow.val= ActCRow->comm;
	WStatToRStat ("IUU", &ActSRow.stat);
	OneValFromMultiKey (&ActSRow, &OldSRow, OneKeyType);
      }
      if (KeyFind)
	result= OneValFromMultiKey (ActCRow, &OldSRow, OneKeyType);
    } // else if (GrpFind)
  } //  for (RowNum= 0; result && RowNum < FcRows.NumConf; RowNum++){
// if is demande one value for multy key
  if (*OneKeyType != '\0')
    OneValFromMultiKey ((struct _ConfRow *) NULL, &OldSRow, OneKeyType);

  return (result);
} // ScanRowConf

/**************************************************************************
 * Function: GetKeyVal
 * Details:  Return the value of the key. 
 * Returns: 
 **************************************************************************/
static int GetKeyVal (char *GrpIn, char *KeyIn, char *ValKeyIn, char *val, struct _StatRow *stat){

  int result, RowNGrp, RowNKey, RowNIns;
  char EnabType[LENSTR + 1];
  struct _StatRow RStat;

  *val= stat->type= stat->lock= stat->edit= '\0';

  strcpy (EnabType, "GET");
  WStatToRStat ("K", &RStat);
  result= CheckGrpKey (GrpIn, KeyIn, ValKeyIn, "", EnabType, &RStat, &RowNGrp, &RowNKey, &RowNIns);

  if (result && !strcmp (EnabType, "GET") && RowNKey > -1){
    strcpy (val, RowFc (RowNKey)->val);
    *stat= RowFc (RowNKey)->stat;
  }

  return (result);
} //GetKeyVal

/**************************************************************************
 * Function: SetKeyVal
 * Details:  Set the value of the key.
 * Returns: 
 **************************************************************************/
static int SetKeyVal (char *GrpIn, char *KeyIn, char *ValKeyIn, char *ValIn){

  int result, NewGrp, NewKey;
  int RowNGrp, RowNKey, RowNIns, RowNGrpLock, RowNKeyLock;
  char EnabType[LENSTR + 1];
  struct _StatRow RStat;

  strcpy (EnabType, "INS");
  WStatToRStat ((char*)(*KeyIn == '\0' ? "G" : "K"), &RStat);
  result= CheckGrpKey (GrpIn, KeyIn, ValKeyIn, ValIn, EnabType, &RStat, &RowNGrp, &RowNKey, &RowNIns);
  if (result){
    NewGrp= *GrpIn != '\0' && RowNGrp < 0;
    NewKey= *KeyIn != '\0' && RowNKey < 0;
    if (NewGrp || NewKey){
      if (!strcmp (EnabType, "INS")){
	WStatToRStat ((char*)(NewGrp ? "GUA" : "KUA"), &RStat);
	result= InsGrpKey (GrpIn, KeyIn, ValIn, &RStat, RowNIns);
      }
    }
    else{
      if (RowNKey > -1){
// if changed value
	if (strcmp (ValIn, RowFc (RowNKey)->val)){
	  if (!strcmp (EnabType, "INS")){
	    result= EditRow (RowFc (RowNKey), ValIn, "M", true);
// verify if changed lock value
	    if (result){
	      WStatToRStat ("k", &RStat);
	      FindGrpKey (GrpIn, KeyIn, ValKeyIn, &RStat, &RowNGrpLock, &RowNKeyLock);
// set the value in the key that signals the lock
	      if (RowNKeyLock > -1){
		if (strcmp (ValIn, RowFc (RowNKeyLock)->val))
		  result= EditRow (RowFc (RowNKeyLock), ValIn, "M", true);
	      }
	    }
	  } // if (!strcmp (EnabType, "INS"))
	} // if (strcmp (ValIn, RowFc (RowNKey)->val))
      } // if (RowNKey > -1)
    } // else if (NewGrp || NewKey)
  } // if (result)

  return (result);
} //SetKeyVal

/**************************************************************************
 * Function: DelGrpKey
 * Details:  Delete the group/key.
 * Returns: 
 **************************************************************************/
static int DelGrpKey (char *GrpIn, char *KeyIn, char *ValKeyIn, char *DelType){

  int result, DelGrpEnab, RowNum, RowNGrp, RowNKey, RowNIns;
  char EnabType[LENSTR + 1];
  struct _ConfRow *ActCRow;
  struct _StatRow RStat;

  strcpy (EnabType, "EDIT");
  DelGrpEnab= *KeyIn == '\0' && strcmp (DelType, "DELCOMM");
  if (DelGrpEnab)
    WStatToRStat ("G", &RStat);
  else
    WStatToRStat ((char*)(strcmp (DelType, "DELCOMM") ? "K" : "I"), &RStat);

  result= CheckGrpKey (GrpIn, KeyIn, ValKeyIn, "", EnabType, &RStat, &RowNGrp, &RowNKey, &RowNIns);

  RowNum= DelGrpEnab ? RowNGrp : RowNKey;
  if (result && !strcmp (EnabType, "EDIT") && RowNum > -1){
// set the status of delete
    if (DelGrpEnab){
      for (RowNum= 0; RowNum < FcRows.NumConf; RowNum++){
	ActCRow= RowFc (RowNum);
	if (ActCRow->stat.type == 'G'){
// if is a same or parent group
	  if (IsSameGroup (GrpIn, ActCRow->WorkGrp))
	    ActCRow->stat.edit= 'D';
	}
      } // for (RowNum= 0; RowNum < FcRows.NumConf; RowNum++)
    } // if (DelGrpEnab)
    else
      RowFc (RowNum)->stat.edit= 'D';

    SaveEnab= true;
  }

  return (result);
} // DelGrpKey

/**************************************************************************
 * Function: SetKeyToComm
 * Details:  Set the key in to comment.
 * Returns: 
 **************************************************************************/
static int SetKeyToComm (char *GrpIn, char *KeyIn, char *ValKeyIn){

  int result, RowNGrp, RowNKey, RowNIns, write;
  char EnabType[LENSTR + 1], row[MAXLENROWS + 1];
  struct _StatRow RStat;
  struct _ConfRow* ActCRow, *OldDRow;

  strcpy (EnabType, "EDIT");
  WStatToRStat ("K", &RStat);
  result= CheckGrpKey (GrpIn, KeyIn, ValKeyIn, "", EnabType, &RStat, &RowNGrp, &RowNKey, &RowNIns);

  if (result && !strcmp (EnabType, "EDIT") && RowNKey > -1){
    OldDRow= (struct _ConfRow *) NULL;

    ActCRow= RowFc (RowNKey);
// set the status of change in to comment
    ActCRow->stat.edit= 'C';
    result= ConfToFile (row, false, ActCRow, OldDRow, &write);
    if (result){
      ActCRow->stat.type= 'I';
      strcpy (ActCRow->key, "");
      result= EditRow (ActCRow, row, "M", true);
    }
  }

  return (result);
} // SetKeyToComm

/**************************************************************************
 * Function: SetLockGrpKey
 * Details:  Set the lock of the group/key. 
 * Returns: 
 **************************************************************************/
static int SetLockGrpKey (char *GrpIn, char *KeyIn, char *ValKeyIn, char *lock, int modify){

  int result, RowNum, RowNGrp, RowNKey, RowNIns, RowNLock;
  char EnabType[LENSTR + 1];
  struct _StatRow RStat;
  struct _ConfRow* ActCRow;

  if (result= (!strcmp (lock, "U") || !strcmp (lock, "L"))){
    WStatToRStat ((char*)(*KeyIn == '\0' ? "G" : "K"), &RStat);
    if (modify){
      strcpy (EnabType, "GET");
      result= CheckGrpKey (GrpIn, KeyIn, ValKeyIn, "", EnabType, &RStat, &RowNGrp, &RowNKey, &RowNIns);
    }
    else
      FindGrpKey (GrpIn, KeyIn, ValKeyIn, &RStat, &RowNGrp, &RowNKey);
  }
  else
    ElAddErr1 (409, "BV", (char *)_("The value of lock %s is not managed"), lock);

  if (result){
    RowNum= *KeyIn == '\0' ? RowNGrp : RowNKey;
// if has been found the row to set to onlock
    if (RowNum > -1){
// the modify at the row must made before inserting one new row 
      ActCRow= RowFc (RowNum);
      modify= modify && ActCRow->stat.lock != *lock;
      ActCRow->stat.lock= *lock;
// if is demanded the modify the status and edit key lock
      if (modify){
	EditRow (ActCRow, "", "M", false);
// inserts the row that signal the lock
	if (*lock == 'L'){
// verify if the group/key lock exist
	  WStatToRStat ((char*)(*KeyIn == '\0' ? "g" : "k"), &RStat);
	  FindGrpKey (GrpIn, KeyIn, ActCRow->val, &RStat, &RowNGrp, &RowNKey);
	  RowNLock= *KeyIn == '\0' ? RowNGrp : RowNKey;
// edit the group/key lock 
	  if (RowNLock > -1){
	    if (strcmp (ActCRow->val, RowFc (RowNLock)->val))
	      result= EditRow (ActCRow, ActCRow->val, "M", true);
	  }
	  else{
	    WStatToRStat ((char*)(*KeyIn == '\0' ? "gUA" : "kUA"), &RStat);
	    result= InsGrpKey (GrpIn, KeyIn, ActCRow->val, &RStat, RowNum + 1);
	  }
	} // if (*lock == 'L')
      } // if (modify)
    } // if (RowNum > -1)
  } // if (result)

  return (result);
} // SetLockGrpKey

/**************************************************************************
 * Function: InsGrpKey
 * Details:  Insert one group or one key in the demanded row number.
 *           If the demanded row number is -1 or is greater to the 
 *           existing last row, the values is added after the 
 *           last row. 
 * Returns: 
 **************************************************************************/
static int InsGrpKey (char *grp, char *key, char *val, struct _StatRow *stat, int RowNIns){

  int result, RowNum, NumIns, RowIniGrp;
  char ActGrp[LENGKV + 1], ActRGrp[LENGKV + 1], IniRGrp[LENGKV + 1];
  char EndPartGrp[LENGKV + 1];
  char str[LENSTR + 1];
  char *KeyAdd, *ValAdd;
  struct _ConfRow NewCRow, *ActCRow, *IniGRow;

  if (stat->type == 'G'){
    NumIns= !ISTFILE("ONEGRP") ? 2 : 1;
    if (*key != '\0' )
      NumIns++;
  }
  else
    NumIns= 1;

  if (result= FcRows.NumConf + NumIns <= MAX_NCONFROWS){
    SortEnab= true;

    if (RowNIns < 0 || RowNIns > FcRows.NumConf)
      RowNIns= FcRows.NumConf;
// update the ConfRow with the new row number
    for (RowNum= FcRows.NumConf - 1; RowNum >= RowNIns; RowNum--){
      ActCRow= RowFc (RowNum);
      RowIniGrp= ActCRow->RowIniGrp;
      if (ActCRow->RowIniGrp >= RowNIns)
	ActCRow->RowIniGrp += NumIns;
      if (ActCRow->RowUpGrp >= RowNIns)
	ActCRow->RowUpGrp += NumIns;
// set the pointer at the group end
      if (RowIniGrp > -1){
	IniGRow= RowFc (RowIniGrp);
	if (RowNum + NumIns > IniGRow->RowEndGrp)
	  IniGRow->RowEndGrp= RowNum + NumIns;
      }
      FcRows.conf[RowNum + NumIns]= *ActCRow;
    }
// if the insert change the row number of the previous search group/key 
    if (RowNIns <=  FindRowNum)
      FindRowNum= -1;
// set the new conf group/key/value/group end
    KeyAdd= ValAdd= StrNull;
    NewCRow.WorkGrp= NewCRow.RealGrp= NewCRow.key= NewCRow.val= NewCRow.comm= StrNull;
    NewCRow.RowIniGrp= NewCRow.RowUpGrp= NewCRow.RowEndGrp= -1;
    RowNum= RowNIns - 1;
// if the previus rows is a group end
    if (RowNum > -1){
      IniGRow= RowFc (RowNum);
      if (IniGRow->stat.type == 'E')
	RowNum= IniGRow->RowUpGrp;
    }
    if (RowNum > -1){
      RowIniGrp= RowFc (RowNum)->RowIniGrp;
      NewCRow.RowIniGrp= RowIniGrp;
      NewCRow.RowUpGrp= RowFc (RowNum)->RowUpGrp;
// set the pointer at the group end
      if (RowIniGrp > -1){
	IniGRow= RowFc (RowIniGrp);
	if (RowNIns > IniGRow->RowEndGrp)
	  IniGRow->RowEndGrp= RowNIns;
      }
    }
    NewCRow.stat= *stat;
    strcpy (IniRGrp, NewCRow.RowIniGrp < 0 ? "" : RowFc (NewCRow.RowIniGrp)->RealGrp);
// set the new conf group
    if (ISGRP(stat->type)){
// set the text of the virtual group
       if (ISKEYTOGRP(key))
	result= AddTxtWithSep (ActGrp, grp, val, GRPSEP, LENGKV);
      else
	strcpy (ActGrp, grp);
      if (result)
	result= AddTextConf (ActGrp, &NewCRow.WorkGrp, false);
      NewCRow.RealGrp= NewCRow.WorkGrp;
// set the text of the real group
      if (result && ISTFILE("XFREE")){
// add the new group to real group
	GetPartGrp (grp, EndPartGrp, "LAST");
	if (result= AddTxtWithSep (ActRGrp, IniRGrp, EndPartGrp, GRPSEP, LENGKV))
	  result= AddTextConf (ActRGrp, &NewCRow.RealGrp, false);
      }
      if (stat->type == 'G'){
	NewCRow.RowUpGrp= NewCRow.RowIniGrp;
	NewCRow.RowIniGrp= RowNIns;
	NewCRow.RowEndGrp= RowNIns + NumIns - 1;
      }
    } // if (ISGRP(stat.type))
    else{
      NewCRow.WorkGrp= RowNum < 0 ? StrNull : RowFc (RowNum)->WorkGrp;
      NewCRow.RealGrp= RowNum < 0 ? StrNull : RowFc (RowNum)->RealGrp;
    }
// set the conf group
    if (result && ISGRP(stat->type)){
      FcRows.conf[RowNIns]= NewCRow;
      RowNIns++;
    }
    if (result && *key != '\0'){
      if (result= AddTextConf (key, &KeyAdd, false))
	result= AddTextConf (val, &ValAdd, true);
// set the conf key
      if (result){
	FcRows.conf[RowNIns]= NewCRow;
	FcRows.conf[RowNIns].key= KeyAdd;
	FcRows.conf[RowNIns].val= ValAdd;
	FcRows.conf[RowNIns].stat.type= ISGRP(stat->type) ? 'K' : stat->type;
	RowNIns++;
      }
    } // if (result && *key != '\0')
// set the conf group end
    if (result && stat->type == 'G' && !ISTFILE("ONEGRP")){

	FcRows.conf[RowNIns]= NewCRow;
	FcRows.conf[RowNIns].stat.type= 'E';
	RowNIns++;
    }
    if (result)
      FcRows.NumConf= FcRows.NumConf + NumIns;
    SaveEnab= true;
  } // if (result= FcRows.NumConf < MAX_NCONFROWS)
  else{
    sprintf(str, "%d", MAX_NCONFROWS);
    ElAddErr2 (413, "FR", (char *)_("With %s the number of inserted lines it exceeds %s"), 
	    key, str);
  }

  return (result);
} //InsGrpKey

/**************************************************************************
 * Function: EditRow
 * Details:  Modify the status and/or the value of the row. 
 * Returns: 
 **************************************************************************/
#define ISLONGTEXT(a, b)(abs (a[0] - a[NUMLTXTROW]) == (abs (a[0] - b) + abs (a[NUMLTXTROW] - b)))

static int EditRow (struct _ConfRow *ActCRow, char *val, char *edit, int SetVal){

  int EditOk= true, result= true;
  char *ValAdd;

// set the status
  if (ActCRow->stat.edit != *edit){
    if (ActCRow->stat.edit == 'A')
      EditOk= *edit != 'M';

    if (EditOk)
      ActCRow->stat.edit= *edit;
    SaveEnab= true;
  } // if (ActCRow->stat.edit != *edit)
// set the value
  if (SetVal && strcmp (val, ActCRow->val)){
// set the long text
    if (!ISLONGTEXT(FcRows.LongText, ActCRow->val) && strlen (val) > LENGKV){
      result= AddTextConf (val, &ValAdd, true);
      if (result)
	ActCRow->val= ValAdd;
    }
    else
      strcpy (ActCRow->val, val);
    SaveEnab= true;
  }

  return (result);
} // EditRow

/**************************************************************************
 * Function: CheckGrpKey
 * Details:  Check group, key, val, status and return the number of row of
 *           the group and the key with demanded status. 
 * Returns: 
 **************************************************************************/
static int CheckGrpKey (char *group, char *key, char *ValKey, char *val, char *EnabType, 
			struct _StatRow *stat, int *RowNGrp, int *RowNKey, int *RowNIns){

  int result= true, NewGrp, NewKey, ChangeGrp, ChangeKey, GrpLock, KeyLock;
  int RowNum, ActRowNGrp, ActRowNKey;
  char WorkGrp[LENGKV + 1], WorkKey[LENGKV + 1], PrevGrp[LENGKV + 1];
  struct _StatRow RStat, WStat;

  *RowNIns= FcRows.NumConf + ISTFILE("ONEGRP") ? -1 : 0;
  ChangeGrp= ChangeKey= GrpLock= KeyLock= false;

  strcpy (WorkGrp, group);
  strcpy (WorkKey, key);
  WStat= *stat;
// set the text of the virtual group
  if (!strcmp (EnabType, "INS") && ISKEYTOGRP(key)){
    result= AddTxtWithSep (WorkGrp, group, val, GRPSEP, LENGKV);
    *WorkKey= '\0';
    WStat.type= 'G';
  }

  FindGrpKey (WorkGrp, WorkKey, ValKey, &WStat, RowNGrp, RowNKey);
  if (result){
    ActRowNGrp= *RowNGrp;
    if (!strcmp (EnabType, "INS")){
      if (result= *group != '\0' || *key != '\0'){
// verify for insert a new key
	NewGrp= *RowNGrp < 0;
	NewKey= WStat.type == 'K' && *RowNKey < 0;
// search the parent of new group
	if (NewGrp){
	  GetPartGrp (group, PrevGrp, "PREVS");
	  if (*PrevGrp != '\0'){
	    WStatToRStat ("G", &RStat);
	    FindGrpKey (PrevGrp, "", "", &RStat, &ActRowNGrp, &ActRowNKey);
	    if (!(result= ActRowNGrp > -1))
	      ElAddErr1 (405, "BG", (char *)_("The group %s is not parent of one group"), group);
	  }
	}
// set the insert position
	if (ActRowNGrp > -1)
	  *RowNIns= RowFc (ActRowNGrp)->RowEndGrp;
	else{
	  if (ISTFILE("ONEGRP") && *group == '\0' && *RowNIns > -1){
	    if (strcmp (group, RowFc (*RowNIns)->WorkGrp))
	      *RowNIns= -1;
	  }
	}
	if (ISTFILE("ONEGRP"))
	  (*RowNIns)++;
// verify if changed value
	if (!NewKey && *RowNKey > -1)
	  ChangeKey= strcmp (val, RowFc (*RowNKey)->val);
	ChangeGrp= ActRowNGrp > -1 && (NewGrp || NewKey || ChangeKey);
// verify is anybody change
	if (!ChangeGrp && !ChangeKey && !NewGrp && !NewKey)
	  *EnabType= '\0';
      } // if (result)
      else
	ElAddErr1 (402, "BK", (char *)_("The group and the key is empty"), group);
    } // if (!strcmp (EnabType, "INS"))
    else{
      if (strcmp (EnabType, "GET"))
	ChangeGrp= ChangeKey= true;
// verify if the group/key is existent
      RowNum= stat->type == 'G' ? *RowNGrp : *RowNKey;
      if (RowNum <= -1 && *ConfPar.WarnType != 'O'){
	result= *ConfPar.WarnType != '\0';
	if (!result || *ConfPar.WarnType == 'W'){
	  if (stat->type == 'G')
	    ElAddErr1 (406, "BG", (char *)_("Group %s nonexistent"), group);
	  else if (stat->type == 'K')
	    ElAddErr (407, "BK", (char *)_("Key %s %s %s nonexistent"), group, key, ValKey);
	  else
	    ElAddErr2 (408, "BV", (char *)_("Comment %s %s nonexistent"), group, ValKey);
	}
      }
    }
  } // if (result)
// verify if the group/key is onlock
  if (result && (ChangeGrp || ChangeKey) && *EnabType != '\0'){
    if (ActRowNGrp > -1 && ChangeGrp)
      GrpLock= RowFc (ActRowNGrp)->stat.lock == 'L' && *ConfPar.WarnType != 'O';
    if (*RowNKey > -1 && ChangeKey)
      KeyLock= RowFc (*RowNKey)->stat.lock == 'L' && *ConfPar.WarnType != 'O';
// verify if the group/key onlock is fatal
    if (GrpLock || KeyLock){
      strcpy (EnabType, "ONLOCK");
      result= *ConfPar.WarnType != '\0';
      if (!result || *ConfPar.WarnType == 'W'){
	if (GrpLock)
	  ElAddErr1 (403, "BG", (char *)_("The group %s is onlock"), group);
	else if (KeyLock)
	  ElAddErr (404, "BK", (char *)_("The key %s %s %s is onlock"), group, key, ValKey);
      }
    }
  } // if (result && (ChangeGrp || ChangeKey) && *EnabType != '\0')
// verify is a virtual group
  if (result && ChangeKey && *EnabType != '\0'){
    if (!(result= !ISKEYTOGRP(key)))
      ElAddErr1 (412, "BK", (char *)_("The names of the groups do not have to be modified"), "");
  }
  return (result);
} // CheckGrpKey

/**************************************************************************
 * Function: FindGrpKey
 * Details:  Return the number of row of the group and the key with 
 *           demanded status. 
 * Returns: 
 **************************************************************************/
static void FindGrpKey (char *group, char *key, char *ValKey, struct _StatRow *stat, 
			int *RowNGrp, int *RowNKey){

  int RowNum, RowNIni, GrpFind, KeyFind, LenGrp;
  struct _ConfRow *ActCRow;

  RowNIni= *RowNGrp= *RowNKey= -1;
  GrpFind= KeyFind= false;
// verify if the group/key to try is the same one of the previous search
  if (FindRowNum > -1){
    ActCRow= RowFc (FindRowNum);
    if (!strcmp (ActCRow->WorkGrp, group)){
      RowNIni= ActCRow->RowIniGrp;
// if is key and row type demanded 
      if (IsFindKey (group, key, ValKey, stat, ActCRow, ""))
	RowNIni= FindRowNum;
    }
  } // if (FindRowNum > -1)
  FindRowNum= -1;
  RowNIni= RowNIni < 0 ? 0 : RowNIni;
// search the group/key
  for (RowNum= RowNIni; RowNum < FcRows.NumConf; RowNum++){
    ActCRow= RowFc (RowNum);
// verify the group
    if (!GrpFind && !strcmp (ActCRow->WorkGrp, group)){
// if is demanded group and row type
      if (ISGRP(stat->type)){
	GrpFind= IsFindGroup (group, stat, ActCRow, "");
	if (GrpFind){
	  *RowNGrp= FindRowNum= RowNum;
	  break;
	}
      }
      else{
	*RowNGrp= FindRowNum= ActCRow->RowIniGrp;
	GrpFind= true;
      }
    } // if (!GrpFind && !strcmp (ActCRow->WorkGrp, group))
    if (GrpFind){
/* if the search of the key has arrived to after of group wants to say that 
   key does not exist */
      if (!IsSameGroup (group, ActCRow->WorkGrp))
	break;
// if is key and row type demanded 
      KeyFind= IsFindKey (group, key, ValKey, stat, ActCRow, "");
      if (KeyFind){
	*RowNKey= FindRowNum= RowNum;
	break;
      }
    } // if (GrpFind)
  } // for (RowNum= RowNIni; RowNum < FcRows.NumConf; RowNum++)
} //FindGrpKey

/**************************************************************************
 * Function: IsFindGroup
 * Details:  Return if is the group demanded. 
 * Returns: 
 **************************************************************************/
static int IsFindGroup (char *group, struct _StatRow *stat, struct _ConfRow *ActCRow, char *FindType){

  int GrpFind= false, LenGrp;
  struct _ConfRow ActFRow;

  ActFRow= *ActCRow;
// if is a group
  if (ISGRP(ActFRow.stat.type)){
// if group of the group end is delete
    if (ActFRow.RowIniGrp > -1){
      if (RowFc (ActFRow.RowIniGrp)->stat.edit == 'D')
	ActFRow.stat.edit= 'D';
    }
// if is demanded match with the initial group (group terminated with GRPSEP) or only one group
    LenGrp= strlen (group) - 1;
    if (LenGrp >= 0 && group[LenGrp] == *GRPSEP)
      GrpFind= !strncmp (ActCRow->WorkGrp, group, LenGrp);
    else
      GrpFind= !strcmp (ActCRow->WorkGrp, group);
// if is demanded group and row type
    GrpFind= GrpFind || (*group == '\0' && !strcmp (FindType, "GKOPT"));
    GrpFind= GrpFind && OKSTYPE(ActFRow.stat.type, stat);
    GrpFind= GrpFind && OKSLOCK(ActFRow.stat.lock, stat) && OKSEDIT(ActFRow.stat.edit, stat);
  } // if (ISGRP(ActFRow.stat.type))

  return (GrpFind);
} // IsFindGroup

/**************************************************************************
 * Function: IsFindKey
 * Details:  Return if is the group/key demanded. 
 * Returns: 
 **************************************************************************/
static int IsFindKey (char *group, char *key, char *ValKey, struct _StatRow *stat, 
		      struct _ConfRow *ActCRow, char *FindType){

  int KeyFind= false;
  struct _ConfRow ActFRow, ActGRow;
  struct _StatRow RStat;

  ActFRow= ActGRow= *ActCRow;
// if is a key
  if (ISKEY(ActFRow.stat.type) || ActFRow.stat.type == 'I'){
// if group of the key is delete
    if (ActFRow.RowIniGrp > -1){
      if (RowFc (ActFRow.RowIniGrp)->stat.edit == 'D')
	ActFRow.stat.edit= 'D';
    }
// if is demanded group
    WStatToRStat ("  *", &RStat);
    ActGRow.stat.type= 'G';
    KeyFind= IsFindGroup (group, &RStat, &ActGRow, FindType);
// if is demanded key and row type
    KeyFind= !strcmp (ActFRow.key, key) || (*key == '\0' && !strcmp (FindType, "GKOPT"));
    KeyFind= KeyFind && OKSTYPE(ActFRow.stat.type, stat);
    KeyFind= KeyFind && OKSLOCK(ActFRow.stat.lock, stat) && OKSEDIT(ActFRow.stat.edit, stat);
    KeyFind= KeyFind && (*ValKey == '\0' || !strcmp (ActFRow.val, ValKey));
  } // if (ISKEY(ActFRow.stat.type) || ActFRow.stat.type == 'I')

  return (KeyFind);
} // IsFindKey

/**************************************************************************
 * Function: IsSameGroup
 * Details:  Return if one group is same or parent of another group
 * Returns: 
 **************************************************************************/
static int IsSameGroup (char *group, char *InGrp){

  int LenGrp, result= false;

  LenGrp= strlen (group);
// if is a parent group
  if (strlen (InGrp) > LenGrp){
    if (InGrp[LenGrp] == *GRPSEP && !strncmp (InGrp, group, LenGrp))
      result= true;
  }
  else{
    if (!strcmp (InGrp, group))
      result= true;
  }

  return (result);
} // IsSameGroup

/**************************************************************************
 * Function: ConfRCmp
 * Details:  Compare the configuration rows in base to alphanumeric order.
 * Returns: 
 **************************************************************************/
static int ConfRCmp (const struct _ConfRow **ConfR1, const struct _ConfRow **ConfR2){

  int NiceGrp1, NiceGrp2, result;
  char *NiceType, ActType, *FindType;
  const struct _ConfRow *ActConfR1, *ActConfR2;

  ActConfR1= *ConfR1;
  ActConfR2= *ConfR2;
  NiceType= NICETYPE;

// Set order from text of group
  result= GrpCmp (ActConfR1->WorkGrp, ActConfR2->WorkGrp);
// Set order from type of configuration rows
  if (result == 0){
    ActType= ActConfR1->stat.type;
    FindType= strchr (NiceType, (int)ActType);
    NiceGrp1= FindType == NULL ? 0 : FindType - NiceType + 1;
    ActType= ActConfR2->stat.type;
    FindType= strchr (NiceType, (int)ActType);
    NiceGrp2= FindType == NULL ? 0 : FindType - NiceType + 1;

    result= (NiceGrp1 == NiceGrp2 ? 0 : NiceGrp1 < NiceGrp2 ? -1 : 1);
  }
// Set order from text of key
  if (result == 0)
    result= strcmp (ActConfR1->key, ActConfR2->key);
// Set order from index of configuratione rows
  if (result == 0)
    result= (ActConfR1 == ActConfR2 ? 0 : ActConfR1 < ActConfR2 ? -1 : 1);

  return (result);
} // ConfRCmp

/**************************************************************************
 * Function: GrpCmp
 * Details:  Compare the groups
 * Returns: 
 **************************************************************************/
static int GrpCmp (char *group1, char *group2){

  int result= 0;
  char *pActGrp1, *pActGrp2;
  char ActGrp1[LENGKV + 1], ActGrp2[LENGKV + 1];
  char IniPartGrp1[LENGKV + 1], IniPartGrp2[LENGKV + 1];

  strcpy (ActGrp1, group1);
  strcpy (ActGrp2, group2);
  pActGrp1= ActGrp1;
  pActGrp2= ActGrp2;
  do{
    strtoken (&pActGrp1, IniPartGrp1, GRPSEP, LENGKV);
    strtoken (&pActGrp2, IniPartGrp2, GRPSEP, LENGKV);
    result= strcmp (IniPartGrp1, IniPartGrp2);
    if (result != 0)
      break;
  } while (*IniPartGrp1 != '\0' && *IniPartGrp2 != '\0');

  return (result);
} // GrpCmp

/**************************************************************************
 * Function: GetPartGrp
 * Details:  Return the previous or end group
 * Returns: 
 **************************************************************************/
static void GetPartGrp (char *GrpIn, char *PartGrp, char *PartType){

  char *FindSep;
  int LenGrp= 0;

  *PartGrp= '\0';
  if (!strcmp (PartType, "FIRST"))
    FindSep= strchr (GrpIn, (int)*GRPSEP);
  else
    FindSep= strrchr (GrpIn, (int)*GRPSEP);
  if (FindSep != NULL){
    LenGrp= FindSep - GrpIn;
    if (!strcmp (PartType, "LAST"))
      LenGrp++;
  }

  if (!strcmp (PartType, "FIRST") || !strcmp (PartType, "PREVS"))
    strncpyV (PartGrp, GrpIn, LenGrp, LenGrp);
  else if (!strcmp (PartType, "LAST")){
    strcpy (PartGrp, &GrpIn[LenGrp]);
  }
} // GetPartGrp

/**************************************************************************
 * Function: RowFc
 * Details:  Update the conf value and return pointer at the FcRows.conf 
 *           demanded.
 * Returns: 
 **************************************************************************/
static struct _ConfRow *RowFc (int RowNum){

  struct _ConfRow *ActCRow;

  ActCRow= &FcRows.conf[RowNum];
  UpdateRowFc (ActCRow);

  return (ActCRow);
} // RowFc

/**************************************************************************
 * Function: RowSortFc
 * Details:  Update the conf value and return pointer at the FcRows.SortConf 
 *           demanded.
 * Returns: 
 **************************************************************************/
static struct _ConfRow *RowSortFc (int SortRNum){

  struct _ConfRow *ActCRow;

  ActCRow= FcRows.SortConf[SortRNum];
  UpdateRowFc (ActCRow);

  return (ActCRow);
} // RowSortFc

/**************************************************************************
 * Function: UpdateRowFc
 * Details:  Update the conf value and return pointer at the FcRows.conf 
 *           demanded.
 * Returns: 
 **************************************************************************/
static void UpdateRowFc (struct _ConfRow *ActCRow){

  int RowIniGrp;
  struct _ConfRow *IniGRow;

  RowIniGrp= ActCRow->RowIniGrp;
  if (RowIniGrp > -1){
    IniGRow= &FcRows.conf[RowIniGrp];
// set the the work group
    if (ActCRow->WorkGrp != IniGRow->WorkGrp)
      ActCRow->WorkGrp= IniGRow->WorkGrp;
// set the pointer at the group end
    if (ActCRow->RowEndGrp != IniGRow->RowEndGrp)
      ActCRow->RowEndGrp= IniGRow->RowEndGrp;
  }
} // UpdateGcRow

/**************************************************************************
 * Function: SortConf
 * Details:  Sort configuration rows in alphanumeric order.
 * Returns: 
 **************************************************************************/
static void SortConf (){

  if (SortEnab){
    qsort (FcRows.SortConf, FcRows.NumConf, sizeof (*FcRows.SortConf), 
	   (int (*) (const void *, const void *)) ConfRCmp);
    SortEnab= false;
  }
} // SortConf

/**************************************************************************
 * Function: ReadFile
 * Details:  Reads the valuse from configuration file.
 * Returns: 
 **************************************************************************/
static int ReadFile (char *file){

  int result= true, RowUpGrp, RowIniGrp;
  char str[LENSTR + 1];
  char StatFile[LENSTR + 1], row[MAXLENROWS + 1];
  char ActGrp[LENGKV + 1], ActRGrp[LENGKV + 1];
  char NewGrp[LENGKV + 1], NewKey[LENGKV + 1], NewVal[LLENGKV + 1];
  char NewComm[LENGKV + 1];
  char *pActTextGrp, *pActTextRGrp, *KeyAdd, *ValAdd, *CommAdd;
  struct _StatRow ActStat;
  struct _ConfRow *ActCRow, *IniGRow;
  FILE *FFile;

// reads the file only if it has still not been read 
  if (!ReadEnab)
    return (result);

  InitConf ("CONF");
  InitConf ("FIND");
// read from the file the group/key/value
  RowUpGrp= RowIniGrp= -1;
  *ActGrp= *ActRGrp= '\0';
  pActTextGrp= pActTextRGrp= StrNull;
  FFile= 0L;
  strcpy (StatFile, "INI");
  result= ReadRowFile (&FFile, file, StatFile, row);
  while (result && strcmp (StatFile, "END")){
    result= ReadRowFile (&FFile, "", StatFile, row);
    result= result && strrepl (row, "\t", " ", MAXLENROWS);
    if (result && strcmp (StatFile, "END") && *row != '\0'){
// set the groups, key and the values readed from the row
      result= FileToConf (row, NewGrp, NewKey, NewVal, NewComm, &ActStat, ActGrp, ActRGrp);
      if (!result)
	break;
      if (ActStat.type == 'G' && ISTFILE("ONEGRP"))
	RowIniGrp= -1;
      pActTextGrp= RowIniGrp < 0 ? StrNull : RowFc (RowIniGrp)->WorkGrp;
      pActTextRGrp= RowIniGrp < 0 ? StrNull : RowFc (RowIniGrp)->RealGrp;
// if is one row 'group'
      if (ActStat.type == 'G'){
	RowUpGrp= ISTFILE("ONEGRP") ? -1 : RowIniGrp;
	RowIniGrp= FcRows.NumConf;
// set the text of new group
	if (result= AddTxtWithSep (ActGrp, pActTextGrp, NewGrp, GRPSEP, LENGKV))
	  result= AddTxtWithSep (ActRGrp, pActTextRGrp, NewGrp, GRPSEP, LENGKV);
	if (!result)
	  break;
      } // if (ActStat.type == 'G')
// set the text of the virtual group
      if (ISKEY(ActStat.type) && ISKEYTOGRP(NewKey)){
	if (result= AddTxtWithSep (ActGrp, pActTextGrp, NewVal, GRPSEP, LENGKV))
	  strcpy (pActTextGrp, ActGrp);
	else
	  break;
      }
// add the new group text
      KeyAdd= ValAdd= CommAdd= StrNull;
      if (ActStat.type == 'G'){
	result= AddTextConf (ActGrp, &pActTextGrp, false);
	pActTextRGrp= pActTextGrp;
	if (result && ISTFILE("XFREE"))
	  result= AddTextConf (ActRGrp, &pActTextRGrp, false);
      }
// add the new key/val text
      if (result){
	if (!ISGRP(ActStat.type)){
	  strcpy (ActGrp, pActTextGrp);
	  if (ISKEY(ActStat.type))
	    if (result= AddTextConf (NewKey, &KeyAdd, false));
	  if (result)
	    result= AddTextConf (NewVal, &ValAdd, true);
	}
// add the new comment text
	if (result && *NewComm != '\0')
	  result= AddTextConf (NewComm, &CommAdd, false);
      } // result
      if (!result)
	break;
// check for add new row to conf buffers
      if (result= FcRows.NumConf < MAX_NCONFROWS){
// add new row to conf buffers
	ActCRow= RowFc (FcRows.NumConf);
// set the new row
	ActCRow->WorkGrp= pActTextGrp;
	ActCRow->RealGrp= pActTextRGrp;
	ActCRow->key= KeyAdd;
	ActCRow->val= ValAdd;
	ActCRow->comm= CommAdd;
	ActCRow->stat= ActStat;
	ActCRow->RowIniGrp= RowIniGrp;
	ActCRow->RowUpGrp= RowUpGrp;
	ActCRow->RowEndGrp= -1;
// set the pointer at the group end
	if (RowIniGrp > -1){
	  IniGRow= RowFc (RowIniGrp);
	  if (FcRows.NumConf > IniGRow->RowEndGrp)
	    IniGRow->RowEndGrp= FcRows.NumConf;
	}
	FcRows.NumConf++;
// if is one row 'group end'
	if (ActStat.type == 'E'){
	  RowIniGrp= RowUpGrp;
	  RowUpGrp= RowIniGrp < 0 ? -1 : RowFc (RowIniGrp)->RowUpGrp;
	}
      } // if (result= FcRows.NumConf < MAX_NCONFROWS)
      else{
	sprintf(str, "%d", MAX_NCONFROWS);
	ElAddErr2 (414, "FR", (char *)_("The number of lines of the file %s exceeds %s"), 
		  file, str);
	break;
      }
    } // if (result && !EndFile && *row != '\0')
  } // while (result && !EndFile)
  strcpy (StatFile, "END");
  if (!(ReadRowFile (&FFile, "", StatFile, row)))
    result= false;
  if (result)
    ReadEnab= false;

  InitConf ("FIND");

  return (result);    
} //ReadFile

/**************************************************************************
 * Function: WriteFile
 * Details:  Write the values in the configuration file.
 * Returns: 
 **************************************************************************/
static int WriteFile (char *file){

  int result= true, RowNum, FormEnab, write;
  char row[MAXLENROWS + 1], StatFile[LENSTR + 1];
  struct _ConfRow *ActCRow, *OldDRow;
  FILE *FFile;

  InitConf ("FIND");

  OldDRow= (struct _ConfRow *) NULL;

  FFile= 0L;
  strcpy (StatFile, "INI");
  result= WriteRowFile (&FFile, file, StatFile, row, false);
  for (RowNum= 0; RowNum < FcRows.NumConf && result; RowNum++){
    ActCRow= RowFc (RowNum);
    FormEnab= RowNum > 0;
// set the row from configuration value
    result= ConfToFile (row, FormEnab, ActCRow, OldDRow, &write);
    if (result){
// write in the configuration file the row
      if (write)
	result= WriteRowFile (&FFile, "", StatFile, row, true);
    }
  } // for (RowNum= 0; RowNum < FcRows.NumConf && result; RowNum++)
  strcpy (StatFile, "END");
  if (!(WriteRowFile (&FFile, "", StatFile, row, false)))
    result= false;

  InitConf ("FIND");

  return (result);    
} // WriteFile

/**************************************************************************
 * Function: FileToConf
 * Details:  Read the file row and convert it on the group/key/value.
 * Returns: 
 **************************************************************************/
static int FileToConf (char *RowIn, char *grp, char *key, char *val, char *comm,
		       struct _StatRow *stat, char *IniGrp, char *IniRGrp){

  int result= true, LenSect, IniSect, EndSect, NewPos;
  int FlagLock, FlagGrp, FlagGrpEnd, FlagKey;
  char SepGKV[LENSTR + 1];
  char WRow[MAXLENROWS + 1], sect[MAXLENROWS + 1], token[MAXLENROWS + 1];
  char* pWRow;

  *grp= *key= *val= *comm= '\0';
  FlagLock= FlagGrp= FlagGrpEnd= FlagKey= false;
// set the type 'ignore' + unlock + unedit 
  WStatToRStat ("?UU", stat);
  strcpy (WRow, RowIn);
  trim (WRow);
// if one row comment
  if (WRow[0] == '#' && strncmp (WRow, "#!#", 3)){
// set the type 'ignore' (comment)
    stat->type= 'I';
    if (!strcpyV (val, WRow, LENGKV))
      result= false;
  }
// check and remove the lock group/key flag
  FlagLock= !strncmp (WRow, "#!#", 3);
  IniSect= FlagLock ? 3 : 0;
  strcpy (sect, &WRow[IniSect]);
  trim (sect);
// check for group
  if (result && stat->type == '?'){
    IniSect= 0;
    if (!ISTFILE("XFREE")){
      if (FlagGrp= sect[0] == '[')
	IniSect= 1;
    }
    else{
      NewPos= 0;
      if (FlagGrpEnd= !strncasecmp (sect, "End", 3))
	NewPos= 3;
      if (!strncasecmp (&sect[NewPos], "Sub", 3))
	NewPos += 3;
      if (!strncasecmp (&sect[NewPos], "Section", 7)){
	NewPos += 7;
	if (FlagGrp= sect[NewPos] == ' ' || sect[NewPos] == '\0')
	  IniSect= NewPos;
      }
    }
// if is one row group
    if (FlagGrp){
// set the type of the group
      stat->type= FlagLock ?  'g' : 'G';
// remove the flag before of the group
      trim (&sect[IniSect]);
// if is one row 'group end'
      if (sect[IniSect] == '/' && !ISTFILE("XFREE")){
	IniSect += 1;
	FlagGrpEnd= true;
      }
// set the group
      strcpy (WRow, &sect[IniSect]);
      if (FlagGrpEnd){
	stat->type= 'E';
	GetPartGrp (IniRGrp, grp, "LAST");
      }
      else{
	pWRow= WRow;
	strcpy (SepGKV, !ISTFILE("XFREE") ? "]" : " ");
	if (!strtoken (&pWRow, sect, SepGKV, MAXLENROWS))
	  result= false;
	if (!strcpyV (grp, sect, LENGKV))
	  result= false;
      }
// if is a row that signals the 'group lock'
      if (result && stat->type == 'g')
// set the lock of the group
	result= SetLockGrpKey (IniGrp, "", "", "L", false);
    } // if (FlagGrp){
  } // if (result && stat->type == '?'){
// check for key/value
  if (result && stat->type == '?'){
    strcpy (SepGKV, !ISTFILE("XFREE") ? "=" : " ");
    strcpy (WRow, sect);
    pWRow= WRow;
    if (!strtoken (&pWRow, token, SepGKV, MAXLENROWS))
      result= false;
    FlagKey= *token != '\0';
// if is one row key/value
    if (FlagKey){
      if (!strcpyV (key, token, LENGKV))
	result= false;
      if (!strtoken (&pWRow, token, "#", MAXLENROWS))
	result= false;
      if (!strcpyV (val, token, LLENGKV))
	result= false;
      LenSect= strlen (sect);
      EndSect= strnspnN (sect, "#", 0);
      if (EndSect < LenSect){
	if (!strcpyV (comm, &sect[EndSect], LENGKV))
	  result= false;
      }
      if (result){
// set the type of the key
	stat->type= FlagLock ?  'k' : 'K';
// if is a row that signals the 'key lock'
	if (stat->type == 'k')
// set the lock of the key
	  result= SetLockGrpKey (IniGrp, key, "", "L", false);
      } // result
    } // if (FlagKey)
  } // if (result && stat->type == '?'){
// check type not managed
  if (stat->type == '?'){
      stat->type= 'I';
      if (!strcpyV (val, WRow, LENGKV))
	result= false;
  }

  return (result);
} // FileToConf

/**************************************************************************
 * Function: ConfToFile
 * Details:  Read the group/key/value and convert its on the file row.
 * Returns: 
 **************************************************************************/
static int ConfToFile (char *RowOut, int FormEnab, struct _ConfRow *ActCRow, struct _ConfRow *OldDRow, int *write){

  int result, RowNGrp, RowNKey, NumNestGrp, InitSpace, LenRow;
  char FlagLock[LENSTR + 1], FlagGrp[LENSTR + 1], FlagDelim[LENSTR + 1];
  char ActGrp[LENGKV + 1], PartGrp[LENGKV + 1], EndPartGrp[LENGKV + 1];
  char str[LENSTR + 1], StrForm[LENSTR + 1], StrNextForm[LENSTR + 1];
  struct _StatRow RStat;

  *FlagLock= *FlagGrp= *EndPartGrp= '\0';

// enable write the values in the configuration file (disable delete group/key)
  if (!*write){
    if (!(*write= OldDRow == 0)){
      if (!(*write= !(ISGRP(OldDRow->stat.type)))){
// if is a same or parent group
	*write= !IsSameGroup (OldDRow->WorkGrp, ActCRow->WorkGrp);
      }
    }
  }
/* the row of signalling of the lock it goes written in the file only if the 
   group/key it is in lock */ 
  if (ActCRow->stat.type == 'g' || ActCRow->stat.type == 'k'){
    strcpy (FlagLock, "#!#");
    WStatToRStat ((char*)(ISGRP(ActCRow->stat.type) ? "GL" : "KL"), &RStat);
    FindGrpKey (ActCRow->WorkGrp, ActCRow->key, "", &RStat, &RowNGrp, &RowNKey);
    *write= *write && (ISGRP(ActCRow->stat.type) ? RowNGrp : RowNKey) > -1;
  } // if (ActCRow->stat.type == 'g' || ActCRow->stat.type == 'k')
// set the num of nested group and the initial space
  NumNestGrp= 0;
  strcpy (PartGrp, ActCRow->RealGrp);
  while (*PartGrp != '\0'){
    NumNestGrp++;
    strcpy (ActGrp, PartGrp);
    GetPartGrp (ActGrp, PartGrp, "PREVS");
  }
  InitSpace= 0;
  if (FormEnab){
    InitSpace= ISGRP(ActCRow->stat.type) ? NumNestGrp - 1 : NumNestGrp;
    InitSpace *= !ISTFILE("XFREE") ? 2 : 4;
  }
// set the format string and the max row lenght
  if (!(ISTFILE("XFREE") && ActCRow->stat.type == 'E'))
    GetPartGrp (ActCRow->RealGrp, EndPartGrp, "LAST");
  LenRow= 0;
  strcpy (FlagDelim, WithDelimRow (ActCRow->RealGrp, ActCRow->key, ActCRow->val, &ActCRow->stat) ? "\"" : "");
  if (ISGRP(ActCRow->stat.type)){
// if group is delete
    if (*write && ActCRow->stat.edit == 'D'){
      *write= false;
      OldDRow= ActCRow;
    }
// set the format of the group
    if (!ISTFILE("XFREE")){
      strcpy (FlagGrp, ActCRow->stat.type == 'E' ? "/" : "");
      sprintf (StrForm, "%%*s%s[%s%%s]", FlagLock, FlagGrp);
    }
    else{
      strcpy (FlagGrp, ActCRow->stat.type == 'E' ? "End" : "");
      strcat (FlagGrp, NumNestGrp > 1? "SubSection" : "Section");
      strcat (FlagGrp, ActCRow->stat.type == 'E' ? "" : " ");
      sprintf (StrForm, "%%*s%s%s%%s%s", FlagGrp, FlagDelim, FlagDelim);
    }
// set the new line for the group
    if (FormEnab){
      if (ActCRow->stat.type != 'E'){
      //	strcat (StrForm, "\n");  
      //      else if (ISTFILE("ONEGRP")){
	strcpy (str, StrForm);
	sprintf (StrForm, "\n%s", str);
      }
    }
    LenRow= InitSpace + strlen (StrForm) + strlen (EndPartGrp);
  }
  else if (ISKEY(ActCRow->stat.type)){
// if key is delete
    if (*write && ActCRow->stat.edit == 'D'){
      *write= false;
      OldDRow= ActCRow;
    }
// if key is changed in to comment
    sprintf (StrForm, "%%*s%s%s%%s", FlagLock, ActCRow->stat.edit == 'C' ? "# " : "");
// set the format of the key
    if (!ISTFILE("XFREE"))
      strcat (StrForm, "=");
    if (*ActCRow->val != '\0')
      strcat (StrForm, " ");
// set the format of after the key
    sprintf (StrNextForm, "%s%%s%s", FlagDelim, FlagDelim);
    strcat (StrForm, StrNextForm);
    strcat (StrForm, *ActCRow->comm == '\0' ? "%s" : " %s");
    LenRow= InitSpace + strlen (StrForm) + strlen (ActCRow->key) + strlen (ActCRow->val) + strlen (ActCRow->comm);
  }
  else{
// if comment is delete
    if (*write && ActCRow->stat.edit == 'D'){
      *write= false;
      OldDRow= ActCRow;
    }
// set the format of the 'ignore' (comment)
//    if (!ISTFILE("XFREE"))
      InitSpace= 0;
    strcpy (StrForm,  "%*s%s");
    LenRow= InitSpace + strlen (StrForm) + strlen (ActCRow->val);
  }
//set the row  
  if (result= LenRow <= MAXLENROWS){
    if (ISGRP(ActCRow->stat.type))
      sprintf (RowOut, StrForm, InitSpace, "", EndPartGrp);
    else if (ISKEY(ActCRow->stat.type))
      sprintf (RowOut, StrForm, InitSpace, "", ActCRow->key, ActCRow->val, ActCRow->comm);
    else
      sprintf (RowOut, StrForm, InitSpace, "", ActCRow->val);
  }
  else{
    sprintf (str, "%d", MAXLENROWS);
    ElAddErr (410, "FR", (char *)_("Adding %s %s to row exceeds %s"), ISGRP(ActCRow->stat.type) ? EndPartGrp : ActCRow->key, 
	      ActCRow->val, str);
  }

  return (result);
} // ConfToFile

/**************************************************************************
 * Function: WithDelimRow
 * Details:  Return if row is with delimitator
 * Returns: 
 **************************************************************************/
static int WithDelimRow (char *grp, char *key, char *val, struct _StatRow *stat){

  int result= false;

  if (ISTFILE("XFREE")){
    result= true;
    if (ISKEY(stat->type)){
      if (strstr (val, "\"") != NULL)
	result= false;
      if (!strcmp (key, "AllowMouseOpenFail") || !strcmp (key, "HorizSync") || 
	  !strcmp (key, "VertRefresh") || !strcmp (key, "VideoRam") || 
	  !strcmp (key, "DefaultColorDepth") || !strcmp (key, "DefaultDepth") 
	  || !strcmp (key, "Depth") || !strcmp (key, "Mode") || !strcmp (key, "DisplaySize") 
	  || !strcmp (key, "Viewport") || !strcmp (key, "Group"))
	result= false;
      if (!strcmp (grp, "Device") && !strcmp (key, "Screen"))
	result= false;
    }
    else if (stat->type == 'E')
      result= false;
  } // if (ISTFILE("XFREE"))

  return (result);
} // WithDelimRow

/**************************************************************************
 * Function: AddTextConf
 * Details:  Add the text of configuration file in the FcRows.text buffer.
 * Returns: 
 **************************************************************************/
static int AddTextConf (char *TextIn, char **TextAdd, int EnabLong){

  int result, AttNumText, MaxLenText, MaxNumText;
  char *AttGcRow, str[LENSTR + 1];

  MaxLenText= LENGKV;
  MaxNumText= NUMTXTROW;
  AttNumText= FcRows.NumText;
  if (EnabLong && strlen (TextIn) > MaxLenText){
    MaxLenText= LLENGKV;
    MaxNumText= NUMLTXTROW;
    AttNumText= FcRows.NumLongText;
  }

  if (result= AttNumText < MaxNumText){
    AttGcRow= MaxLenText > LENGKV ? FcRows.LongText[AttNumText] : FcRows.text[AttNumText];
    result= strcpyV (AttGcRow, TextIn, MaxLenText);
    *TextAdd= AttGcRow;
    MaxLenText > LENGKV ? FcRows.NumLongText++ : FcRows.NumText++;
  }
  else{
    sprintf(str, "%d", MaxNumText);
    ElAddErr2 (411, "FR", (char *)_("With %s the number of row exceeds %s"), TextIn, str);
  }
  return (result);
} // AddTextConf

/**************************************************************************
 * Function: WStatToRStat
 * Details:  Convert the work status in the row status.
 * Returns: 
 **************************************************************************/
static void WStatToRStat (char *WStat, struct _StatRow *stat){
  
  char FillWStat [LENSSTAT + 1];

  strfromstr (FillWStat, WStat, 1, LENSSTAT);
  stat->type= FillWStat[STYPE];
  stat->lock= FillWStat[SLOCK];
  stat->edit= FillWStat[SEDIT];
} // WStatToRStat

/**************************************************************************
 * Function: WStatFormRStat
 * Details:  Convert the row status in the work status.
 * Returns: 
 **************************************************************************/
static void WStatFromRStat (char *WStat, struct _StatRow *stat){

  WStat[STYPE]= stat->type;
  WStat[SLOCK]= stat->lock;
  WStat[SEDIT]= stat->edit;
  WStat[LENSSTAT]= '\0';

} // WStatFromRStat

/**************************************************************************
 * Function: InitConf
 * Details:  Initialize the parameters of the configuration.  
 * Returns: 
 **************************************************************************/
void InitConf (char *tipo){

  int RowNum;

  if (!strcmp (tipo, "CONF")){
    ReadEnab= true;
    SortEnab= true;
    SaveEnab= false;

    FcRows.NumText= FcRows.NumLongText= FcRows.NumConf= 0;
    for (RowNum= 0; RowNum < MAX_NCONFROWS; RowNum++)
      FcRows.SortConf[RowNum]= &FcRows.conf[RowNum];
  }
  else if (!strcmp (tipo, "FIND")){
    FindRowNum= -1;
  }
  else if (!strcmp (tipo, "OPT")){
    ConfPar.OneKeyEnab= false;
    strcpy (ConfPar.VerbLev, "3");
    *ConfPar.FileType= *ConfPar.ConfFile= *ConfPar.WarnType= *ConfPar.ParSep= *ConfPar.ValSep= '\0';
    ConfPar.pfExeConfVal= NULL;
  }
  else
    ElAddErr1 (401, "BV", (char *)_("The init %s it is not managed"), tipo);     
} //InitConf
