/* -*- c++ -*- (enables emacs c++ mode)                                    */
/* *********************************************************************** */
/*                                                                         */
/* Library :  GEneric Tool for Finite Element Methods (getfem)             */
/* File    :  getfem_norm.h: computes various norms on pde solutions.      */
/*     									   */
/* Date : November 17, 2000.                                               */
/* Authors : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                     */
/*           Julien Pommier, pommier@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */
 

#ifndef __GETFEM_NORM_H
#define __GETFEM_NORM_H

#include <getfem_mesh_fem.h>

namespace getfem
{
  /* ********************************************************************* */
  /*                                                                       */
  /*  Calcul de normes L2 et H1.                                           */
  /*                                                                       */
  /* ********************************************************************* */


  template<class MESH_FEM, class VECT>
    scalar_type L2_norm(MESH_FEM &mf, const VECT &U, size_type N)
  { /* optimisable */
    size_type cv;
    scalar_type no = 0.0;
    dal::bit_vector nn = mf.convex_index();
    dal::dynamic_array<base_vector, 2> vval;
    base_tensor t;
    pfem pf1, pf1prec = NULL;
    pintegration_method pim, pimprec = (bgeot::papprox_integration)(NULL);

    bgeot::pgeometric_trans pgt, pgtprec = NULL;
    pmat_elem_type pme; pmat_elem_computation pmec;
    
    for (cv << nn; cv != ST_NIL; cv << nn)
    {
      pf1 =     mf.fem_of_element(cv);
      pgt = mf.linked_mesh().trans_of_convex(cv);
      pim = mf.int_method_of_element(cv);
      size_type nbd = mf.nb_dof_of_element(cv);
      if (pf1prec != pf1 || pgtprec != pgt || pimprec != pim)
      {
	pme = mat_elem_product(mat_elem_base(pf1), mat_elem_base(pf1));
	pmec = mat_elem(pme, pim, pgt);
	pf1prec = pf1; pgtprec = pgt; pimprec = pim;
      }
      pmec->gen_compute(t, mf.linked_mesh().points_of_convex(cv));
      base_tensor::iterator p = t.begin();

      for (size_type i = 0; i < nbd; i++)
      { 
	size_type dof1 = mf.ind_dof_of_element(cv)[i];
	if (vval[i].size() != N) vval[i] = base_vector(N); 
	for (size_type k = 0; k < N; k++) (vval[i])[k] = U[dof1*N+k];
      }

      for (size_type i = 0; i < nbd; i++)
	for (size_type j = 0; j < nbd; j++, ++p)
	  no += bgeot::vect_sp(vval[i], vval[j]) * (*p);
      
    }
    return sqrt(no);
  }

  template<class MESH_FEM, class VECT>
    scalar_type H1_semi_norm(MESH_FEM &mf, const VECT &U, size_type N)
  { /* optimisable */
    size_type cv, NN = mf.linked_mesh().dim();
    scalar_type no = 0.0;
    dal::bit_vector nn = mf.convex_index();
    dal::dynamic_array<base_vector, 2> vval;
    base_tensor t;
    pfem pf1, pf1prec = NULL;
    pintegration_method pim, pimprec = (bgeot::papprox_integration)(NULL);
    bgeot::pgeometric_trans pgt, pgtprec = NULL;
    pmat_elem_type pme; pmat_elem_computation pmec;

    for (cv << nn; cv != ST_NIL; cv << nn)
    {
      pf1 =     mf.fem_of_element(cv);
      pgt = mf.linked_mesh().trans_of_convex(cv);
      pim = mf.int_method_of_element(cv);
      size_type nbd = mf.nb_dof_of_element(cv);
      if (pf1prec != pf1 || pgtprec != pgt || pimprec != pim)
      {
	pme = mat_elem_product(mat_elem_grad(pf1), mat_elem_grad(pf1));
	pmec = mat_elem(pme, pim, pgt);
	pf1prec = pf1; pgtprec = pgt; pimprec = pim;
      }
      pmec->gen_compute(t, mf.linked_mesh().points_of_convex(cv));
      base_tensor::iterator p = t.begin();
      for (size_type i = 0; i < nbd; i++)
      { 
	size_type dof1 = mf.ind_dof_of_element(cv)[i];
	if (vval[i].size() != N) vval[i] = base_vector(N); 
	for (size_type k = 0; k < N; k++) (vval[i])[k] = U[dof1*N+k];
      }
      for (size_type l = 0; l < NN; l++)
	for (size_type i = 0; i < nbd; i++)
	  for (size_type k = 0; k < NN; k++)
	    for (size_type j = 0; j < nbd; j++, ++p)
	      if (k == l)
		no += (*p) * bgeot::vect_sp(vval[i], vval[j]);
    }
    return sqrt(no);
  }

  template<class MESH_FEM, class VECT>
    scalar_type H1_norm(MESH_FEM &mf, const VECT &U, size_type N) {
      return sqrt( dal::sqr(L2_norm(mf, U, N)) 
		   + dal::sqr(H1_semi_norm(mf, U, N)));
    }
}
#endif 
