/* *********************************************************************** */
/*                                                                         */
/* Library :  Dynamic Array Library (dal)                                  */
/* File    :  dal_algobase.h : generic algorithms on containers.           */
/*     									   */
/*                                                                         */
/* Date : September 28, 2000.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */


#ifndef __DAL_ALGOBASE_H
#define __DAL_ALGOBASE_H


#include <dal_std.h>
#include <functional>

namespace dal
{

  /* ********************************************************************* */
  /* Definitition de classes de comparaison.                               */
  /* retournant un int.                                                    */
  /* ********************************************************************* */
  
  template <class T>
    struct less : public std::binary_function<T, T, int>
  {
    inline int operator()(const T& x, const T& y) const
    { return (x < y) ? -1 : ((y < x) ? 1 : 0); }
  };

  template<> struct less<int> : public std::binary_function<int, int, int>
  { int operator()(int x, int y) const { return x-y; } };
  template<> struct less<char> : public std::binary_function<char, char, int>
  { int operator()(char x, char y) const { return int(x-y); } };
  template<> struct less<short> : public std::binary_function<short,short,int>
  { int operator()(short x, short y) const { return int(x-y); } };
  template<> struct less<unsigned char>
     : public std::binary_function<unsigned char, unsigned char, int>
  {
    int operator()(unsigned char x, unsigned char y) const
    { return int(x)-int(y); }
  };
  

  template <class T>
    struct greater : public std::binary_function<T, T, int>
  {
    inline int operator()(const T& x, const T& y) const
    { return (y < x) ? -1 : ((x < y) ? 1 : 0); }
  };

  template<> struct greater<int> : public std::binary_function<int, int, int>
  { int operator()(int x, int y) const { return y-x; } };
  template<> struct greater<char> : public std::binary_function<char,char,int>
  { int operator()(char x, char y) const { return int(y-x); } };
  template<> struct greater<short>
      : public std::binary_function<short, short, int>
  { int operator()(short x, short y) const { return int(y-x); } };
  template<> struct greater<unsigned char>
      : public std::binary_function<unsigned char, unsigned char, int>
  { 
    int operator()(unsigned char x, unsigned char y) const
      { return int(y)-int(x); }
  };

  template <class T>
    struct approx_less : public std::binary_function<T, T, int>
  { 
    double eps;
    inline int operator()(const T &x, const T &y) const
    { if (dal::abs(x - y) < eps) return 0; if (x < y) return -1; return 1; }
    approx_less(double e = 1E-13) { eps = e; }
  };

  template <class T>
    struct approx_greater : public std::binary_function<T, T, int>
  { 
    double eps;
    inline int operator()(const T &x, const T &y) const
    { if (dal::abs(x - y) < eps) return 0; if (x > y) return -1; return 1; }
    approx_greater(double e = 1E-13) { eps = e; }
  };

  template<class ITER1, class ITER2, class COMP>
    int lexicographical_compare(ITER1 b1, const ITER1 &e1,
				ITER2 b2, const ITER2 &e2, const COMP &c) 
  {
    int i;
    for ( ; b1 != e1 && b2 != e2; ++b1, ++b2)
      if ((i = c(*b1, *b2)) != 0) return i;
    if (b1 != e1) return 1; if (b2 != e2) return -1; return 0; 
  }

  template<class CONT, class COMP = dal::less<typename CONT::value_type> >
    struct lexicographical_less : public std::binary_function<CONT, CONT, int>
  { 
    COMP c;
    int operator()(const CONT &x, const CONT &y) const
    { return dal::lexicographical_compare(x.begin(), x.end(),
				     y.begin(), y.end(), c);
    }
    lexicographical_less(const COMP &d = COMP()) { c = d; }
  };

  template<class CONT, class COMP = dal::less<typename CONT::value_type> >
    struct lexicographical_greater
                                : public std::binary_function<CONT, CONT, int>
  { 
    COMP c;
    int operator()(const CONT &x, const CONT &y) const
    { return -dal::lexicographical_compare(x.begin(), x.end(),
				      y.begin(), y.end(), c);
    }
    lexicographical_greater(const COMP &d = COMP()) { c = d; }
  };


  /* ********************************************************************* */
  /* "Virtual" iterators on sequences.                                     */
  /* The class T represent a class of sequence.                            */
  /* ********************************************************************* */

  template<class T> struct sequence_iterator
  {
    typedef T             value_type;
    typedef value_type*   pointer;
    typedef value_type&   reference;
    typedef const value_type& const_reference;
    typedef std::forward_iterator_tag iterator_category;

    T Un;

    sequence_iterator(T U0 = T(0)) { Un = U0; }
    
    sequence_iterator &operator ++()
    { ++Un; return *this; }
    sequence_iterator operator ++(int)
    { sequence_iterator tmp = *this; (*this)++; return tmp; }
	
    const_reference operator *() const { return Un; }
    reference operator *() { return Un; }
    
    bool operator ==(const sequence_iterator &i) const { return (i.Un==Un);}
    bool operator !=(const sequence_iterator &i) const { return (i.Un!=Un);}
  };

  /* ********************************************************************* */
  /* generic algorithms.                                                   */
  /* ********************************************************************* */

  template <class ITER1, class SIZE, class ITER2>
    void copy_n(ITER1 first, SIZE count, ITER2 result)
  { for ( ; count > 0; --count) { *result = *first; ++first; ++result; } }

  template<class ITER>
    typename std::iterator_traits<ITER>::value_type
      mean_value(ITER first, const ITER &last)
  {
    typename std::iterator_traits<ITER>::value_type res;
    size_t n = 1;
    if (first != last) { res = *first; ++first; }
    while (first != last) { res += *first; ++first; ++n; }
    res /= n;
    return res;
  }

  template<class ITER> /* hum ... */
    void minmax_box(typename std::iterator_traits<ITER>::value_type &pmin,
		    typename std::iterator_traits<ITER>::value_type &pmax,
		    ITER first, const ITER &last)
  {
    typedef typename std::iterator_traits<ITER>::value_type PT;
    if (first != last) { pmin = pmax = *first; ++first; }
    while (first != last)
    {
      typename PT::const_iterator b = (*first).begin(), e = (*first).end();
      typename PT::iterator b1 = pmin.begin(), b2 = pmax.begin();
      while (b != e)
      { *b1 = std::min(*b1, *b); *b2 = std::max(*b2, *b); ++b; ++b1; ++b2; }
    }
  }

} /* end of namespace dal.                                                 */



#endif /* __DAL_ALGOBASE_H */
