#define GETFEMINT_WORKSPACE_C

#include <dal_singleton.h>
#include <bgeot_config.h>
#include <getfemint_workspace.h>

namespace getfemint
{
  workspace_stack& workspace() { 
    return dal::singleton<workspace_stack>::instance();
  }

  void workspace_stack::set_dependance(getfem_object *user, getfem_object *used)
  {
    used->used_by.push_back(user->get_id());
  }

  /* check if object 'id' can be deleted 
     (all objects which depend on it should be marked as anonymous)
  */
  bool workspace_stack::is_deletable_object(id_type id, dal::bit_vector& v) const {
    const getfem_object *o = obj[id];  

    //mexPrintf("is_deletable: id=%d, o = %p\n", id, o);
    if (!o) THROW_INTERNAL_ERROR;
    if (!o->is_anonymous()) return false; /* this object was never requested for deletion */
    
    v[id] = true;
    
    /* look for every object which uses object 'id' */
    for (std::vector<id_type>::const_iterator it = o->used_by.begin();
	 it != o->used_by.end(); it++) {
      /* if it has not been marqued for delete */
      if (!v[*it]) {
	/* mark it and try to find if it is deletable */
	if (!is_deletable_object(*it, v)) return false;
      }
    }
    return true;
  }

  /* inserts a new object (and gives it an id) */
  id_type workspace_stack::push_object(getfem_object *o) {
    id_type obj_id = obj.add(o);
    o->set_workspace(current_workspace);
    o->set_id(obj_id);
    newly_created_objects.push_back(obj_id);
    return obj_id;
  }

  /* at least mark the objet for future deletion (object becomes anonymous)
     and if possible, destroy the object (and all the objects which use this one
     if they are all anonymous) */
  void workspace_stack::delete_object(id_type id) {
    if (obj.index()[id]) {
      getfem_object *o = obj[id]; 
      //cerr << "delete_object requested: id=" << id << ", type = " << name_of_getfemint_class_id(o->class_id()) << "\n";

      if (!o) THROW_INTERNAL_ERROR;

      /* mark the object as an anonymous one */
      o->set_workspace(anonymous_workspace);
      
      /* list of objects to delete */
      dal::bit_vector dellst; 
      
      /* check if the object is deletable */
      if (is_deletable_object(id, dellst)) {
	//	mexPrintf("object %d is deletable (%d objects will be deleted)\n", id, dellst.card());
	/* suppress the dependancies on an object which is on the deletion list */
	for (obj_ct::tas_iterator it = obj.tas_begin();
	     it != obj.tas_end(); it++) {
	  int cnt = 0;
	  for (unsigned i = 0; i < (*it)->used_by.size(); i++) {
	    if (dellst[(*it)->used_by[i]]) {
	      (*it)->used_by[i] = workspace_stack::invalid_id;
	      cnt ++;
	    }
	  }
	  if (cnt) {
	    int j = 0;
	    for (unsigned i = 0; i < (*it)->used_by.size(); i++) {
	      if ((*it)->used_by[i] != workspace_stack::invalid_id) {
		(*it)->used_by[j] = (*it)->used_by[i]; j++;
	      }
	    }
	    (*it)->used_by.resize(j);

	    /* anonymous objects floating around which are no used by anyone 
	       are added to the deletion list */
	    if (j == 0 && (*it)->is_anonymous()) dellst.add(it.index());
	  }
	}
        //if (dellst.card()) cerr << " dellst = " << dellst << "\n";
        /* clear each deletable objects. After the clear there should not__ be anymore dependance
           between them (i.e. we can delete them in any order)
        */
	for (dal::bv_visitor n(dellst); !n.finished(); ++n) obj[n]->clear();
        /* kill them all */
	for (dal::bv_visitor n(dellst); !n.finished(); ++n) {
	  //cerr << "suppressing object " << o->get_id() << "(" << name_of_getfemint_class_id(o->class_id()) << ")\n";
	  delete obj[n]; 
	  //cerr << "object successfully deleted -- obj.index()[n]=" << obj.index()[n] << "\n";
	  obj.sup(n);
	}
        dellst.clear();
      }
    } else {
      std::stringstream s;
      s << "object number " << id << " no longer exists : can't delete it";
      throw getfemint_error(s.str());
    }
  }

  /* create a new workspace on top of the stack */
  void workspace_stack::push_workspace(std::string n) { 
    id_type new_workspace = wrk.add(workspace_data(n, current_workspace));
    current_workspace = new_workspace;
  }

  /* move the object in the parent workspace, in order to prevent
     the object from being deleted when the current workspace will
     be 'poped' */
  void workspace_stack::send_object_to_parent_workspace(id_type obj_id) {
    getfem_object *o = obj[obj_id];
    if (!o) { THROW_ERROR("this object does not exist\n"); }
    if (!wrk.index()[o->get_workspace()]) THROW_INTERNAL_ERROR;
    
    o->set_workspace(wrk[current_workspace].parent_workspace);
  }

  void workspace_stack::send_all_objects_to_parent_workspace() {
    for (obj_ct::tas_iterator it = obj.tas_begin(); 
	 it != obj.tas_end(); ++it) {
      if ((*it)->get_workspace() == current_workspace) {
	(*it)->set_workspace(wrk[current_workspace].parent_workspace);
      }
    }
  }
  /* delete every object in the workspace, but *does not* delete the workspace itself */
  void workspace_stack::clear_workspace(id_type wid) {
    if (wid == anonymous_workspace) THROW_INTERNAL_ERROR;
    for (dal::bv_visitor oid(obj.index()); !oid.finished(); ++oid) {
      id_type owid = obj[oid]->get_workspace();
      if (owid != anonymous_workspace && !wrk.index_valid(owid))
	THROW_INTERNAL_ERROR;
      if (owid == wid) {
	delete_object(oid);
      }
    }
  }

  /* deletes the current workspace and returns to the parent workspace */
  void workspace_stack::pop_workspace(bool keep_all) {
    if (!wrk.index()[current_workspace]) THROW_INTERNAL_ERROR;
    if (current_workspace == base_workspace) THROW_INTERNAL_ERROR;
    
    if (keep_all) send_all_objects_to_parent_workspace();
    else clear_workspace();
    id_type tmp = current_workspace;
    current_workspace = wrk[current_workspace].parent_workspace;
    wrk.sup(tmp);
  }

  getfem_object* workspace_stack::object(id_type id, const char *expected_type) {
    getfem_object *o = NULL;
    //cout << "obj.index() == " << obj.index() << ", id= " << id << "\n";
    if (obj.index()[id] &&
	std::find(newly_created_objects.begin(), newly_created_objects.end(),id) == newly_created_objects.end()) {
      o = obj[id]; 
      if (!o) THROW_INTERNAL_ERROR;
    } else {
      THROW_ERROR("object " << expected_type  << " [id=" << id << "] not found");
    }
    return o;
  }

  void workspace_stack::commit_newly_created_objects() { 
    newly_created_objects.resize(0); 
  }

  void workspace_stack::destroy_newly_created_objects() { 
    while (newly_created_objects.size()) {
      delete_object(newly_created_objects.back()); 
      newly_created_objects.pop_back();
    }      
  }

}
