#include <getfemint.h>
#include <getfemint_workspace.h>
#include <getfemint_mesh.h>
#include <getfem_import.h>
/*
  $Id: gf_mesh.C,v 1.11 2004/08/18 10:16:12 pommier Exp $

  ChangeLog:
  $Log: gf_mesh.C,v $
  Revision 1.11  2004/08/18 10:16:12  pommier
  many updates

  Revision 1.10  2004/06/21 09:03:46  pommier
  more work on gf_spmat and python interface

  Revision 1.9  2004/06/11 10:10:28  pommier
  commit recent work: handling of integer and complex arrays, interface to gmm preconditioners and sparse matrices

  Revision 1.8  2004/03/03 16:04:05  pommier
  renamed matlabint_* to getfemint_*

  Revision 1.7  2004/01/13 16:02:51  pommier
  Fixes for dec/alpha ..

  Revision 1.6  2004/01/12 17:54:30  pommier
  Minor fixes.

  Revision 1.5  2003/10/05 16:03:40  pommier
  change all bit_vector loops to bv_visitor

  Revision 1.4  2003/10/04 10:56:08  pommier
  *** empty log message ***

  Revision 1.3  2003/07/25 09:04:30  pommier
  *** empty log message ***

  Revision 1.2  2003/07/23 18:31:14  pommier
  *** empty log message ***

  Revision 1.1  2003/05/22 13:18:03  pommier
  regroupement de tous les fichiers dans ./src , et mise en place des RPC

  Revision 1.20  2003/05/05 07:59:10  pommier
  ajout gf_slice , gf_slice_get

  Revision 1.19  2003/02/27 16:34:17  pommier
  improve friendlyness with gcc2.95

  Revision 1.18  2003/02/17 11:44:15  pommier
  doc updates

  Revision 1.17  2003/02/14 13:02:26  pommier
  divers modifs

  Revision 1.16  2003/02/11 11:01:45  pommier
  ajout des interfaces oo matlab

  Revision 1.15  2003/02/04 10:33:30  pommier
  modifs diverses pour les triangles/tetra avec trans geo non lin

  Revision 1.14  2003/01/30 16:11:00  pommier
  *** empty log message ***

  Revision 1.13  2003/01/28 13:50:58  pommier
  updates sgi + update eval_on_P1_tri_mesh

 */
using namespace getfemint;

static void
cartesian_mesh(getfem::getfem_mesh *pmesh, getfemint::mexargs_in &in)
{
  getfemint::size_type dim = in.remaining();
  
  if (dim == 0) THROW_BADARG( "not enough input arguments");

  std::vector<darray> ppos(dim);
  std::vector<size_type> npts(dim);
  dal::uint32_type grid_npoints=1, grid_nconvex=1;
  for (size_type i = 0; i < dim; i++) {
    ppos[i] = in.pop().to_darray();
    npts[i] = ppos[i].size();
    grid_npoints *= npts[i];
    grid_nconvex *= (npts[i]-1);
  }
  
  /* add the points in 'fortran style' order */
  getfem::base_node pt(dim);
  for (dal::uint32_type i=0; i < grid_npoints; i++) {
    dal::uint32_type k = i;    
    for (getfemint::size_type j = 0; j < dim; j++) {
      pt[j] = ppos[j][k % (npts[j])];
      k /= (npts[j]);
    }

    size_type id_pt = pmesh->add_point(pt);
    if (id_pt != i) {
      THROW_ERROR( 
		"something has changed in getfem, you need to reconsider "
		"gf_mesh('cartesian')\nfor point " << i << 
		", the index is " << id_pt << endl);
    }
    /*    if (i == grid_npoints-1) {
      cerr << "nb de pts ajoutes: " << grid_npoints << " id dernier =" << id_pt << endl;
      }*/
  }
  

  std::vector<int> ipt(dim);
  std::vector<getfem::base_node> pts(1 << dim+1);
  
  bgeot::pgeometric_trans pgt = bgeot::parallelepiped_linear_geotrans(dim);

  /* add the convexes */
  for (dal::uint32_type i=0; i < grid_nconvex; i++) {
    dal::uint32_type k = i;

    /* find point location */
    for (getfemint::size_type j = 0; j < dim; j++) {
      ipt[j] = k % (npts[j]-1);
      k /= (npts[j]-1);
    }

    /* build the vertices list */
    for (getfemint::size_type j = 0; j < (unsigned(1)<<dim); j++) {
      pts[j].resize(dim);
      for (dal::uint32_type d=0; d < dim; d++) {
	if ((j >> d) & 1) {
	  pts[j][d] = ppos[d][ipt[d]+1];
	} else {
	  pts[j][d] = ppos[d][ipt[d]];
	}
      }
    }

    // we don't use the add_parall since the geometric transformatoin
    // is linear (the mesh is cartesian)
    //pmesh->add_parallelepiped_by_points(dim, pts.begin());
    pmesh->add_convex_by_points(pgt, pts.begin()); 
  }
}


static void
triangles_grid_mesh(getfem::getfem_mesh *pmesh, getfemint::mexargs_in &in)
{
  if (in.remaining() != 2) THROW_BADARG( "not enough input arguments");
  
  darray X = in.pop().to_darray();
  darray Y = in.pop().to_darray();
  if (X.size() < 1 || Y.size() < 1) THROW_BADARG( "bad dimensions");

  size_type ni = Y.size(), nj = X.size();
  for (size_type i=0; i < ni; i++) {
    for (size_type j=0; j < nj; j++) {
      getfem::base_node pt(2);
      pt[0] = X[j]; pt[1] = Y[i];
      //      cerr << "pt = " << pt << endl;
      pmesh->add_point(pt);
    }
  }
  for (size_type i=0; i < ni-1; i++) {
    for (size_type j=0; j < nj-1; j++) {
      //cerr << "i=" << i <<" j=" << j << endl;
      pmesh->add_triangle(i*nj + j, (i+1)*nj + j  , (i+1)*nj + j+1);
      pmesh->add_triangle(i*nj + j,     i*nj + j+1, (i+1)*nj + j+1);
    }
  }
}

static void
curved_mesh(getfem::getfem_mesh *dest_mesh, getfemint::mexargs_in &in)
{
  const getfem::getfem_mesh *src_mesh = in.pop().to_const_mesh();
  darray F = in.pop().to_darray(src_mesh->points().index().last()+1);

  int dim = src_mesh->dim();
  bgeot::base_node pt(dim+1);
  dest_mesh->clear();
  for (dal::bv_visitor i(src_mesh->points().index()); !i.finished(); ++i) {
    std::copy(src_mesh->points()[i].begin(), src_mesh->points()[i].end(), pt.begin());
    pt[dim] = F[i];
    size_type k = dest_mesh->add_point(pt);
    if (k != i) dest_mesh->swap_points(i,k); /* ca meriterait d'etre teste sur un maillage a trous ca.. */
  }

  for (dal::bv_visitor cv(src_mesh->convex_index()); !cv.finished(); ++cv) {
    dest_mesh->add_convex(src_mesh->trans_of_convex(cv),
			  src_mesh->ind_points_of_convex(cv).begin());
  }
}

static void
prismatic_mesh(getfem::getfem_mesh *dest_mesh, getfemint::mexargs_in &in)
{
  const getfem::getfem_mesh *src_mesh = in.pop().to_const_mesh();
  unsigned nblay = in.pop().to_integer(0,2500000);
  int dim = src_mesh->dim();
  bgeot::base_node pt(dim+1);

  dest_mesh->clear();
  bgeot::size_type nbpt = src_mesh->points().index().last()+1;
  if (nbpt != src_mesh->points().index().card()) 
    THROW_ERROR("please optimize the mesh before using it as a base for prismatic mesh");
  for (size_type i = 0; i < nbpt; ++i)
  {
    std::copy(src_mesh->points()[i].begin(), src_mesh->points()[i].end(),pt.begin());
    pt[dim] = 0.0;
    for (size_type j = 0; j <= nblay; ++j, pt[dim] += 1.0 / nblay)
      dest_mesh->add_point(pt);
  }
  
  std::vector<bgeot::size_type> tab(5);
  for (dal::bv_visitor cv(src_mesh->convex_index()); !cv.finished(); ++cv) {
    size_type nbp = src_mesh->nb_points_of_convex(cv);
    if (2*nbp > tab.size()) tab.resize(2*nbp);
    for (size_type j = 0; j < nblay; ++j)
    {
      for (size_type k = 0; k < nbp; ++k)
	tab[k] = (nblay+1)*src_mesh->ind_points_of_convex(cv)[k] + j;
      for (size_type k = 0; k < nbp; ++k)
	tab[k+nbp] = (nblay+1)*src_mesh->ind_points_of_convex(cv)[k] + j + 1;
      bgeot::pgeometric_trans pgt = 
	//bgeot::product_geotrans(src_mesh->trans_of_convex(i), bgeot::simplex_geotrans(1,1));
      
	bgeot::geometric_trans_descriptor("GT_LINEAR_PRODUCT(" +
					  bgeot::name_of_geometric_trans(src_mesh->trans_of_convex(cv)) +
					  ",GT_PK(1,1))");
      
      dest_mesh->add_convex(pgt, tab.begin());
      
      /*
      bgeot::convex_product_trans(
		         src_mesh->trans_of_convex(i),
			 bgeot::simplex_trans(1, 1)), tab.begin());
      */
    }
  }
}

static void
ptND_mesh(getfem::getfem_mesh *mesh, bool is2D, getfemint::mexargs_in &in)
{
  darray P = in.pop().to_darray(-1, -1);
  iarray T = in.pop().to_iarray(-1, -1);
  cout << "T(" << T.getm() << ", " << T.getn() << "), size=" << T.size() << "\n";
  size_type mdim = P.getm();
  size_type N = is2D ? 2 : T.getm() - 1;
  if (is2D && T.getm() != 3 && T.getm() != 4) {
    THROW_BADARG("wrong nb of rows for t, 3 or 4 rows were expected, got " << T.getm());
  } else if (T.getm() < 1 || N > 10) {
    THROW_BADARG("wrong nb of rows for t (dim = 0 or dim > 10)");
  }
  if (mdim == 0 || mdim < N) {
    THROW_BADARG("cannot build simplexes of dimension " << N << " with points of dimension " << mdim);
  }
  id_type zone = 0;
  if (in.remaining()) zone = in.pop().to_integer(1,65000);

  size_type warn_cnt = 0;
  std::vector<id_type> id_tab(P.getn());
  for (size_type i = 0; i < P.getn(); ++i) {
    id_tab[i] = mesh->add_point(P.col_to_bn(i));

    /* une hypothse bien commode pour la "compatibilit pdetool" */
    if (id_tab[i] != i && warn_cnt++ == 0) {
      DAL_WARNING(1, "The numbering of mesh points will be different, pt#" << 
                  i+config::base_index() << " gets id#" << id_tab[i] + config::base_index());
    }
  }

  std::vector<size_type> ipts(N+1);
  for (size_type i = 0; i < T.getn(); ++i) {
    for (size_type k = 0; k < N+1; ++k) {
      ipts[k] = T(k,i) - config::base_index();
      if (ipts[k] >= P.size()) THROW_BADARG( "Bad triangulation.");
    }
    if (zone == 0 || (T.getm() == N+2 && zone == id_type(T(N+1,i))))
      mesh->add_convex(bgeot::simplex_geotrans(N,1), dal::index_ref_iterator(id_tab.begin(), ipts.begin()));
  }
}

/*MLABCOM

  FUNCTION M = gf_mesh([operation [, args]])
  General constructor for mesh object. Returns a getfem handle to the newly
  created mesh object.
  
  @INIT MESH:INIT ('empty')
  @INIT MESH:INIT ('cartesian')
  @INIT MESH:INIT ('triangles grid')
  @INIT MESH:INIT ('curved')
  @INIT MESH:INIT ('prismatic')
  @INIT MESH:INIT ('pt2D')

  @INIT MESH:INIT ('ptND')
  @INIT MESH:INIT ('load')
  @INIT MESH:INIT ('from string')
  @INIT MESH:INIT ('import')
  @INIT MESH:INIT ('clone')
 
  $Id: gf_mesh.C,v 1.11 2004/08/18 10:16:12 pommier Exp $
MLABCOM*/
void gf_mesh(getfemint::mexargs_in& in, getfemint::mexargs_out& out)
{
  getfemint_mesh *mi_mesh = new getfemint_mesh();
  out.pop().from_object_id(workspace().push_object(mi_mesh), MESH_CLASS_ID);
  getfem::getfem_mesh *pmesh = &mi_mesh->mesh();


  if (in.narg() < 1) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  std::string cmd    = in.pop().to_string();
  if (check_cmd(cmd, "empty", in, out, 1, 1, 0, 1)) {
    /*@INIT  M=MESH:INIT('empty', @int dim)
      Create a new empty mesh.
      @*/
    size_type dim = in.pop().to_integer(1,255);
    getfem::base_node pt(dim); pt.fill(0.);
    /* just to initialize the dimension of the mesh
       (this is not very nice, i know) */
    pmesh->sup_point(pmesh->add_point(pt)); 
  } else if (check_cmd(cmd, "cartesian", in, out, 1, 32, 0, 1)) {
    /*@INIT  M=MESH:INIT('cartesian', @dvec X[, @dvec Y[, @dvec Z,..]])
      Build quickly a cartesian mesh.
      @*/
    cartesian_mesh(pmesh, in);
  } else if (check_cmd(cmd, "triangles grid", in, out, 2, 2, 0, 1)) {
    /*@INIT  M=MESH:INIT('triangles grid', @dvec X[, @dvec Y[, @dvec Z,..]])
      Build quickly a regular mesh of triangles. See also MESH:INIT('ptND').
      @*/
    triangles_grid_mesh(pmesh, in);
  } else if (check_cmd(cmd, "curved", in, out, 2, 2, 0, 1)) {
    /*@INIT  M=MESH:INIT('curved', M0, @dvec F)
      Build a curved (n+1)-dimensions mesh from a n-dimensions mesh M0. 

      The points of the new mesh have one additional coordinate, given by
      the vector F. This can be used to obtain meshes for shells. M0 may
      be a @tmf object, in that case its linked mesh will be used. 
      @*/
    curved_mesh(pmesh, in);
  } else if (check_cmd(cmd, "prismatic", in, out, 2, 2, 0, 1)) {
    /*@INIT  M=MESH:INIT('prismatic', M0, @int NLAY)
      Extrude a prismatic mesh M from a mesh M0. 
      
      In the additional dimension there are NLAY layers of elements built
      from 0 to 1. @*/
    prismatic_mesh(pmesh, in);
  } else if (check_cmd(cmd, "pt2D", in, out, 2, 3, 0, 1)) {
    /*@INIT  M=MESH:INIT('pt2D', @dmat P, @ivec T[, @int N])
      Build a mesh from a 2D triangulation.

      Each column of P contains a point coordinate, and each column of T contains the point indices of a triangle. N is optional and
      is a zone number. If N is specified then only the zone number
      'N' is converted (in that case, T is expected to have 4 rows,
      the fourth containing these zone numbers).@MATLAB{<Par>Can be used to Convert a "pdetool" triangulation exported in variables P and T into a GETFEM mesh.}
      @*/
    ptND_mesh(pmesh, 2, in);
  } else if (check_cmd(cmd, "ptND", in, out, 2, 2, 0, 1)) {
    /*@INIT  M=MESH:INIT('ptND', @dmat P, @imat T)
      Build a mesh from a N-dimensional "triangulation".

      Similar function to 'pt2D', for building simplexes meshes from a
      triangulation given in T, and a list of points given in P. The
      dimension of the mesh will be the number of rows of P, and the
      dimension of the simplexes will be the number of rows of T.
      @*/
    ptND_mesh(pmesh, size_type(-1), in);
  } else if (check_cmd(cmd, "load", in, out, 1, 1, 0, 1)) {
    /*@INIT  M=MESH:INIT('load', @str FILENAME)
      Load a mesh from a GETFEM++ ascii mesh file. See also
      MESH:GET('save',FILENAME). @*/
    std::string fname = in.pop().to_string();
    pmesh->read_from_file(fname);
  } else if (check_cmd(cmd, "from string", in, out, 1, 1, 0, 1)) {
    /*@INIT  M=MESH:INIT('from string', @str S)
      Load a mesh from a string description. For example, a string returned
      by MESH:GET('char').
      @*/
    std::stringstream ss(in.pop().to_string());
    pmesh->read_from_file(ss);
  } else if (check_cmd(cmd, "import", in, out, 2, 2, 0, 1)) {
    /*@INIT  M=MESH:INIT('import', @str FORMAT, @str FILENAME)
      Import a mesh, FORMAT may be:<Par>
      - 'gmsh'   for a mesh created with gmsh ( http://www.geuz.org/gmsh )<par>
      - 'gid'    for a mesh created with GiD  ( http://gid.cimne.upc.es )<par>
      - 'am_fmt' for a mesh created with emc2 ( http://pauillac.inria.fr/cdrom/www/emc2/fra.htm )
      @*/
    std::string fmt = in.pop().to_string();
    std::string fname = in.pop().to_string();
    getfem::import_mesh(fname, fmt, *pmesh);
  } else if (check_cmd(cmd, "clone", in, out, 1, 1, 0, 1)) {
    /*@INIT  M=MESH:INIT('clone', @tmesh M2)
      Create a copy of a mesh.
      @*/
    const getfem::getfem_mesh *m2 = in.pop().to_const_mesh();
    pmesh->copy_from(*m2);
  } else bad_cmd(cmd);
}
