#include <getfemint_misc.h>
#include <getfemint_mesh_fem.h>

using namespace getfemint;


static void set_fem(getfem::mesh_fem *mf, getfemint::mexargs_in& in)
{
  getfem::pfem               fem = in.pop().to_fem();
  getfem::pintegration_method pim = 0;
  if (in.remaining() && 
      in.front().is_object_id()) pim = in.pop().to_fem_interpolation();

  /* check or build the convex list */
  dal::bit_vector bv;
  if (in.remaining() == 1) {
    bv = in.pop().to_bit_vector(&mf->linked_mesh().convex_index(), -1);
  } else {
    bv = mf->linked_mesh().convex_index();
  }
  
  /* check for the validity of the operation */
  for (dal::bv_visitor cv(bv); !cv.finished(); ++cv) {
    if (!mf->linked_mesh().convex_index().is_in(cv))
      THROW_ERROR("Convex " << cv+config::base_index() << " was not found in mesh");
    if (fem->basic_structure(cv) != mf->linked_mesh().structure_of_convex(cv)->basic_structure())
      infomsg() << "Warning: structure of the FEM seems to be incompatible with the structure of the convex (if you are using high degree geom. transf. ignore this)\n";
  }
    
  /* all the work done here */
  mf->set_finite_element(bv, fem, pim);
}

/* set the classical fem of order on the mesh_fem, with a classical integration
   method */
static void set_classical_fem(getfem::mesh_fem *mf, getfemint::mexargs_in& in, bool discontinuous) {
  dim_type K = in.pop().to_integer(0,255), IM_DEGREE = dim_type(-1);
  if (in.remaining()) IM_DEGREE = in.pop().to_integer(-1,255);
  dal::bit_vector bv;
  if (in.remaining() == 1) {
    bv = in.pop().to_bit_vector(&mf->linked_mesh().convex_index(), -1);
  } else {
    bv = mf->linked_mesh().convex_index();
  }
  if (!discontinuous) {
    mf->set_classical_finite_element(bv,K,IM_DEGREE);
  } else {
    mf->set_classical_discontinuous_finite_element(bv,K,IM_DEGREE);
  }
}

static void set_boundary_num(getfem::mesh_fem *mf, getfemint::mexargs_in& in)
{
  unsigned boundary_num  = in.pop().to_integer(1,100000);
  iarray v               = in.pop().to_iarray(2,-1);

  const getfem::getfem_mesh &mesh = mf->linked_mesh();

  /* loop over the edges of mxEdge */
  for (size_type j=0; j < v.getn(); j++) {
    size_type cv = size_type(v(0,j))-config::base_index();
    size_type f  = size_type(v(1,j))-config::base_index();
    if (!mesh.convex_index().is_in(cv)) {
      THROW_BADARG( "Invalid convex number '" << cv+config::base_index() << "' at column " << j+config::base_index());
    }
    if (f >= mesh.structure_of_convex(cv)->nb_faces()) {
      THROW_BADARG( "Invalid face number '" << f+config::base_index() << "' at column " << j+config::base_index());
    }
    mf->add_boundary_elt(boundary_num, cv, f);
    if (!mf->is_convex_on_boundary(cv, boundary_num)) THROW_INTERNAL_ERROR;
  }
}

/*MLABCOM
  FUNCTION [x] = gf_mesh_fem_set(meshfem MF, operation [, args])

  General function for editing mesh_fem objects
  
  @SET MESHFEM:SET('fem')
  @SET MESHFEM:SET('classical fem')
  @SET MESHFEM:SET('classical discontinuous fem')
  @SET MESHFEM:SET('boundary')
  @SET MESHFEM:SET('delete boundary')
  @SET MESHFEM:SET('qdim')

  $Id: gf_mesh_fem_set.C,v 1.10 2005/01/05 16:33:32 pommier Exp $
MLABCOM*/

void gf_mesh_fem_set(getfemint::mexargs_in& in, getfemint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }

  getfem::mesh_fem *mf = in.pop().to_mesh_fem();
  std::string cmd        = in.pop().to_string();
  if (check_cmd(cmd, "fem", in, out, 1, 3, 0, 0)) {
    /*@SET MESHFEM:SET('fem', @tfem FEM[, @integ PPI ][, @ivec CVIDX])
      Set the FEM and integration methods.
      
      Sets the finite element method to FEM with PPI as integration method
      for all the convexes of #id listed in CVIDX. If CVIDX is not used,
      the FEM/PPI is assigned to all convexes.  If PPI is omitted, the
      dummy integration method IM_NONE will be used.
      @*/
    set_fem(mf, in);
  } else if (check_cmd(cmd, "classical fem", in, out, 1, 3, 0, 0)) {
    /*@SET MESHFEM:SET('classical fem', @int K, [@int IM_DEGREE [,@ivec CVIDX]])
      Assigns a classical (Lagrange polynomial) fem or order K to the @tmf.
      
      Uses FEM_PK for simplexes, FEM_QK for parallelepipeds etc.  An
      approximate integration method will be choosen (if available),
      such that polynomials of degree <= IM_DEGREE are exactly
      integrated. If IM_DEGREE=-1, then the dummy integration method
      IM_NONE will be used. @*/
    set_classical_fem(mf, in, false);
  } else if (check_cmd(cmd, "classical discontinuous fem", in, out, 1, 3, 0, 0)) {
    /*@SET MESHFEM:SET('classical discontinuous fem', @int K, [@int IM_DEGREE [,@ivec CVIDX]])
      Assigns a classical (Lagrange polynomial) discontinuous fem or order K.
      
      Similar to MESHFEM:SET('classical fem') except that
      FEM_PK_DISCONTINUOUS is used.
      @*/
    set_classical_fem(mf, in, true);
  } else if (check_cmd(cmd, "boundary", in, out, 2, 2, 0, 0)) {
    /*@SET MESHFEM:SET('boundary', @int bnum, @dmat FACES)
      Assigns the boundary number bnum to the convex faces stored in each
      column of the matrix FACES. Each column contains the list of point
      numbers that composes the face (ie the first row of FACES contains a
      convex number, and the second row contains a face number in the
      convex).
      @*/
    set_boundary_num(mf, in);
  } else if (check_cmd(cmd, "delete boundary", in, out, 1, 1, 0, 0)) {
    /*@SET MESHFEM:SET('delete boundary', @ivec BLST)
      Removes the boundaries listed in BLST
      @*/
    for (dal::bv_visitor_c b(in.pop().to_bit_vector(&mf->get_valid_boundaries(),0)); !b.finished(); ++b)
      mf->sup_boundary(b);
  } else if (check_cmd(cmd, "qdim", in, out, 1, 1, 0, 0)) {
    /*@SET MESHFEM:SET('qdim', @int Q)
      Change the Q dimension of the field that is interpolated by the mesh_fem.
      
      Q=1 means that the mesh_fem describes a scalar field, Q=N means
      that the mesh_fem describes a vector field of dimension N. @*/
    size_type q_dim = in.pop().to_integer(1,255);
    mf->set_qdim(q_dim);
  } else bad_cmd(cmd);
}
