#include <getfemint_misc.h>
#include <getfemint_mesh.h>

using namespace getfemint;

static void check_empty_mesh(const getfem::getfem_mesh *pmesh)
{
  if (pmesh->dim() == bgeot::dim_type(-1) || pmesh->dim() == 0) {
    THROW_ERROR( "mesh object has an invalid dimension");
  }
}

/*MLABCOM
  FUNCTION [x] = gf_mesh_set(mesh M, operation [, args])

    General function for modification of a mesh object.

    @SET MESH:SET('add point')
    @SET MESH:SET('del point')
    @SET MESH:SET('add convex')
    @SET MESH:SET('del convex')
    @SET MESH:SET('del convex of dim')
    @SET MESH:SET('translate')
    @SET MESH:SET('transform')
    @SET MESH:SET('merge')
    @SET MESH:SET('optimize structure')
  MLABCOM*/
void gf_mesh_set(getfemint::mexargs_in& in, getfemint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }

  getfem::getfem_mesh *pmesh = in.pop().to_mesh();
  std::string cmd            = in.pop().to_string();
  if (check_cmd(cmd, "add point", in, out, 1, 1, 0, 1)) {
    /*@SET [IDX]=MESH:SET('add point', @mat PT)
    Inserts new points in the mesh and return their #id.

    PT should be an [n x m] matrix , where n is the mesh dimension,
    and m is the number of points that will be added to the mesh. On
    output, IDX contains the indices of these new points.
    <Par>
    Remark: if some points are already part of the mesh (with a small
    tolerance of approximately 1e-8), they won't be inserted again,
    and IDX will contain the previously assigned indices of the
    points. @*/
    check_empty_mesh(pmesh);
    darray v = in.pop().to_darray(pmesh->dim(), -1);
    iarray w = out.pop().create_iarray_h(v.getn());
    for (size_type j=0; j < v.getn(); j++) {
      w[j] = pmesh->add_point(v.col_to_bn(j)) + config::base_index();
    }
  } else if (check_cmd(cmd, "del point", in, out, 1, 1, 0, 0)) {
    /*@SET MESH:SET('del point', @ivec PIDLST)
    Removes one or more points from the mesh.

    PIDLST should contain the 
    point #id, such as the one returned by the 'add point' command.
    @*/
    check_empty_mesh(pmesh);
    iarray v = in.pop().to_iarray();
    
    for (size_type j=0; j < v.size(); j++) {
      id_type id = v[j]-config::base_index();
      if (pmesh->point_is_valid(id)) {
	THROW_ERROR( "Can't remove point " << id+config::base_index()
		     << ": a convex is still attached to it.");
      }
      pmesh->sup_point(id);
    }
  } else if (check_cmd(cmd, "add convex", in, out, 2, 2, 0, 1)) {
    /*@SET IDX = MESH:SET('add convex', @geotrans CVTR, @mat CVPTS)
    Add a new convex into the mesh.

    The convex structure (triangle, prism,...) is given by CVTR
    (obtained with GEOTRANS:INIT('...')), and its points are given by
    the columns of CVPTS. On return, IDX contains the convex number.
    CVPTS might be a 3-dimensional array in order to insert more than
    one convex (or a two dimensional array correctly shaped according
    to Fortran ordering). @*/
    check_empty_mesh(pmesh);
    bgeot::pgeometric_trans pgt = in.pop().to_pgt();
    darray v = in.pop().to_darray(pmesh->dim(), pgt->nb_points(), -1);
    iarray w = out.pop().create_iarray_h(v.getp());

    std::vector<getfemint::id_type> qp(pgt->nb_points());
    /* loop over convexes */
    for (size_type k=0; k < v.getp(); k++) {
      /* loop over convex points */
      for (size_type j=0; j < v.getn(); j++) {
	qp[j] = pmesh->add_point(v.col_to_bn(j,k));
      }
      id_type cv_id = pmesh->add_convex(pgt, qp.begin());
      w[k] = cv_id+config::base_index();
    }
  } else if (check_cmd(cmd, "del convex", in, out, 1, 1, 0, 0)) {
    /*@SET MESH:SET('del convex', IDX)
    Remove one or more convexes from the mesh.

    IDX should contain the convexes #ids, such as the ones returned bu
    MESH:SET('add convex'). @*/
    check_empty_mesh(pmesh);
    iarray v = in.pop().to_iarray();
    
    for (size_type j=0; j < v.size(); j++) {
      id_type id = v[j]-config::base_index();
      if (pmesh->convex_index().is_in(id)) {
	pmesh->sup_convex(id);
      } else {
	THROW_ERROR("can't delete convex " << id+config::base_index() << ", it is not part of the mesh");
      }
    }
  } else if (check_cmd(cmd, "del convex of dim", in, out, 1, 1, 0, 0)) {  
    /*@SET MESH:SET('del convex of dim', @ivec DIM)
      Remove all convexes of dimension listed in DIM.

      For example MESH:SET('del convex of dim', [1,2]) remove all line
      segments, triangles and quadrangles.
      @*/
    dal::bit_vector bv = in.pop().to_bit_vector(NULL, 0);
    for (dal::bv_visitor_c cv(pmesh->convex_index()); !cv.finished(); ++cv) {
      if (bv.is_in(pmesh->structure_of_convex(cv)->dim())) pmesh->sup_convex(cv);
    }
  } else if (check_cmd(cmd, "translate", in, out, 1, 1, 0, 0)) {
    /*@SET MESH:SET('translate', @vec V)
    Translates each point of the mesh from V.
    @*/
    check_empty_mesh(pmesh);
    darray v = in.pop().to_darray(pmesh->dim(),1);
    pmesh->translation(v.col_to_bn(0));
  } else if (check_cmd(cmd, "transform", in, out, 1, 1, 0, 0)) {
    /*@SET MESH:SET('transform', @mat T)
    Applies the matrix T to each point of the mesh.
    @*/
    check_empty_mesh(pmesh);
    darray v = in.pop().to_darray(-1,-1); //pmesh->dim());
    pmesh->transformation(v.row_col_to_bm());
  } else if (check_cmd(cmd, "merge", in, out, 1, 1, 0, 0)) {

    /*@SET MESH:SET('merge', @mesh M2) 
    Merge with the mesh M2.

    Overlapping points won't be duplicated. If M2 is a mesh_fem
    object, its linked mesh will be used. @*/
    const getfem::getfem_mesh *pmesh2 = in.pop().to_const_mesh();
    for (dal::bv_visitor cv(pmesh2->convex_index()); !cv.finished(); ++cv)
      pmesh->add_convex_by_points(pmesh2->trans_of_convex(cv), pmesh2->points_of_convex(cv).begin());
  } else if (check_cmd(cmd, "optimize structure", in, out, 0, 0, 0, 0)) {
    /*@SET MESH:SET('optimize structure')
    Reset point and convex numbering.

    After optimisation, the points (resp. convexes) will be consecutively numbered from @MATLAB{1 to MESH:GET('max pid') (resp. MESH:GET(,'max cvid'))}@PYTHON{0 to MESH:GET('max pid')-1 (resp. MESH:GET(,'max cvid')-1)}.
    @*/
    pmesh->optimize_structure();
  } else bad_cmd(cmd);
}
