#include <getfemint_gsparse.h>
#include <getfemint_workspace.h>
#include <gmm_inoutput.h>

using namespace getfemint;

template <typename MAT> 
void copydiags(const MAT &M, const std::vector<size_type> &v, 
	       garray<typename MAT::value_type> &w) {
  size_type m = gmm::mat_nrows(M), n = gmm::mat_ncols(M);
  for (size_type ii=0; ii < v.size(); ++ii) {
    int d = v[ii], i,j;
    if (d < 0) { i = -d; j = 0; } else { i = 0; j = d; }
    cout << "m=" << m << "n=" << n << ", d=" << d << ", i=" << i << ", j=" << j << "\n";
    for (; i < int(m) && j < int(n); ++i,++j)
      w(i,ii) = M(i,j);
  }
}

template <typename T> static void
gf_spmat_get_full(gsparse &gsp, getfemint::mexargs_in& in, getfemint::mexargs_out& out, T) {
  gmm::dense_matrix<T> ww;
  size_type n,m;
  if (!in.remaining()) { 
    m = gsp.nrows(); n = gsp.ncols();
    gmm::resize(ww, m, n);
    switch (gsp.storage()) {
    case gsparse::CSCMAT: gmm::copy(gsp.csc(T()), ww); break;
    case gsparse::WSCMAT: gmm::copy(gsp.wsc(T()), ww); break;
    default: THROW_INTERNAL_ERROR;
    }
  } else {
    sub_index ii = in.pop().to_sub_index().check_range(gsp.nrows());
    sub_index jj = in.remaining() ? 
      in.pop().to_sub_index().check_range(gsp.ncols()) : ii.check_range(gsp.ncols());
    m = ii.size(); n = jj.size();
    gmm::resize(ww, m, n);
    switch (gsp.storage()) {
    case gsparse::CSCMAT: gmm::copy(gmm::sub_matrix(gsp.csc(T()),ii,jj), ww); break;
    case gsparse::WSCMAT: gmm::copy(gmm::sub_matrix(gsp.wsc(T()),ii,jj), ww); break;
    default: THROW_INTERNAL_ERROR;
    }
  }
  std::copy(ww.begin(), ww.end(), out.pop().create_array(m,n,T()).begin());
}

template <typename T> static void 
gf_spmat_mult_or_tmult(gsparse &gsp, 
		       getfemint::mexargs_in& in, getfemint::mexargs_out& out, 
		       bool tmult, T) {
  size_type nj = gsp.ncols(), ni = gsp.nrows();
  if (tmult) std::swap(ni,nj);
  //cout << "NJ=" << nj << "NI=" << ni << ", tmaul=" << tmult << "\n";
  garray<T> v = in.pop().to_garray(nj,T());
  garray<T> w = out.pop().create_array_v(ni, T());
  gsp.mult_or_transposed_mult(v,w,tmult);
  /*switch (gsp.storage()) {
  case gsparse::CSCMAT: 
    if (!tmult) gmm::mult(gsp.csc(T()), v, w);
    else gmm::mult(gmm::conjugated(gsp.csc(T())), v, w);
    break;
  case gsparse::WSCMAT: 
    if (!tmult) gmm::mult(gsp.wsc(T()), v, w);
    else gmm::mult(gmm::conjugated(gsp.wsc(T())), v, w);
    break;
  default: THROW_INTERNAL_ERROR;
  }*/

}

template <typename T> static void
gf_spmat_get_diag(gsparse &gsp,
		  getfemint::mexargs_in& in, getfemint::mexargs_out& out, T) {
  std::vector<size_type> v;
  if (in.remaining()) {
    iarray vv = in.pop().to_iarray(-1);
    for (size_type i=0; i < vv.size(); ++i)
      v.push_back(vv[i]);
  } else v.push_back(0);
  garray<T> w = out.pop().create_array(std::min(gsp.nrows(), gsp.ncols()), v.size(), T());
  switch (gsp.storage()) {
  case gsparse::CSCMAT: copydiags(gsp.csc(T()), v, w); break;
  case gsparse::WSCMAT: copydiags(gsp.wsc(T()), v, w); break;
  default: THROW_INTERNAL_ERROR;
  }
}

template <typename T> static void
gf_spmat_get_data(gmm::csc_matrix_ref<const T*, const unsigned int *, const unsigned int *> M,
		  getfemint::mexargs_out& out, int which) {
  size_type nz = M.jc[M.nc];
  if (which == 0) {
    iarray w = out.pop().create_iarray_h(M.nc+1);
    for (size_type i=0; i < M.nc+1; ++i) { w[i] = M.jc[i] + config::base_index(); }
    if (out.remaining()) {
      w = out.pop().create_iarray_h(nz);
      for (size_type i=0; i < nz; ++i) { w[i] = M.ir[i] + config::base_index(); }
    }
  } else {
    garray<T> w = out.pop().create_array_h(nz, T());
    for (size_type i=0; i < M.nc+1; ++i) { w[i] = M.pr[i]; }
  }
}

/*MLABCOM
  FUNCTION [...]=gf_spmat_get(M, args)

  General getfem sparse matrix inquiry function. M might also be a
  native matlab sparse matrix.

  @GET SPMAT:GET('size')
  @GET SPMAT:GET('nnz')
  @GET SPMAT:GET('is_complex')
  @GET SPMAT:GET('storage')
  @GET SPMAT:GET('full')
  @GET SPMAT:GET('mult')
  @GET SPMAT:GET('tmult')
  @GET SPMAT:GET('diag')
  @GET SPMAT:GET('csc_ind')
  @GET SPMAT:GET('csc_val')
  @GET SPMAT:GET('info')
  @GET SPMAT:GET('save')
MLABCOM*/

void gf_spmat_get(getfemint::mexargs_in& in, getfemint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }  
  dal::shared_ptr<gsparse> pgsp = in.pop().to_sparse(); 
  gsparse &gsp = *pgsp;

  std::string cmd = in.pop().to_string();

  if (check_cmd(cmd, "nnz", in, out, 0, 0, 0, 1)) {
    /*@GET SPMAT:GET('nnz')
      Return the number of non-null values stored in the sparse matrix.
      @*/
    out.pop().from_integer(gsp.nnz());
  } else if (check_cmd(cmd, "full", in, out, 0, 2, 0, 1)) {
    /*@GET SPMAT:GET('full' [,I [,J]])
      Return a full (sub-)matrix of M.

      The optional arguments I, are the sub-intervals for the rows and
      columns that are to be extracted. @*/
    if (gsp.is_complex()) gf_spmat_get_full(gsp, in, out, complex_type());
    else gf_spmat_get_full(gsp, in,out,scalar_type());
  } else if (check_cmd(cmd, "mult", in, out, 1, 1, 0, 1)) {
    /*@GET SPMAT:GET('mult', V)
      Product of the sparse matrix K with a vector V.

      For matrix-matrix multiplications, see SPMAT:INIT('mult')
      @*/
    if (!gsp.is_complex())
      gf_spmat_mult_or_tmult(gsp, in, out, false, scalar_type());
    else gf_spmat_mult_or_tmult(gsp, in, out, false, complex_type());
  } else if (check_cmd(cmd, "tmult", in, out, 1, 1, 0, 1)) {
    /*@GET SPMAT:GET('tmult', V)
      Product of K transposed (conjugated if K is complex) with the vector V.
      @*/
    if (!gsp.is_complex())
      gf_spmat_mult_or_tmult(gsp, in, out, true, scalar_type());
    else gf_spmat_mult_or_tmult(gsp, in, out, true, complex_type());
  } else if (check_cmd(cmd, "diag", in, out, 0, 1, 0, 1)) {
    /*@GET SPMAT:GET('diag' [, E])
      Return the diagonal of K as a vector. 

      If E is used, return the sub-diagonals whose ranks are given in
      E. @*/
    if (!gsp.is_complex()) 
      gf_spmat_get_diag(gsp, in, out, scalar_type());
    else gf_spmat_get_diag(gsp, in, out, complex_type());
  } else if (check_cmd(cmd, "storage", in, out, 0, 0, 0, 1)) {
    /*@GET SPMAT:GET('storage')
      Return the storage type currently used for the matrix.

      The storage is returned as a string, either 'CSC' or 'WSC'.
      @*/
    out.pop().from_string(gsp.name());
  } else if (check_cmd(cmd, "size", in, out, 0, 0, 0, 1)) {
    /*@GET SPMAT:GET('size')
      Return a vector [ni, nj] where ni and nj are the dimensions of the matrix.
      @*/
    iarray sz = out.pop().create_iarray_h(2);
    sz[0] = gsp.nrows();
    sz[1] = gsp.ncols();
  } else if (check_cmd(cmd, "is_complex", in, out, 0, 0, 0, 1)) {
    /*@GET SPMAT:GET('is_complex') 
      Return 1 if the matrix contains complex values.
      @*/
    out.pop().from_integer(gsp.is_complex());
  } else if (check_cmd(cmd, "csc_ind", in, out, 0, 0, 0, 2)) {
    /*@GET [JC,IR]=SPMAT:GET('csc_ind') 
      Return the two usual index arrays of CSC storage.
      
      If K is not stored as a CSC matrix, it is converted into CSC.
      @*/
    gsp.to_csc();
    if (!gsp.is_complex()) gf_spmat_get_data(gsp.csc(scalar_type()),  out, 0);
    else                   gf_spmat_get_data(gsp.csc(complex_type()), out, 0);
  } else if (check_cmd(cmd, "csc_val", in, out, 0, 0, 0, 1)) {
    /*@GET [V]=SPMAT:GET('csc_val')
      Return the array of values of all non-zero entries of K.
      
      If K is not stored as a CSC matrix, it is converted into CSC.
      @*/
    gsp.to_csc();
    if (!gsp.is_complex()) gf_spmat_get_data(gsp.csc(scalar_type()),  out, 1);
    else                   gf_spmat_get_data(gsp.csc(complex_type()), out, 1);
  } else if (check_cmd(cmd, "info", in, out, 0, 1)) {
    /*@GET S=SPMAT:GET('info') 
      Return a string contains a short summary on the sparse matrix
      (dimensions, filling, ..).
      @*/
    std::stringstream ss;
    ss << gsp.nrows() << "x" << gsp.ncols() << " " << (gsp.is_complex() ? "COMPLEX" : "REAL") 
       << " " << gsp.name() << ", NNZ=" << gsp.nnz() 
       << " (filling=" << double(gsp.nnz())/(double(gsp.nrows())*gsp.ncols())*100. << "%)";
    out.pop().from_string(ss.str().c_str());
  } else if (check_cmd(cmd, "save", in, out, 2, 2, 0, 0)) {
    /*@GET SPMAT:GET('save', @str format, @str filename)
      Export the sparse matrix.

      the format of the file may be 'hb' for Harwell-Boeing, or 'mm' for Matrix-Market.
      @*/
    std::string fmt = in.pop().to_string();
    int ifmt;
    if (cmd_strmatch(fmt, "hb") || cmd_strmatch(fmt, "harwell-boeing")) ifmt = 0;
    else if (cmd_strmatch(fmt, "mm") || cmd_strmatch(fmt, "matrix-market")) ifmt = 1;
    else THROW_BADARG("unknown sparse matrix file-format : " << fmt);
    std::string fname = in.pop().to_string();
    gsp.to_csc();
    if (!gsp.is_complex()) {
      if (ifmt == 0) gmm::Harwell_Boeing_save(fname.c_str(), gsp.csc_w(scalar_type()));
      else           gmm::MatrixMarket_save(fname.c_str(), gsp.csc_w(scalar_type()));
    } else {
      if (ifmt == 0) gmm::Harwell_Boeing_save(fname.c_str(), gsp.csc_w(complex_type()));
      else           gmm::MatrixMarket_save(fname.c_str(), gsp.csc_w(complex_type()));
    }
  } else bad_cmd(cmd);
}
