# -*- coding: utf-8 -*-
# Copyright (C) 2008, 2009 Leonardo Gastón De Luca leo[at]kde.org.ar
# Copyright (C) 2010 Osama Khalid osamak[at]gnu.org
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Based on youogg's plugins.py file from SVN revision 50.


"""
GetMediumURL plugin for YouTube videos.
"""


import re
import urllib

from getmediumurl.cache import cachedproperty
from getmediumurl.format import SimpleFormat
from getmediumurl.htmlplugin import HTMLPlugin
from getmediumurl.compat import parse_qs


__all__ = ("YouTube",)


class YouTube(HTMLPlugin):

    """Plugin to get YouTube video URLs."""

    match_re = re.compile(r'^http://(?:www\.|.{2}\.)?youtube\.com/' \
                              '(?:watch(?:_popup)?\?v=|v/)(?P<mediumid>.{11})')
    raw_url_format = u"http://www.youtube.com/watch?v=%s&skipcontrinter=1"
    url_format = u"http://www.youtube.com/watch?v=%s"
    website_name = u"YouTube"
    high_quality = True

    @cachedproperty
    def _flashvars(self):
        """A dictionary of variables used by the site's Flash player."""
        data = self.get_content_substring(start=r'flashvars=\"', end='\"')
        return parse_qs(urllib.unquote(data.decode("ascii")))

    # Needed for tests, to avoid HTTP request.
    @cachedproperty
    def url(self):
        """URL which was matched to the plugin, or equivalent."""
        return self.url_format % self.mediumid

    @property
    def thumbnail(self):
        """Thumbnail image URL."""
        return u"http://i3.ytimg.com/vi/%s/default.jpg" % self.mediumid

    @property
    def language(self):
        """Video language."""
        assert self
        # Not sure how this could be obtained.
        return u""

    @cachedproperty
    def author_name(self):
        """Return submitter's user name."""
        for element in self.page.getiterator("div"):
            if element.get("id") == "watch-headline-user-info":
                return element.findtext("a/strong")

    @cachedproperty
    def author_url(self):
        """Return submitter's profile URL."""
        for element in self.page.getiterator("div"):
            if element.get("id") == "watch-headline-user-info":
                return self.make_absolute(element.find("a").get("href"))

    def formats(self):
        """Get data about formats of the video."""
        fmt_url_map = urllib.unquote(self.get_content_substring(\
                start="fmt_url_map=", end="&").decode("ascii"))
        fmt_list = self._flashvars["fmt_list"][0]
        fmts = {}
        for fmt in fmt_list.split(u","):
            parts = fmt.split(u"/")
            width, height = (int(x) for x in parts[1].split(u"x"))
            flash = u"/".join(parts[2:])
            # Formats 34 and 35 use H264 in FLV.
            if flash == u"7/0/0" or parts[0] in (u"34", u"35"):
                mime_type = "video/flv"
            elif flash == u"9/0/115":
                mime_type = "video/mp4"
            else:
                raise NotImplementedError("unknown video format %s" % flash)
            # At least one video has it.
            if parts[0] == u"18":
                width, height = 480, 270
            if parts[0] == u"5":
                width, height = 400, 226
            fmts[parts[0]] = (mime_type, width, height)
        for fmt in fmt_url_map.split(u","):
            code, url = fmt.split(u"|")
            yield SimpleFormat(url, *fmts[code])
