# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Base class for reading text located at specific URLs.
"""


__all__ = ("ReaderError", "NotFoundError", "URLReader", "StaticURLReader")


class ReaderError(ValueError):
    """Exception raised if any operation on a `URLReader` fails.

    The argument might an string describing why the exception
    occurred, or an exception raised in some inner code.
    """


class NotFoundError(ReaderError):
    """Exception raised when using a `URLReader` with non-existent document.

    Not all operations will raise it.
    """


class URLReader(object):

    """Finding documents at URLs.

    The methods are implemented in classes extending this one.
    """

    def __init__(self, url):
        """Read an `url`."""
        self._url = url
        """
        The original URL of the document.
        """

    @property
    def content(self):
        """The document content as a string.

        Must be overridden in subclasses to find any document.
        """
        assert self
        raise ReaderError("URLReader.content not overridden")

    @property
    def content_type(self):
        """The document's MIME Content-Type header.

        An empty string is returned in case of insuffiecient data for
        a meaningful answer.
        """
        assert self
        return u""

    @property
    def url(self):
        """The target URL of the document.

        Should be different from the constructor's argument in case of
        e.g. HTTP redirects.
        """
        return self._url


class StaticURLReader(URLReader):

    """An object storing data obtained by an `URLReader`."""

    def __init__(self, url, target=None, content=None, content_type=u""):
        """Initialize with result data."""
        super(StaticURLReader, self).__init__(url)
        self._target = target
        self._content = content
        self._content_type = content_type

    @property
    def content(self):
        """The document content as a string."""
        if self._content is not None:
            return self._content
        else:
            return super(StaticURLReader, self).content

    @property
    def content_type(self):
        """The document Content-Type header or empty string."""
        return self._content_type

    @property
    def url(self):
        """The target URL of the document."""
        if self._target is not None:
            return self._target
        else:
            return super(StaticURLReader, self).url
