# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Break.com plugin for GetMediumURL."""


import re

from getmediumurl.cache import cachedproperty
from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium
from getmediumurl.format import SimpleFormat
from getmediumurl.xmlhtml import read_html
from getmediumurl.utils import parse_urlpart


__all__ = ("Break",)


#: Regex matched by embed URLs.
_EMBED = re.compile("^http://embed\.break\.com/(\d+)$")


class BreakMedium(HTMLMedium):

    """A video on Break.com."""

    website_name = "Break.com"

    @cachedproperty
    def _embed_query(self):
        """Query parameters of URL to which the embed link redirects."""
        url = "http://embed.break.com/%s" % self.mediumid
        return parse_urlpart(self.urlreader(url).url, "query")

    @property
    def url(self):
        """Page URL."""
        return self._embed_query["contentURL"][0]

    def __iter__(self):
        """Iterate video formats."""
        yield SimpleFormat(url="%s?%s" % (self._embed_query["sVidLoc"][0],
                                          self._embed_query["icon"][0]))

    @property
    def title(self):
        """Video title."""
        return self.page.xpath("/html/head/meta[@name='embed_video_title']"
                               "/@content")[0]


class Break(HTMLPlugin):

    """Plugin for Break.com."""

    medium_class = BreakMedium

    @classmethod
    def match(cls, url, matcher):
        """Return plugin's object handling this URL, or `None`."""
        if not url.startswith("http://www.break.com/"):
            return None
        embed_url = None
        dom = read_html(matcher.urlreader(url))
        embed_url = dom.xpath("/html/head/meta[@name='embed_video_url']"
                              "/@content")[0]
        re_match = _EMBED.match(embed_url)
        assert re_match is not None
        return cls(re_match.group(1), matcher)
