# -*- coding: utf-8 -*-
# Copyright (C) 2008, 2009 Leonardo Gastón De Luca leo[at]kde.org.ar
# Copyright (C) 2010 Osama Khalid osamak[at]gnu.org
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Based on youogg's plugins.py file from SVN revision 50.  Plugin
# originally contributed by freemind.


"""
GetMediumURL plugin for Vimeo videos.
"""


import re

from getmediumurl.cache import cachedproperty
from getmediumurl.format import SimpleFormat
from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium
from getmediumurl.xmlhtml import read_xml


__all__ = ("Vimeo",)


class VimeoMedium(HTMLMedium):

    """Data of a Vimeo video."""

    url_format = u"http://www.vimeo.com/%s"
    website_name = u"Vimeo"

    @cachedproperty
    def author_name(self):
        """Video author name."""
        return self.page.xpath("//div[@class='byline']//a/text()")[0]

    @cachedproperty
    def author_url(self):
        """Video author URL."""
        return self.make_absolute(self.page.xpath("//div[@class='byline']"
                                                  "//a/@href")[0])

    @property
    def language(self):
        """Video language."""
        assert self
        return u""  # No data on it.

    @cachedproperty
    def license(self):
        """Video license URL."""
        # They don't use rel="license".
        return self.page.xpath("//ul[@id='cc_symbols']//li/a/@href")[0]

    def __iter__(self):
        """Return an interable of supported medium formats."""
        api_url = u"http://www.vimeo.com/moogaloop/load/clip:" + \
            self.mediumid
        api_xml = read_xml(self.urlreader(api_url).content)
        signature = api_xml.findtext("//request_signature")
        expiration = api_xml.findtext("//request_signature_expires")
        url = u"http://vimeo.com/moogaloop/play/clip:%s/%s/%s/?q=sd" \
            % (self.mediumid, signature, expiration)
        yield SimpleFormat(url, mime_type="video/mp4",
                           width=int(api_xml.findtext("//width")),
                           height=int(api_xml.findtext("//height")))


class Vimeo(HTMLPlugin):

    """Plugin for Vimeo videos."""

    match_re = re.compile(r'^http://(?:www\.)?vimeo\.com/(?P<mediumid>\d+)')
    medium_class = VimeoMedium
