# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test `getmediumurl.clsutils`.
"""


import unittest

from getmediumurl.clsutils import overrides


__all__ = ("OverridesTestCase",)


class Parent(object):

    """A parent class for testing."""

    def __init__(self):
        """Do nothing."""

    def abc(self, bcd):
        """Return something."""
        assert self
        return not bcd

    @classmethod
    def xyz(cls, bcd):
        """Return something different."""
        assert cls
        return bcd


class Child(Parent):

    """A subclass for testing."""

    def abc(self, cde):
        """Return something."""
        assert self
        return not cde

    @classmethod
    def xyz(cls, bcd):
        """Return something different."""
        assert cls
        return False


class OtherChild(Parent):

    """Other subclass for testing."""

    def __init__(self):
        """Do something."""
        self.efg = True
        super(OtherChild, self).__init__()


class Unrelated(object):

    """Not a subclass of `Parent`."""

    def __init__(self):
        """Do something different."""
        self.efg = False

    def abc(self, cde):
        """Return something."""
        assert self
        return cde


class OverridesTestCase(unittest.TestCase):

    """Test `getmediumurl.clsutils.overrides`."""

    def test_overrides_init(self):
        """Test if __init__ is overridden"""
        self.assert_(overrides("__init__", Parent, object))
        self.assert_(not overrides("__init__", Child, Parent))
        self.assert_(overrides("__init__", OtherChild, Parent))

    def test_overrides_method(self):
        """Test if a method is overridden"""
        self.assert_(overrides("abc", Child, Parent))
        self.assert_(not overrides("abc", OtherChild, Parent))

    def test_no_method(self):
        """Test checking overriding non-existing methods fails"""
        self.assertRaises(AttributeError, overrides, "abc", Parent, object)
        self.assertRaises(AttributeError, overrides, "cde", Child, Parent)

    def test_overrides_classmethod(self):
        """Test if a class method is overridden"""
        self.assert_(overrides("xyz", Child, Parent))
        self.assert_(not overrides("xyz", OtherChild, Parent))

    def test_self(self):
        """Test class does not override its methods"""
        for method in ("__init__", "abc", "xyz", "__repr__"):
            self.assert_(not overrides(method, Parent, Parent))

    def test_unrelated(self):
        """Test that only subclasses override methods"""
        for method in ("__init__", "abc", "xyz", "__repr__"):
            self.assertRaises(ValueError, overrides,
                              method, Unrelated, Parent)

    def test_nonstring(self):
        """Test that overriding non-string named methods fails"""
        self.assertRaises(TypeError, overrides, 42, Child, Parent)
