# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test functions used by `getmediumurl.utils.get_page_substring`.
"""


import unittest

from getmediumurl.getsubstring import get_substring, set_get_substring_defaults


__all__ = ("GetSubstringTestCase", "DefaultsTestCase", "BytesTestCase")

#: Page used for testing.
PAGE = """Any
text with<some>fragment</s>.

And a blank &line;.
"""


class GetSubstringTestCase(unittest.TestCase):

    """Tests for `get_substring`."""

    def test_simple(self):
        """Test get_substring with single end"""
        cases = (
            ("A", "n", "An", True, True),
            ("And", ";", "And a blank &line;", True, True),
            (">", "<", ">fragment<", True, True),
            ("A", "n", "A", True, False),
            ("And", ";", "And a blank &line", True, False),
            (">", "<", ">fragment", True, False),
            ("A", "n", "n", False, True),
            ("And", ";", " a blank &line;", False, True),
            (">", "<", "fragment<", False, True),
            ("A", "n", "", False, False),
            ("And", ";", " a blank &line", False, False),
            (">", "<", "fragment", False, False),
            )
        for start, end, substring, include_start, include_end in cases:
            self.assertEquals(get_substring(PAGE, start, end, include_start,
                                            include_end),
                              substring)

    def test_notfound(self):
        """Test get_substring with start not found"""
        include_start = True
        include_end = True
        cases = (
            ("B", "n"),
            ("End", ";"),
            (">/", "<"),
            )
        for start, end in cases:
            for include_start in True, False:
                for include_end in True, False:
                    self.assertEquals(get_substring(PAGE, start, end,
                                                    include_start,
                                                    include_end),
                                      None)

    def test_noend(self):
        """Test get_substring without end"""
        cases = (
            "A",
            "And",
            ">",
            )
        for start in cases:
            for include_start in True, False:
                for include_end in True, False:
                    self.assertEquals(get_substring(PAGE, start, tuple(),
                                                    include_start,
                                                    include_end),
                                      None)

    def test_many_ends(self):
        """Test get_substring with many ends"""
        cases = (
            ("Any", ("text", "with"), "Any\ntext", True, True),
            ("An", ("y", "d"), "Any", True, True),
            ("with", ("</s>", "<some>"), "with<some>fragment</s>", True, True),
            ("And", ("lulz", "blank"), "And a blank", True, True),
            ("\n\n", (",\n", ";\n"), None, True, True),
            ("Any", ("text", "with"), "Any\n", True, False),
            ("An", ("y", "d"), "An", True, False),
            ("with", ("</s>", "<some>"), "with<some>fragment", True, False),
            ("And", ("lulz", "blank"), "And a ", True, False),
            ("\n\n", (",\n", ";\n"), None, True, False),
            ("Any", ("text", "with"), "\ntext", False, True),
            ("An", ("y", "d"), "y", False, True),
            ("with", ("</s>", "<some>"), "<some>fragment</s>", False, True),
            ("And", ("lulz", "blank"), " a blank", False, True),
            ("\n\n", (",\n", ";\n"), None, False, True),
            ("Any", ("text", "with"), "\n", False, False),
            ("An", ("y", "d"), "", False, False),
            ("with", ("</s>", "<some>"), "<some>fragment", False, False),
            ("And", ("lulz", "blank"), " a ", False, False),
            ("\n\n", (",\n", ";\n"), None, False, False),
            )
        for start, end, substring, include_start, include_end in cases:
            self.assertEquals(get_substring(PAGE, start, end, include_start,
                                            include_end),
                              substring)


def make_args():
    """Iterate some possible inputs for `set_get_substring_defaults`."""
    for start in ("finish", "http://", ".ogg"):
        for end in (("start",),
                    ("https://",),
                    (".el",),
                    (".jpg", ".png"),
                    (".svg", ".pdf"),
                    (".ogg", ".ogv"),
                    ):
            for include_start in True, False:
                for include_end in True, False:
                    yield (start, end, include_start, include_end)


class DefaultsTestCase(unittest.TestCase):

    """Test `set_get_substring_defaults`."""

    def test_identity(self):
        """Test explicit arguments"""
        for args in make_args():
            self.assertEquals(set_get_substring_defaults(args[0],
                                                         args[1],
                                                         args[2],
                                                         args[3]), args)

    def test_singletons(self):
        """Test using string instead of singleton sequence"""
        for args in make_args():
            self.assertEquals(set_get_substring_defaults(args[0],
                                                         args[1][0],
                                                         args[2], args[3]),
                              (args[0], args[1][:1], args[2], args[3]))

    def test_include_start(self):
        """Test default include_start"""
        for args in make_args():
            for start, include_start in (
                ("http://", True),
                ("https://", True),
                ("other", False),
                ("ftp://", True),
                ("mailto:", False),
                (".ogg", False),
                ):
                self.assertEquals(
                    set_get_substring_defaults(start, args[1], None, args[3]),
                    (start, args[1], include_start, args[3]))

    def test_include_end(self):
        """Test default include_end"""
        for args in make_args():
            for end, include_end in (
                (("http://",), False),
                (("https://",), False),
                (("other",), False),
                (("ftp://",), False),
                (("mailto:",), False),
                ((".ogg",), True),
                (("=ogg",), False),
                ((".tar.gz",), True),
                (("http://", "https://"), False),
                (("other", ".png"), False),
                ((".jpg", "ftp://"), False),
                ((".ogg", ".tar.gz"), True),
                ):
                self.assertEquals(
                    set_get_substring_defaults(args[0], end, args[2], None),
                    (args[0], end, args[2], include_end))


class BytesTestCase(unittest.TestCase):

    """Test using `get_substring` on bytes instead of strings."""

    def test_simple(self):
        """Test get_substring on bytes with single end"""
        # Skip on Python 2.
        try:
            bytepage = bytes("""Any
text with<some>fragment</s>.

And a blank &line;.
""", "ascii")
        except (NameError, TypeError):
            return
        cases = (
            ("A", "n", bytes("An", "ascii"), True, True),
            ("And", ";", bytes("And a blank &line;", "ascii"), True, True),
            (">", "<", bytes(">fragment<", "ascii"), True, True),
            ("A", "n", bytes("A", "ascii"), True, False),
            ("And", ";", bytes("And a blank &line", "ascii"), True, False),
            (">", "<", bytes(">fragment", "ascii"), True, False),
            ("A", "n", bytes("n", "ascii"), False, True),
            ("And", ";", bytes(" a blank &line;", "ascii"), False, True),
            (">", "<", bytes("fragment<", "ascii"), False, True),
            ("A", "n", bytes("", "ascii"), False, False),
            ("And", ";", bytes(" a blank &line", "ascii"), False, False),
            (">", "<", bytes("fragment", "ascii"), False, False),
            )
        for start, end, substring, include_start, include_end in cases:
            self.assertEquals(get_substring(bytepage, start, end,
                                            include_start, include_end),
                              substring)
