# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test GetMediumURL's Matcher class.
"""


import unittest

from getmediumurl import Matcher
from getmediumurl.plugins.youtube import YouTube


__all__ = ("DefaultPluginsTestCase", "NoPluginsTestCase")


class DefaultPluginsTestCase(unittest.TestCase):

    """Test finding default plugins in Matcher."""

    def setUp(self):
        """Prepare the Matcher."""
        self.matcher = Matcher(urlreader=None)

    def test_has_default_plugin(self):
        """Test that the YouTube plugin is found in default plugin list"""
        self.assert_("YouTube" in self.matcher.available_plugins())

    def test_all_plugins_usable(self):
        """Test that all default plugins are usable"""
        for plugin in self.matcher.available_plugins():
            for disabled in self.matcher.disabled_plugins():
                self.assert_(plugin != disabled[0])

    def test_not_all_urls_match(self):
        """Test that not all URLs match"""
        for url in ("invalid-url;//with strange głyphs",
                    "any-other-thing",
                    ):
            self.assert_(self.matcher.match(url) is None)

    def test_some_urls_match(self):
        """Test that some URLs match"""
        url = "http://www.youtube.com/watch?v=xxxxxxxxxxx"
        plugin = self.matcher.match(url)
        self.assert_(plugin is not None)


class NoPluginsTestCase(unittest.TestCase):

    """Test using Matcher without any plugins."""

    def setUp(self):
        """Prepare the Matcher."""
        self.matcher = Matcher(tuple())

    def test_empty_repr(self):
        """Test representation of empty Matcher"""
        self.assertEqual("<Matcher []>", repr(self.matcher))

    def test_no_plugin(self):
        """Test that URLs won't be matched without plugins"""
        for url in ("http://example.com/",
                    "http://www.youtube.com/watch?v=xxxxxxxxxxx",
                    "any-other-thing"):
            self.assert_(self.matcher.match(url) is None)
