/* 
   Copyright 2002 Cyril Picard

   This file is part of the GEDCOMParser library 
   (developed within the Genealogy Free Software Tools project).

   The GEDCOMParser library is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   The GEDCOMParser library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with the GEDCOMParser library ; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "GEDCOMHelper/IndividualRecordHelper.hh"

GEDCOMParser::PersonalName *
  GEDCOMHelper::IndividualRecordHelper::setPersonalName (std::
							 string const
							 &first_name,
							 std::
							 string const
							 &last_name)
{
  GEDCOMParser::PersonalName * personal_name;
  GEDCOMParser::PersonalNames_t personal_names =
    _individual_record->getPersonalNames ();
  if (personal_names.size () == 0)
    {
      personal_name = _individual_record->addPersonalName ();
    }
  else
    {
      personal_name = personal_names[0].getPtr ();
    }
  personal_name->setGivn (first_name);
  personal_name->setSurn (last_name);
  return personal_name;
}

GEDCOMParser::IndividualEvent * const
GEDCOMHelper::IndividualRecordHelper::getBirth (void) const
{
  GEDCOMParser::IndividualEvent * const
    res =
    getEvent (GEDCOMParser::IndividualEvent::e_Birt);
  return res;
}

GEDCOMParser::IndividualEvent * const
GEDCOMHelper::IndividualRecordHelper::getDeath (void) const
{
  GEDCOMParser::IndividualEvent * const
    res =
    getEvent (GEDCOMParser::IndividualEvent::e_Deat);
  return res;
}

GEDCOMParser::IndividualEvent * const
GEDCOMHelper::IndividualRecordHelper::getEvent (GEDCOMParser::
						IndividualEvent::
						enumIndividualEvents const
						&event) const
{
  GEDCOMParser::IndividualEvent * res = 0;
  GEDCOMParser::IndividualEvents_t const &
    events =
    _individual_record->
    getIndividualEvents ();
  GEDCOMParser::IndividualEvents_t::const_iterator iter =
    std::find (events.begin (), events.end (), event);
  if (iter != events.end ())
    {
      res = (*iter).getPtr ();
    }
  return res;
}

GEDCOMParser::IndividualEvent *
  GEDCOMHelper::IndividualRecordHelper::setEvent (GEDCOMParser::
						  IndividualEvent::
						  enumIndividualEvents const
						  &event_type,
						  std::string const &date,
						  std::string const &location)
{
  GEDCOMParser::IndividualEvent * event = getEvent (event_type);
  if (event == 0)
    {
      event = _individual_record->addIndividualEvent ();
      event->setType (event_type);
    }
  SmartPtr < GEDCOMParser::Event > event_detail = event->getEvent ();
  if (event_detail.Null ())
    {
      event->setEvent ();
      event_detail = event->getEvent ();
    }
  event_detail->setDate (date);
  SmartPtr < GEDCOMParser::Place > place = event_detail->getPlace ();
  if (place.Null ())
    {
      event_detail->setPlace ();
      place = event_detail->getPlace ();
    }
  place->setPlaceValue (location);
  return event;
}

GEDCOMParser::IndividualEvent *
  GEDCOMHelper::IndividualRecordHelper::setBirth (std::string const &date,
						  std::string const &location)
{
  GEDCOMParser::IndividualEvent::enumIndividualEvents const
    event_type =
    GEDCOMParser::IndividualEvent::e_Birt;
  GEDCOMParser::IndividualEvent * event =
    setEvent (event_type, date, location);
  return event;
}

GEDCOMParser::IndividualEvent *
  GEDCOMHelper::IndividualRecordHelper::setDeath (std::string const &date,
						  std::string const &location)
{
  GEDCOMParser::IndividualEvent::enumIndividualEvents const
    event_type =
    GEDCOMParser::IndividualEvent::e_Deat;
  GEDCOMParser::IndividualEvent * event =
    setEvent (event_type, date, location);
  return event;
}

GEDCOMParser::IndividualAttribute * const
GEDCOMHelper::IndividualRecordHelper::getOccupationObject (void) const
{
  GEDCOMParser::IndividualAttribute * res = 0;
  GEDCOMParser::IndividualAttributes_t attributes =
    _individual_record->getIndividualAttributes ();
  GEDCOMParser::IndividualAttributes_t::const_iterator iter =
    std::find (attributes.begin (), attributes.end (),
	       GEDCOMParser::IndividualAttribute::e_Occu);
  if (iter != attributes.end ())
    {
      res = (*iter).getPtr ();
    }
  return res;
}

std::string GEDCOMHelper::IndividualRecordHelper::getOccupation (void) const
{
  std::string res;
  GEDCOMParser::IndividualAttribute * ia = getOccupationObject ();
  if (ia != 0)
    {
      res = ia->getValue ();
    }
  return res;
}

GEDCOMParser::IndividualAttribute *
  GEDCOMHelper::IndividualRecordHelper::setOccupation (std::
						       string const
						       &occupation)
{
  GEDCOMParser::IndividualAttribute * ia = getOccupationObject ();
  if (ia == 0)
    {
      ia = _individual_record->addIndividualAttribute ();
      ia->setType (GEDCOMParser::IndividualAttribute::e_Occu);
    }
  ia->setValue (occupation);
  return ia;
}

GEDCOMParser::FamilyRecord *
  GEDCOMHelper::IndividualRecordHelper::setChildInFamily (GEDCOMParser::
							  FamilyRecord *
							  const family = 0)
{
  GEDCOMParser::FamilyRecord * res = family;
  if ((_individual_record != 0) && (_lineage != 0))
    {
      GEDCOMParser::ChildToFamilyLink * const
	link =
	_individual_record->
	addChildToFamilyLink ();
      if (res == 0)
	{
	  res = _lineage->addFamilyRecord ();
	}
      link->setFamXref (res->getId ());
      res->addChilXref (_individual_record->getId ());
    }
  return res;
}

GEDCOMParser::FamilyRecord *
  GEDCOMHelper::IndividualRecordHelper::setSpouseInFamily (GEDCOMParser::
							   FamilyRecord *
							   const family = 0)
{
  GEDCOMParser::FamilyRecord * res = family;
  if ((_individual_record != 0) && (_lineage != 0))
    {
      GEDCOMParser::SpouseToFamilyLink * const
	link =
	_individual_record->
	addSpouseToFamilyLink ();
      if (res == 0)
	{
	  res = _lineage->addFamilyRecord ();
	}
      link->setFamXref (res->getId ());
      if (_individual_record->getSex () ==
	  GEDCOMParser::IndividualRecord::MALE)
	{
	  res->setHusb (_individual_record->getId ());
	}
      else
	{
	  res->setWife (_individual_record->getId ());
	}
    }
  return res;
}

GEDCOMParser::FamilyRecord *
  GEDCOMHelper::IndividualRecordHelper::getFamilySpouse (void)
{
  GEDCOMParser::FamilyRecord * res = 0;
  if ((_individual_record != 0) && (_lineage != 0))
    {
      GEDCOMParser::SpouseToFamilyLinks_t links =
	_individual_record->getSpouseToFamilyLinks ();
      if (links.size () > 0)
	{
	  GEDCOMParser::SpouseToFamilyLink const *const
	    link =
	    links[0].
	    getPtr ();
	  std::string const
	    family_id =
	    link->
	    getFamXref ();
	  GEDCOMParser::FamilyRecords_t families =
	    _lineage->getFamilyRecords ();
	  SmartPtr < GEDCOMParser::FamilyRecord > family =
	    families[family_id];
	  res = family.getPtr ();
	}
    }
  return res;
}

GEDCOMParser::FamilyRecord *
  GEDCOMHelper::IndividualRecordHelper::getFamilyChild (void)
{
  GEDCOMParser::FamilyRecord * res = 0;
  if ((_individual_record != 0) && (_lineage != 0))
    {
      GEDCOMParser::ChildToFamilyLinks_t links =
	_individual_record->getChildToFamilyLinks ();
      if (links.size () > 0)
	{
	  GEDCOMParser::ChildToFamilyLink const *const
	    link =
	    links[0].
	    getPtr ();
	  std::string const
	    family_id =
	    link->
	    getFamXref ();
	  GEDCOMParser::FamilyRecords_t families =
	    _lineage->getFamilyRecords ();
	  SmartPtr < GEDCOMParser::FamilyRecord > family =
	    families[family_id];
	  res = family.getPtr ();
	}
    }
  return res;
}

GEDCOMParser::IndividualRecord *
  GEDCOMHelper::IndividualRecordHelper::getIndividualParent (std::
							     string const
							     &sex) const
{
  std::string fam_id;
  std::string parent_id;
  GEDCOMParser::FamilyRecords_t const &
    families =
    _lineage->
    getFamilyRecords ();
  GEDCOMParser::IndividualRecords_t const &
    individus =
    _lineage->
    getIndividualRecords ();
  GEDCOMParser::FamilyRecords_t::const_iterator fam_iter;
  GEDCOMParser::IndividualRecords_t::const_iterator parent_iter;
  GEDCOMParser::IndividualRecords_elem_t tmp_res;
  GEDCOMParser::IndividualRecord * res = 0;

  if (_individual_record->getChildToFamilyLinks ().size () == 1)
    {
      fam_id = _individual_record->getChildToFamilyLinks ()[0]->getFamXref ();
      fam_iter = families.find (fam_id);
      if (fam_iter != families.end ())
	{
	  if (sex == GEDCOMParser::IndividualRecord::MALE)
	    {
	      parent_id = (*fam_iter).second->getHusb ();
	    }
	  else
	    {
	      parent_id = (*fam_iter).second->getWife ();
	    }
	  parent_iter = individus.find (parent_id);
	  if (parent_iter != individus.end ())
	    {
	      tmp_res = *parent_iter;
	      res = tmp_res.second.getPtr ();
	    }
	}
    }
  return res;
}

void
GEDCOMHelper::IndividualRecordHelper::RemoveChildFromFamily (GEDCOMParser::
							     FamilyRecord *
							     const family)
{
  if ((_individual_record != 0) && (_lineage != 0))
    {
      family->RemoveChilXref (_individual_record->getId ());
      GEDCOMParser::ChildToFamilyLinks_t links =
	_individual_record->getChildToFamilyLinks ();
      GEDCOMParser::ChildToFamilyLinks_t::iterator link_iter =
	find_if (links.begin (), links.end (),
		 GEDCOMParser::ChildToFamilyLink::IsEqualByFamXref (family->
								    getId
								    ()));
      if (link_iter != links.end ())
	{
	  _individual_record->RemoveChildToFamilyLink (*link_iter);
	}
      RemoveNotUsedFamily (family);
    }
  return;
}

void
GEDCOMHelper::IndividualRecordHelper::RemoveSpouseFromFamily (GEDCOMParser::
							      FamilyRecord *
							      const family)
{
  if ((_individual_record != 0) && (_lineage != 0))
    {
      if (_individual_record->getSex () ==
	  GEDCOMParser::IndividualRecord::MALE)
	{
	  family->setHusb ("");
	}
      else
	{
	  family->setWife ("");
	}
      GEDCOMParser::SpouseToFamilyLinks_t links =
	_individual_record->getSpouseToFamilyLinks ();
      GEDCOMParser::SpouseToFamilyLinks_t::iterator link_iter =
	find_if (links.begin (), links.end (),
		 GEDCOMParser::SpouseToFamilyLink::IsEqualByFamXref (family->
								     getId
								     ()));
      if (link_iter != links.end ())
	{
	  _individual_record->RemoveSpouseToFamilyLink (*link_iter);
	}
      RemoveNotUsedFamily (family);
    }
  return;
}


void
GEDCOMHelper::IndividualRecordHelper::RemoveNotUsedFamily (GEDCOMParser::
							   FamilyRecord *
							   const family)
{
  if ((family->getChilXrefs ().size () == 0) &&
      (family->getWife () == "") && (family->getHusb () == ""))
    {
      GEDCOMParser::FamilyRecords_t families = _lineage->getFamilyRecords ();
      GEDCOMParser::FamilyRecords_t::iterator family_iter =
	families.find (family->getId ());
      if (family_iter != families.end ())
	{
	  _lineage->RemoveFamilyRecord (family_iter->second);
	}
    }
  return;
}
