/* 
    Copyright 2002 Cyril Picard

    This file is part of the GEDCOMParser library 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMParser library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMParser library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMParser library ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#ifndef _GEDCOMPARSER_ASSOCIATION_HH_
#define _GEDCOMPARSER_ASSOCIATION_HH_

#include <string>
#include <vector>
#include "patterns/SmartPtr.hh"
#include "GEDCOMParser/AttachmentManager.hh"
#include "GEDCOMParser/Visitor/GEDCOMElement.hh"

//@Include: ../../doc/reference/GEDCOMPARSER.desc
namespace GEDCOMParser
{
  class GEDCOMVisitor;

  ///@name typedefs
  //@{
  /// @doc handling of many individus
  typedef std::vector < std::string > IndiXrefs_t;
  class Association;
  /// @doc handling of many associations
  typedef std::vector < SmartPtr < Association > >Associations_t;
  //@}
  
  /** The class GEDCOMParser::Association implements the GEDCOM 5 5 ASSOCIATION_STRUCTURE structure.
 
     {\bf GEDCOM 5.5 definition:} 
      \begin{verbatim}
      ASSOCIATION_STRUCTURE: =
      n  ASSO @<XREF:INDI>@  {0:M}
      +1 TYPE <RECORD_TYPE>  {1:1}
      +1 RELA <RELATION_IS_DESCRIPTOR>  {1:1}
      +1 <<NOTE_STRUCTURE>>  {0:M}
      +1 <<SOURCE_CITATION>>  {0:M}
      \end{verbatim}

      @see ASSOCIATION_STRUCTURE
   */
  class Association:public GEDCOMElement, public AttachmentManager
  {
  public:
    /// Destructor
    virtual ~ Association (void)
    {
    };
    /** Add a value to #XREF:INDI#.

        @see XREF:INDI
    */
    void addIndiXref (std::string const &);
    /** set the #RECORD_TYPE# value.

	@see RECORD_TYPE
    */
    void setType (std::string const &);
    /** set the #RELATION_IS_DESCRIPTOR# value.

	@see RELATION_IS_DESCRIPTOR
    */
    void setRela (std::string const &);
    /** get the #XREF:INDI# values
	
        @see XREF:INDI
    */
    IndiXrefs_t const &getIndiXrefs (void) const;
    /** get the #RECORD_TYPE# value.

	@see RECORD_TYPE
    */
    virtual std::string const &getType (void) const;
    /** get the #RELATION_IS_DESCRIPTOR# value.

        @see RELATION_IS_DESCRIPTOR
    */
    virtual std::string const &getRela (void) const;
    //@Include: ../../doc/reference/ACCEPT.desc    
    virtual void Accept (GEDCOMParser::GEDCOMVisitor *);
    /** @name ASSOCIATION_STRUCTURE
	@doc ASSOCIATION_STRUCTURE values
    */
    //@{
    /** @name XREF:INDI
	@doc A pointer to, or a cross-reference identifier of, an individual record.
    */
     /** @name RECORD_TYPE
	@doc [ FAM | INDI | NOTE | OBJE | REPO | SOUR | SUBM | SUBN ]
	An indicator of the record type being pointed to or used. For example if in an ASSOciation, an INDIvidual record were to be ASSOciated with a FAM record then:
      \begin{verbatim}	
      0 INDI
        1 ASSO @F1@
          2 TYPE FAM   (ASSOCIATION is with a FAM record.)
          2 RELA Witness at marriage
      \end{verbatim}
     */
    /** @name RELATION_IS_DESCRIPTOR
	@doc A word or phrase that states object 1's  relation  is object 2. For example you would read the following as "Joe Jacob's great grandson is the submitter pointed to by the @XREF:SUBM@":
	\begin{verbatim}
        0 INDI
	  1 NAME Joe /Jacob/
	  1 ASSO @<XREF:SUBM>@
	    2 TYPE great grandson
        \end{verbatim}	    
    */
  //@}
  protected:
      IndiXrefs_t _indi_xrefs;
      std::string _type;
      std::string _rela;
  };
};
#endif
