/*
    Copyright 2002 Cyril Picard

    This file is part of the GEDCOMParser library 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMParser library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMParser library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMParser library ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#ifndef _GEDCOMPARSER_HEADER_HH_
#define _GEDCOMPARSER_HEADER_HH_

#include <string>

#include "GEDCOMParser/AddressManager.hh"
#include "GEDCOMParser/Data.hh"
#include "patterns/SmartPtr.hh"
#include "GEDCOMParser/DateManagement/DateValue.hh"
#include "GEDCOMParser/Visitor/GEDCOMElement.hh"

#define GEDC_VERS "5.5"
#define GEDC_FORM "LINEAGE-LINKED"
#define CHARSET "ASCII"

///
namespace GEDCOMParser
{
  class GEDCOMVisitor;

  /** @memo Implements the GEDCOM 5.5 HEADER structure 
   */
  class Header:public GEDCOMElement, public AddressManager
  {
  protected:
    ///
    std::string _sour;
    ///
    std::string _vers;
    ///
    std::string _name;
    ///
    std::string _corp;
    ///
    SmartPtr < Data > _data;
    ///
    std::string _dest;
    ///
    GEDCOMParser::DateManagement::DateValue * _date;
    ///
    std::string _time;
    ///
    std::string _subm;
    ///
    std::string _subn;
    ///
    std::string _file;
    ///
    std::string _copr;
    ///
    std::string _gedc_vers;
    ///
    std::string _gedc_form;
    ///
    std::string _charset;
    ///
    std::string _charset_vers;
    ///
    std::string _lang;
    ///
    std::string _place_form;
    ///
    std::string _note;
  public:
    ///
    Header (void):_sour (),
      _vers (),
      _name (),
      _corp (),
      _data (),
      _dest (),
      _date (0),
      _time (),
      _subm (),
      _subn (),
      _file (),
      _copr (),
      _gedc_vers (),
      _gedc_form (),
      _charset (), _charset_vers (), _lang (), _place_form (), _note ()
    {
    };
      ///
      Header (Header const &h):_sour (h._sour),
      _vers (h._vers),
      _name (h._name),
      _corp (h._corp),
      _data (h._data),
      _dest (h._dest),
      _date (0),
      _time (h._time),
      _subm (h._subm),
      _subn (h._subn),
      _file (h._file),
      _copr (h._copr),
      _gedc_vers (h._gedc_vers),
      _gedc_form (h._gedc_form),
      _charset (h._charset),
      _charset_vers (h._charset_vers),
      _lang (h._lang), _place_form (h._place_form), _note (h._note)
    {
      if (h._date != 0)
	{
	  _date = new GEDCOMParser::DateManagement::DateValue (*h._date);
	}
      return;
    };
    ///
    virtual ~ Header (void)
    {
      delete _date;
      _date = 0;
    };
    ///@name Accessors (set) 
    //@{    
    ///
    void setSour (std::string const &);
    ///
    void setVers (std::string const &);
    ///
    void setName (std::string const &);
    ///
    void setCorp (std::string const &);
    ///
    GEDCOMParser::Data * setData (Data * const data = 0);
    ///
    void setDest (std::string const &);
    ///
    void setDate (std::string const &);
    ///
    void setTime (std::string const &);
    ///
    void setSubm (std::string const &);
    ///
    void setSubn (std::string const &);
    ///
    void setFile (std::string const &);
    ///
    void setCopr (std::string const &);
    ///
    void setGedcVers (std::string const &);
    ///
    void setGedcForm (std::string const &);
    ///
    void setCharset (std::string const &);
    ///
    void setCharsetVersion (std::string const &);
    ///
    void setLang (std::string const &);
    ///
    void setPlaceForm (std::string const &);
    ///
    void concNote (std::string const &);
    ///
    void contNote (std::string const &);
    ///  Initialize the Header with default data (SOUR, GEDC, ...)
    void setDefaults (void);
    ///
    //@}

    ///@name Accessors (get)
    //@{ 
    virtual std::string const &getSour (void) const;
    ///
    virtual std::string const &getVers (void) const;
    ///
    virtual std::string const &getName (void) const;
    ///
    virtual std::string const &getCorp (void) const;
    ///
    SmartPtr < Data > const &getData (void) const;
    ///
    virtual std::string const &getDest (void) const;
    ///
    virtual std::string const getDate (void) const;
    ///
    GEDCOMParser::DateManagement::
      DateValue const *const getDateObject (void) const;
    ///
    virtual std::string const &getTime (void) const;
    ///
    virtual std::string const &getSubm (void) const;
    ///
    virtual std::string const &getSubn (void) const;
    ///
    virtual std::string const &getFile (void) const;
    ///
    virtual std::string const &getCopr (void) const;
    ///
    virtual std::string const &getGedcVers (void) const;
    ///
    virtual std::string const &getGedcForm (void) const;
    ///
    virtual std::string const &getCharset (void) const;
    ///
    virtual std::string const &getCharsetVersion (void) const;
    ///
    virtual std::string const &getLang (void) const;
    ///
    virtual std::string const &getPlaceForm (void) const;
    ///
    virtual std::string const &getNote (void) const;
    ///
    //@}
    void Accept (GEDCOMParser::GEDCOMVisitor *);
  };
};
#endif
