/* 
    Copyright 2002 Cyril Picard

    This file is part of the GEDCOMParser library 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMParser library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMParser library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMParser library ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GEDCOMParser/Parser.hh"
#include "GEDCOMParser/GEDCOMFactory.hh"
#include "GEDCOMParser/GEDCOMFactoryAccessor.hh"
#include "GEDCOMParser/ParseErrorManagement/ParseErrorManager.hh"

#include <string>
#include <map>
#include <iostream>
#include <algorithm>
#include <functional>
#include <iterator>

#include <getopt.h>
#include <time.h>

#include "gettext.h"
#define _(String) gettext(String)

#define HELP_MESSAGE "Usage : gedcomcheck [--help] [--metrics] [--brief | --verbose] [--version] [filename]"

#define PARSE_ERROR 1

struct arguments
{
  std::string gedcom_filename;
  int metrics;
  int verbose;
  int version;
  int help;
};

int
main (int argc, char **argv)
{
  int res = 0;
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain (PACKAGE);

  struct arguments arguments;

  /* Default values. */
  arguments.verbose = 0;
  arguments.metrics = 0;
  arguments.version = 0;
  arguments.help = 0;
  int c;

  static struct option long_options[] = {
    /* These options set a flag. */
    {"verbose", no_argument, &arguments.verbose, 1},
    {"brief", no_argument, &arguments.verbose, 0},
    {"metrics", no_argument, &arguments.metrics, 1},
    {"version", no_argument, &arguments.version, 1},
    {"help", no_argument, &arguments.help, 1},
    {0, 0, 0, 0}
  };
  /* getopt_long stores the option index here. */
  int option_index = 0;

  c = getopt_long (argc, argv, "vbmVh", long_options, &option_index);
  while (c != -1)
    {
      switch (c)
	{
	case 0:
	  break;
	case 'v':
	  arguments.version = 1;
	  break;
	case 'b':
	  arguments.verbose = 0;
	  break;
	case 'm':
	  arguments.metrics = 1;
	  break;
	case 'V':
	  arguments.verbose = 1;
	  break;
	case 'h':
	  arguments.help = 1;
	  break;
	default:
	  abort ();
	}
      c = getopt_long (argc, argv, "", long_options, &option_index);
    }


  if (arguments.help == 1)
    {
      std::cout << HELP_MESSAGE << std::endl;
    }
  else if (arguments.version == 1)
    {
      std::
	cout << _("gedcomcheck is a part of ") << PACKAGE << " " << VERSION <<
	std::endl;
    }
  else
    {
      GEDCOMParser::GEDCOMFactoryAccessor::Init ();
      GEDCOMParser::GEDCOMFactory * factory =
	GEDCOMParser::GEDCOMFactoryAccessor::getFactory ();
      GEDCOMParser::Parser viewparser (factory);

      if (optind == (argc - 1))
	{
	  arguments.gedcom_filename = argv[optind++];
	}
      else
	{
	  arguments.gedcom_filename = "";
	}
      GEDCOMParser::ParseErrorManagement::
	ParseErrorManager parse_error_manager =
	viewparser.Parse (arguments.gedcom_filename, arguments.verbose);
      if (parse_error_manager.ErrorCount () == 0)
	{
	  std::cout << _("Parsing passed") << std::endl;
	}
      else
	{
	  res = PARSE_ERROR;
	  std::cout << _("Parsing finished with errors : ") << std::endl;
	  for (std::vector <
	       GEDCOMParser::ParseErrorManagement::ParseError >::
	       const_iterator err_iter =
	       parse_error_manager.getErrors ().begin ();
	       err_iter != parse_error_manager.getErrors ().end ();
	       err_iter++)
	    {
	      std::cerr << err_iter->getDescription () << std::endl;
	    }
	}


      GEDCOMParser::LineageLinkageGedcom const *const lineage =
	viewparser.getLineageLinkageGedcom ();

      if (arguments.metrics == 1)
	{
	  if (lineage != 0)
	    {
	      GEDCOMParser::IndividualRecords_t individus =
		lineage->getIndividualRecords ();
	      std::cout << _("Individuals records : ") << individus.
		size () << std::endl;
	      GEDCOMParser::FamilyRecords_t families =
		lineage->getFamilyRecords ();
	      std::cout << _("Family records : ") << families.
		size () << std::endl;
	    }
	  else
	    {
	      std::cerr << _("No transmission found") << std::endl;
	    }
	}
    }
  return res;
}
