; Copyright (c) 2005 by Joseph Crayne
; All rights reserved.
; 
; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions
; are met:
; 1. Redistributions of source code must retain the above copyright
;    notice, this list of conditions and the following disclaimer.
; 2. Redistributions in binary form must reproduce the above copyright
;    notice, this list of conditions and the following disclaimer in 
;    the documentation and/or other materials provided with the
;    distribution.
; 3. The name of the author may not be used to endorse or promote
;    products derived from this software without specific prior
;    written permission.
; 
; THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY
; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
; IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; c-function.scm 
;;; This file creates the language for describing a set of prototypes
;;; for external functions.


(define (string-replace-char! s oc nc)
  (let loop ((i 0))
    (let ((i (string-index s oc i)))
      (cond (i
	     (string-set! s i nc)
	     (loop i))
	    (else s)))))

(define (cname sym)
  (string-replace-char! 
    (string-copy (symbol->string sym))
    #\- #\_)) 

(define (get-address s) 
  (external-value (get-external s)))


;;; This table will hold signatures for glue functions
;;; in case we need to generate them.  A signature is
;;; added when the c-function macro is invoked.
(define missing-glue (make-string-table))

;;; This will become true if we need to generate glue.
(define generate-flag #f)

;;; This table will hold the names of functions declared
;;; with c-function so we can generate an inteface for
;;; a scheme48 structure.
(define c-functions-table (make-symbol-table))

; better to use srfi-1 "any" instead of
; a continuation.
;(define (table-empty? table)
;  (call-with-current-continuation
;    (lambda (return)
;      (table-walk (lambda _ (return #f)) table)
;      (return #t))))



(define (show-missing)
  (table-walk 
    (cut format #t "~s   ~s~%" <> <>)
    missing-glue))

(define (reset-missing)
  (table-walk (lambda (_ lst)
		(for-each force (cadr lst)))
	      missing-glue)
 ; (set! missing-glue (make-string-table))
  )

(define (missing-glue-list)
  (let ((lst '()))
    (table-walk (lambda (n t) 
		  (if (not (null? (car t)))
		    (set! lst 
		      (cons (cons n (car t)) 
			    lst))))
		missing-glue)
    lst))


(define (my-dll fname)
  (with-errno-handler 
    ((errno packet) (else (format #t "missing ~s~%" fname)))
    (dynamic-load fname)))


(define glue-dir #f)

(define (load-glue dir)
  (set! glue-dir dir)
  (my-dll (string-append glue-dir "/glucoscsh.so"))
  (for-each 
    (lambda (glue-file)
      (let ((name (file-name-sans-extension
		    (file-name-nondirectory glue-file))))
	;Assign special value in table to indicate the glue
	;is already generated and available
	(table-set! missing-glue name '( () () ))))
    (glob (string-append glue-dir "/*.o")))
  )

(define (generate-glue)
  (define (create-directory*  d)
    (format #t "creating ~a/~a/.~%" (cwd) d)
    (create-directory d))
  (cond (generate-flag
	  (if (file-not-exists? glue-dir)
	    ;create-directory with parent directories
	    (let mkdir ((lst (split-file-name glue-dir)))
	      (if (not (null? lst))
		(let* ((dir (car lst))
		       (dir (if (equal? dir "") "/" dir)))
		  (if (file-not-exists? dir)
		    (create-directory* dir))
		  (with-cwd dir (mkdir (cdr lst)))))))
	   (with-cwd glue-dir
	     (let ((lst (drop-while
			  (lambda (name-value)
			    (let* ((name (car name-value))
				   (g (cdr name-value))
				   (glue.c (string-append name ".c"))
				   (glue.o (string-append name ".o")))
			      (format #t "Generating module ~a.~%" name)
			      (let ((retv
				      (and (generate-c-module glue.c (list g))
					   (compile-glue glue.c glue.o))))
				(if retv
				  (table-set! missing-glue
					      name
					      '( () () ) ))
				retv)))
			  (missing-glue-list))))
	       (cond ((and (null? lst)
			   (let ((o-files (glob "*.o")))
			     (link-glue o-files "glucoscsh.so")))
		      (dynamic-load "./glucoscsh.so")
		      (lookup-all-externals)))))
	   (reset-missing))))


(define-syntax c-function-internal
  (syntax-rules 
    ()

    ((c-funciton-internal (slot-name ...) types func )
     (define func
       (let* ((glue-name (glue-func-name 'types))
	      (address (get-address (cname 'func)))
	      (the-proc
		(delay
		  (let ((glue (get-external glue-name)))
		    (lambda (slot-name ...)
		      (call-external
			glue
			address
			slot-name ...)))))
	      (bind-it (delay (set! func (force the-proc)))))
	 (define (update-table old-value)
	   (table-set! missing-glue 
		       glue-name
		       (list 'types (cons bind-it (cadr old-value))))
	   (set! generate-flag #t)
	   (lambda a 
	     (generate-glue)
	     (force bind-it)
	     (apply func a)))
	 (if (not glue-dir)
	   (load-glue 
	     (string-append (getenv "HOME")
			    "/.glucoscsh/cache")))
	 (table-set! c-functions-table 'func 1)
	 (cond 
	   ((table-ref missing-glue glue-name)
	    => (lambda (types-bindfuncs)
		 (cond ((null? (car types-bindfuncs))
			;glue is already available.
			(force the-proc))
		       (else
			 ;glue is unavailable but is required
			 ;by another function.  Add our bind-it to
			 ;the list.
			 (update-table types-bindfuncs)))))
	   (else
	     ;glue is unavailable and we are the first function
	     ;to need it.
	     (update-table '(types ()) ))))))


    ((c-function-internal (slot-name ...) (t ...)  func arg_t . tl)
     (c-function-internal (slot-name ... x) (t ... arg_t) func . tl))))
    

(define-syntax c-function
  (syntax-rules ()
		((c-function ret_t func arg_t ...)
		 (c-function-internal () (ret_t) func arg_t ...))))



(define (table-keys tbl)
  (let ((keys '()))
    (table-walk (lambda (k _) 
		  (set! keys (cons k keys)))
		tbl)
    keys))

(define (generate-interface name)
  (let* ((int-name (string-append name "-interface"))
	 (file-name (string-append int-name ".scm"))
	 (funs (table-keys c-functions-table)))
  (with-output-to-file file-name
    (lambda ()
      (write `(define-interface 
		,(string->symbol int-name)
		(export ,@funs)))))))



;(embedded-c (pointer-deref p (size ui) (index ui)) "
;	    char *cp = S48_EXTRACT_VALUE(p,(char*));
;	    return s48_enter_byte_vector( cp+index*size, size );
;	    ")

(define-syntax lambda-external-call
  (syntax-rules ()
		((lambda-external-call name . args )
		 (let ((func (get-external name)))
		   (lambda args
		     (call-external func . args))))))

(define (have-glue? glue-name)
  (let ((g (table-ref missing-glue 
		      glue-name)))
    (and g (null? (car g)))))

(define-syntax embedded-c-internal
  (syntax-rules ()
		((embedded-c-internal args () func)
		   (lambda-external-call (cname 'func) . args))

		((embedded-c-internal x       ((a b) . z ) func)
		 (embedded-c-internal (a . x) z            func))

		((embedded-c-internal x       (y . z)      func)
		 (embedded-c-internal (y . x) z            func))))

(define-syntax embedded-c
  (syntax-rules ()
    ((embedded-c (func . args) body)
     (define func
       (let* ((args2 (map (lambda (x) 
			    (if (list? x)
			      (cons (symbol->string (car x)) 
				    (cdr x))
			      (list (symbol->string x)))
			    ) 
			  'args))
	      (name (cname 'func))
	      (the-proc (delay (embedded-c-internal () args func)))
	      (bind-it (delay (set! func (force the-proc)))))
	 (cond ((have-glue? name) (force the-proc))
	       (else
		 (set! generate-flag #t)
		 (table-set! missing-glue
			     name
			     (list 
			       (make-embedded-function name 
						       args2 
						       body)
			       (list bind-it)))
		 (lambda a
		   (generate-glue)
		   (force bind-it)
		   (apply func a))))) ))))

