; Copyright (c) 2005 by Joseph Crayne
; All rights reserved.
; 
; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions
; are met:
; 1. Redistributions of source code must retain the above copyright
;    notice, this list of conditions and the following disclaimer.
; 2. Redistributions in binary form must reproduce the above copyright
;    notice, this list of conditions and the following disclaimer in 
;    the documentation and/or other materials provided with the
;    distribution.
; 3. The name of the author may not be used to endorse or promote
;    products derived from this software without specific prior
;    written permission.
; 
; THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY
; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
; IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


;;; glue-lang.scm
;;; This file creates the language for describing the
;;; the interface to c.

(define type-table (make-symbol-table))

;;; Read list of type data into our hash table.
(define (read-type-data type-data)
  (let ((table type-table))
    (let loop ((lst type-data))
      (cond ((not (null? lst))
	     (let* ((item (car lst))
		    (t (car item))
		    (tname (cadr item))
		    (marshaller (caddr item)))
	       (table-set! table t (list->vector (cdr item)))
	       (loop (cdr lst)))))
      table)))

(define (tname t)
  (vector-ref (table-ref type-table t) 0))

(define (tmarshall t)
  (vector-ref (table-ref type-table t) 1))

(define (tunmarshall t)
  (vector-ref (table-ref type-table t) 2))


(define (subst-arg str arg)
  (regexp-substitute/global #f (rx (: "$1")) str 'pre arg 'post))

(define (make-extract-macro t name body)
  (string-append "#define " (symbol->string name) "(v) "
		 (subst-arg body "v")))

(define make-enter-macro make-extract-macro)

(define (make-enter-inline t name body)
  (string-append 
    "static inline\n"
    "s48_value " (symbol->string name) "(" (tname t) " v)\n"
    "{\n" (subst-arg body "v") "}\n" ))



(define prelude-code (make-symbol-table))

(define-syntax interfacing-rule
    (syntax-rules (extract enter macro inline)
      ((interfacing-rule t ctype 
	  (extract macro extract-name extract-body)
	  (enter inline enter-name enter-body))
       (begin
	 (table-set! type-table 't (list->vector `(,ctype 
				      ,(symbol->string 'extract-name) 
				      ,(symbol->string 'enter-name) )))
	 (table-set!  prelude-code 't 
	   (string-append
	     (make-extract-macro 't
				 'extract-name
				 'extract-body)
	     "\n"
	     (make-enter-inline 't
				'enter-name
				'enter-body)
	     "\n"))))
      ((interfacing-rule t ctype 
	  (extract macro extract-name extract-body)
	  (enter macro enter-name enter-body))
       (begin
	 (table-set! type-table 't (list->vector `(,ctype 
				      ,(symbol->string 'extract-name) 
				      ,(symbol->string 'enter-name) )))
	 (table-set!  prelude-code 't 
	   (string-append
	     (make-extract-macro 't
				 'extract-name
				 'extract-body)
	     "\n"
	     (make-enter-macro 't
				'enter-name
				'enter-body)
	     "\n"))))))

(define (show-prelude)
  (table-walk (cut format #t "~s =>~%~s~%~%" <> <>) 
	      prelude-code ))

(define (show-types)
  (table-walk (lambda (k v)
		(format #t "~s => (~s ~s ~s)~%"
			k 
			(vector-ref v 0)
			(vector-ref v 1)
			(vector-ref v 2)))
	      type-table))


(define (make-glue-suffix types)
  (string-append
    (string-join 
      (map symbol->string types)
      "_")))

(define (func-typedef-name types)
  (string-append "func_" (make-glue-suffix types) "_t"))

(define (make-func-typedef types)
  (let ((name (func-typedef-name types))
	(ret (tname (car types)))
	(args (map tname (cdr types))))
    (string-concatenate
      `( "typedef " ,ret " " ,name 
	 "(" 
	 ,(string-join args ", ") 
	 ");" ))))


(define (glue-func-name types)
  (string-append "glue_" (make-glue-suffix types)))

(define (make-glue-header types)
  (let* ((name (glue-func-name types))
	 (args (map (cut format #f "s48_value a~a" <>)
		    (iota (length (cdr types)) 1 1)))
	 (args (cons "s48_value func" args)))
    (string-concatenate
      `( "s48_value " ,name
	 "(" 
	 ,(string-join args ", ")
	 ")"))))

(define (make-local-declaration name type c_ )
  (string-append "\t" (tname type) " " c_ name ";\n"))

(define (make-local-vars types)
  (let ((ret_t (car types))
	(func_t (func-typedef-name types))
	(args_t (cdr types)))
    (let* ((ac (if (eqv? ret_t 'v) 
		 '("\t")
		 (list (make-local-declaration "retv" ret_t "c_"))))
	   (ac (cons 
		 (string-append "\t" func_t " *c_func;\n") 
		 ac))
	   (args (map (lambda (t n)
			(make-local-declaration 
			  (string-append "a" (number->string n))
			  t
			  "c_"))
		      args_t
		      (iota (length args_t) 1 1)))
	   (ac (append ac args))
	   (ac (apply string-append ac)))
      ac)))

(define (marshal-argument name type c_ scheme_)
  (let ((arg_name (if (number? name)
		    (string-append "a" (number->string name))
		    name)))
    (string-append 
      "\t" c_ arg_name " = " (tmarshall type) "( " scheme_ arg_name " );\n")))


(define (marshal-arguments types)
  (string-concatenate (map (cut marshal-argument <> <> "c_" "") 
			   (iota (length types) 1 1)
			   types)))


;c_func = S48_EXTRACT_VALUE( func, func_ui_p_t* );
(define (marshal-function types)
  (string-append
    "\tc_func = S48_EXTRACT_VALUE( func, " 
    (func-typedef-name types) "* );"))


;	c_retv = c_func( c_p );


; returns a string contaning c code
; for v-glue-p and such given a list
; of symbols (v p)
(define (make-glue types)
  (define ret_t (car types))
  (define args_t (cdr types))
  (define argnames (map (lambda (n) 
		      (string-append "c_a" (number->string n)))
		    (iota (length args_t) 1 1)))
  (define args (string-join argnames ", "));
  (string-append
    (make-func-typedef types) "\n"
    (make-glue-header types) "\n"
    "{\n"
    (make-local-vars types) "\n"
    (marshal-arguments args_t) 
    (marshal-function types) "\n"
    "\t"
    (if (eqv? ret_t 'v) "" "c_retv = ")
    "c_func( " args " );\n"
    "\n"
    "\treturn " (tunmarshall ret_t) 
    (if (eqv? ret_t 'v) "" "( c_retv )" ) ";\n"
    "}\n"))



(define (make-interface-helpers)
  (define ac "")
  (define (build-string _ str)
    (set! ac (string-append ac str)))
  (table-walk build-string prelude-code)
  ac)


(define (glue-prelude)
    "#include <stddef.h>\n#include <scheme48.h>\n\n")

;(define embedded-c-code '())

(define (make-embedded-function name args body)
  (define (marshal a t)
    (marshal-argument a t "" "_"))
  (define (declare a t)
    (make-local-declaration a t "" ))
  (let ((typed-args (remove (lambda (a) (null? (cdr a)))
			    args)))
    (string-append
      (format #f "s48_value ~a ( " name)
      (string-join (map
		     (lambda (a) (string-append
				   "s48_value "
				   (if (null? (cdr a))
				     (car a)
				     (string-append "_" (car a)))))
		     args)
		   ", ")
      ")\n{\n" 
      (string-concatenate (map (cut apply declare <>) typed-args ))
      (string-concatenate (map (cut apply marshal <>) typed-args ))
      "\t{\n" body "\n\t}\n"
      "}\n")))

(define (for-each-rec func lst)
  (if (not (null? lst))
    (let ((hd (car lst))
	  (tl (cdr lst)))
      (cond ((null? hd))
	    ((list? hd)
	     (for-each-rec func hd))
	    (else (func hd)))
      (for-each-rec func tl))))

(define (table-keys tbl)
  (let ((keys '()))
    (table-walk (lambda (k _) 
		  (set! keys (cons k keys)))
		tbl)
    keys))

(define (generate-c-module fname glue-funcs)
  (let* ((ttable (make-symbol-table)))
    (for-each-rec 
      (lambda (s)
	(if (symbol? s) (table-set! ttable s 1)))
      glue-funcs)
  (with-output-to-file 
    fname
    (lambda ()
      (display (glue-prelude))
      (newline)
      (for-each (lambda (t)
		  (cond ((table-ref prelude-code t)
			 => (lambda (s)
			      (display s)
			      (newline)))))
		(table-keys ttable))
      (for-each (lambda (types)
		  (if (list? types)
		    (display (make-glue types))
		    ;embedded-c code.
		    (display types))
		  (newline))
		glue-funcs)
      (newline)))))


