/* webserver.c - Web server callback routines
 *
 * Copyright (C) 2005-2006   Ivo Clarysse
 *
 * This file is part of GMediaRender.
 *
 * GMediaRender is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GMediaRender is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GMediaRender; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA 02110-1301, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <upnp/upnp.h>
#include <upnp/ithread.h>
#include <errno.h>

typedef struct {
	size_t pos;
	const char *contents;
	size_t len;
} WebServerFile;

struct virtual_file;

static struct virtual_file {
	const char *virtual_fname;
	const char *contents;
	size_t len;
	struct virtual_file *next;
} *virtual_files = NULL;

int webserver_register_buf(const char *path, const char *contents)
{
	struct virtual_file *entry;
	entry = malloc(sizeof(struct virtual_file));
	if (entry == NULL) {
		return -1;
	}
	entry->len = strlen(contents);
	entry->contents = contents;
	entry->virtual_fname = path;
	entry->next = virtual_files;
	virtual_files = entry;
	return 0;
}

int webserver_register_file(const char *path)
{
	char *local_fname;
	struct stat buf;
	struct virtual_file *entry;
	int rc;

	local_fname = strrchr(path, '/') + 1;

	rc = stat(local_fname, &buf);
	if (rc) {
		perror("Stat failed");
		return -1;
	}

	entry = malloc(sizeof(struct virtual_file));
	if (entry == NULL) {
		return -1;
	}
	if (buf.st_size) {
		char *cbuf;
		FILE *in;
		in = fopen(local_fname, "r");
		if (in == NULL) {
			free(entry);
			return -1;
		}
		cbuf = malloc(buf.st_size);
		if (cbuf == NULL) {
			free(entry);
			return -1;
		}
		fread(cbuf, buf.st_size, 1, in);
		fclose(in);
		entry->len = buf.st_size;
		entry->contents = cbuf;

	} else {
		entry->len = 0;
		entry->contents = NULL;
	}
	entry->virtual_fname = path;
	entry->next = virtual_files;
	virtual_files = entry;
	return 0;
}

static int webserver_get_info(const char *filename, struct File_Info *info)
{
	struct virtual_file *virtfile = virtual_files;
	fprintf(stderr, "%s:(filename='%s',info)\n", __FUNCTION__,
		filename);


	while (virtfile != NULL) {
		if (strcmp(filename, virtfile->virtual_fname) == 0) {
			info->file_length = virtfile->len;
			info->last_modified = 0;
			info->is_directory = 0;
			info->is_readable = 1;
			info->content_type =
			    ixmlCloneDOMString("text/xml");
			return 0;
		}
		virtfile = virtfile->next;
	}
	return -1;
}

static UpnpWebFileHandle
webserver_open(const char *filename, enum UpnpOpenFileMode mode)
{
	struct virtual_file *virtfile = virtual_files;
	WebServerFile *file;

	if (mode != UPNP_READ) {
		fprintf(stderr,
			"%s: ignoring request to open file for writing\n",
			filename);
		return NULL;
	}

	while (virtfile != NULL) {
		if (strcmp(filename, virtfile->virtual_fname) == 0) {
			file = malloc(sizeof(WebServerFile));
			file->pos = 0;
			file->len = virtfile->len;
			file->contents = virtfile->contents;
			return file;
		}
		virtfile = virtfile->next;
	}

	return NULL;
}

static inline int min(int a, int b)
{
	return (a<b)?a:b;
}

static int webserver_read(UpnpWebFileHandle fh, char *buf, size_t buflen)
{
	WebServerFile *file = (WebServerFile *) fh;
	ssize_t len = -1;

	len = min(buflen, file->len - file->pos);
	memcpy(buf, file->contents + file->pos, len);

	if (len < 0)
		fprintf(stderr, "%s: cannot read: %s\n", __FUNCTION__,
			strerror(errno));
	else
		file->pos += len;

	return len;
}

static int webserver_write(UpnpWebFileHandle fh, char *buf, size_t buflen)
{
	return -1;
}

static int webserver_seek(UpnpWebFileHandle fh, long offset, int origin)
{
	WebServerFile *file = (WebServerFile *) fh;
	long newpos = -1;

	switch (origin) {
	case SEEK_SET:
		fprintf(stderr, "Attempting to seek to %ld (was at %d)\n",
			offset, file->pos);
		newpos = offset;
		break;
	case SEEK_CUR:
		fprintf(stderr, "Attempting to seek by %ld from %d\n",
			offset, file->pos);
		newpos = file->pos + offset;
		break;
	case SEEK_END:
		fprintf(stderr,
			"Attempting to seek by %ld from end (was at %d)\n",
			offset, file->pos);
		newpos = file->len + offset;
		break;
	}

	if (newpos < 0 || newpos > file->len) {
		fprintf(stderr, "%s: cannot seek: %s\n", __FUNCTION__,
			strerror(EINVAL));
		return -1;
	}

	file->pos = newpos;
	return 0;
}

static int webserver_close(UpnpWebFileHandle fh)
{
	WebServerFile *file = (WebServerFile *) fh;

	free(file);

	return 0;
}

struct UpnpVirtualDirCallbacks virtual_dir_callbacks = {
	webserver_get_info,
	webserver_open,
	webserver_read,
	webserver_write,
	webserver_seek,
	webserver_close
};
