/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* grub.c implements functions to save and restore the configuration of GRUB
 * Include:
 * - restore_grub
 * - install_grub
 * - create_device_map
 * - is_boot_partition
 */

#include "grub.h"
#include "disk.h"

#define _FILE_OFFSET_BITS 64
#include <sys/types.h>

#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <regex.h>

#include <libintl.h>
#include <locale.h>

#define _(x) gettext(x)
#define N_(x) (x)

/* Write in the info_grub variable */
int
write_grub_info (char *device_path, int part_num)
{
	strcpy (info_grub.device_path, device_path);
	info_grub.part_num = part_num;
	info_grub.readed = true;

	return 0;
}

/* It verifies if the partition is a boot partition
 * "Boot partition" means a partition created only for boot files,
 * *NOT* means a partition with boot flag switched on
 * */
bool
is_boot_partition (char *mount_path)
{
	char boot_path[256];
	struct stat filestat;

	sprintf (boot_path, "%s/boot", mount_path);

	if (stat (boot_path, &filestat) < 0)
	{
		return true;
	}
	else //If contains a /boot folder, then is not a boot partition
	{
		return false;
	}
}

/* Create the device.map file content */
char *
create_device_map_content()
{
	FILE *f;
	char line[100], nompt[100];
	char *content=malloc(1);
	int maj, min, tam;
	
	int device_count=0;
	
	strncpy(content, "\0", 1);
	
	if (!(f = fopen ("/proc/partitions", "r")))
	{
		msgerror (_("Error while opening /proc/partitions"));
		return -1;
	}
	
	
	while (fgets (line, sizeof (line), f)) // For each line in /proc/partitions
	{
		regex_t expr;
		// device name not ends with number
		regcomp(&expr, "^[\t ]+[0-9]+[\t ]+[0-9]+[\t ]+[0-9]+[\t ]+([^0-9]+)$", REG_EXTENDED);
		int match=regexec(&expr, line, 0, NULL, 0);
		
		if( match==0) //If the line match
		{
			char path[64];
			char tmp_line[100];
			
			if (sscanf (line, " %d %d %d %[^\n ]", &maj, &min, &tam, nompt) !=
			4)
			continue;
			
			sprintf(&path, "/dev/%s", nompt);
			sprintf(&tmp_line, "(hd%d)\t%s\n", device_count, path);
			
			content=realloc(content, strlen(content)+strlen(tmp_line));
			if(strlen(content)>0)
			{
				strcat(content, tmp_line);
			}
			else
			{
				strcpy(content, tmp_line);
			}
			
			device_count++;
		}
	}
	fclose (f);
	//printf("%s\n", content);
	return content;
}

/* Creates device.map so that grub-install command uses it */
int
create_device_map ()
{
	struct stat filestat;
	
	if (stat ("/boot/grub/device.map", &filestat) < 0) // If device.map doesn't exist
	{
		char *content=create_device_map_content();
		
		mkdir ("/boot", 0755);
		mkdir ("/boot/grub", 0755);
		int fdw = open ("/boot/grub/device.map", O_CREAT | O_WRONLY, 0);		
		write (fdw, content, strlen(content));
		
		close (fdw);
		free(content);

		return 0;
	}
	else
	{
		return 0;
	}
}


/* Install grub on MBR using grub-install command */
int
install_grub ()
{
	bool boot_partition;		/* If the partition where the file “stage2” is located 
								 * is a boot partition or is the root partition */

	create_device_map ();

	char mount_cmdline[256];
	sprintf (mount_cmdline, "mount %s%d /tmp/boot", info_grub.device_path,
			 info_grub.part_num);

	mkdir ("/tmp/boot", 0755);
	if (system (mount_cmdline))
	{
		msgerror (_("Error mounting %s%d"), info_grub.device_path,
				  info_grub.part_num);
		return -1;
	}

	boot_partition = is_boot_partition ("/tmp/boot");

	if (boot_partition)			//GRUB config files are in a boot partition
	{

		char grub_install_cmdline[256];

		sprintf (grub_install_cmdline,
				 "grub-install --root-directory=/tmp %s >/dev/null 2>&1",
				 info_grub.device_path);

		if (system (grub_install_cmdline))
		{
			sprintf (grub_install_cmdline,
				 "grub-install --root-directory=/tmp %s --recheck --force >/dev/null 2>&1",
				 info_grub.device_path);
			if (system (grub_install_cmdline))
			{
				msgerror (_("Error installing GRUB on %s"), info_grub.device_path);
				system ("umount /tmp/boot");
				sleep (1);
				unlink ("/tmp/boot");
				sleep (1);
				return -1;
			}
		}
	}
	else
	{
		char grub_install_cmdline[256];

		sprintf (grub_install_cmdline,
				 "grub-install --root-directory=/tmp/boot %s >/dev/null 2>&1",
				 info_grub.device_path);

		if (system (grub_install_cmdline))
		{
			sprintf (grub_install_cmdline,
				 "grub-install --root-directory=/tmp/boot %s --recheck --force >/dev/null 2>&1",
				 info_grub.device_path);
			
			if (system (grub_install_cmdline))
			{
				msgerror (_("Error installing GRUB on %s"), info_grub.device_path);
				system ("umount /tmp/boot");
				sleep (1);
				unlink ("/tmp/boot");
				sleep (1);
				return -1;
			}
		}
	}

	system ("umount /tmp/boot");
	sleep (1);
	unlink ("/tmp/boot");
	sleep (1);
	return 0;
} 

// Restores grub in a device
int
restore_grub (char *device, char *grub_work_partition)
{
	strcpy (info_grub.device_path, device);
	info_grub.part_num = get_minor (grub_work_partition) % (disk_max_partitions+1);
	info_grub.readed = true;

	printf ("Restoring GRUB... ");
	fflush (stdout);

	if (install_grub () < 0)
	{
		printf ("Error!");
		fflush (stdout);
		msgerror ("Error installing GRUB");
		return -1;
	}
	else
	{
		printf ("OK!");
		fflush(stdout);
	}

	return 0;
}
