/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* image.c implements a set of functions which are used to save all the
 * data in a partition/disk in a image */

#define _FILE_OFFSET_BITS 64
#define _GNU_SOURCE

#include "image.h"
#include "util.h"
#include "disk.h"
#include "slash.h"
#include "net.h"
#include "grub.h"
#include "bootsector.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/vfs.h>
#include <sys/mount.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <parted/parted.h>
#include <dirent.h>
#include <fcntl.h>
#include <string.h>
#include <utime.h>
#include <errno.h>

#include <selinux/selinux.h>

#include <libintl.h>
#include <locale.h>

#define _(x) gettext(x)
#define N_(x) (x)

// For hard links
#define max_hln 8192			// hln = hard links
#define S_IFHLN		9<<12
#define EndOfImage	7<<12

// File type flags:
// 12 S_IFSOCK
// 10 S_IFLNK
// 8  S_IFREG
// 6  S_IFBLK
// 4  S_IFDIR
// 2  S_IFCHR
// 1  S_IFIFO

/* This function verifies that indeed is a dna image */
int
dna_check (const char *arg)
{
	struct image_header header;
	int fd;
	unsigned int nbytes;
	int amount_kbs = 0;
	int i;

	if ((fd = open (arg, O_RDONLY)) < 0)
	{
		msgerror (_("Can't open %s"), arg);
		return -1;
	}

	if (lseek (fd, 0, SEEK_SET) < 0)
	{
		msgerror (_("Error reading %s"), arg);
		return -1;
	}

	if ((nbytes = read (fd, &header, sizeof (struct image_header))) < 0)
	{
		msgerror (_("Can't read %s"), arg);
		return -1;
	}

	if (nbytes != sizeof (struct image_header))
	{
		msgerror (_("%s is not a DNA image."), arg);
		return -1;
	}

	for (i = 0; header.partitions[i].used_part && i < 4; i++)
	{
		amount_kbs += header.partitions[i].min_size;
	}

	if (amount_kbs != header.image_size)
	{
		msgerror (_("%s is not a DNA image."), arg);
		return -1;
	}

	close (fd);

	return 0;
}

/* Net server/Create image		Net client/Restore image *
 *                                                               *
 *   image_header_read  ----------------->  image_header_write   *
 *          dev2fd                              fd2dev           *
 *          x2fd                                fd2x             *
 *          nodes2fd    -------------------->   fd2nodes         *
 *          fat2fd      -------------------->   fd2fat           *
 *                                                               *
 * dev2fd and fd2dev verify if the image is of disc or partition.*
 * x2fd and fd2x find out what filesystem is.                    *
 */

/* Writes all the files in 'path' recursively to a file descriptor (that can be
 * a socket). fd2nodes() can read from that fd and restore all the files. */
int
nodes2fd (const char *path, int fdd)
{
	static int global_did;
	static int current_did;
	static int prefix_size;
	security_context_t context;	// Selinux context of this file

	if (!prefix_size)
		prefix_size = strlen (path);

	int i;
	struct stat filestat;
	char *ab_path, *link_path;
	static unsigned int n_nodes;
	static ino_t nodes[max_hln];
	static char *hard_links[max_hln];
	/* FIXME:
	 * the program will fail in a system with more than max_hln hard links */

	DIR *directory;
	struct dirent *d_file;		// a file in *directory
	struct nodes_file file_nodes;

	if (lstat (path, &filestat) < 0 || !S_ISDIR (filestat.st_mode))
	{
		msgerror (_("Error, invalid path"));
		return -1;
	}

	// write the dir in fd
	strcpy (file_nodes.name, basename (path));
	file_nodes.mode = filestat.st_mode;
	file_nodes.uid = filestat.st_uid;
	file_nodes.gid = filestat.st_gid;
	file_nodes.size = 0;
	file_nodes.atime = filestat.st_atime;
	file_nodes.mtime = filestat.st_mtime;
	//Selinux extended attributes
	if(getfilecon (path, &context)<0)
	{
		strcpy(file_nodes.se_xattr, "");
	}
	else
	{
		strcpy (file_nodes.se_xattr, context);
	}

	global_did++;
	file_nodes.did = current_did;
	file_nodes.other = global_did;
	current_did = file_nodes.other;

	if (write (fdd, &file_nodes, sizeof (struct nodes_file)) < 0)
	{
		msgerror (_("Unable to write on descriptor"));
		return -1;
	}

	// follow down to the directory
	if ((directory = opendir (path)) == NULL)
		return -1;

	while ((d_file = readdir (directory)) != NULL)
	{
		asprintf (&ab_path, "%s/%s", path, d_file->d_name);

		if (lstat (ab_path, &filestat) < 0)
		{
			msgerror (_("Unable to get file information"));
			return -1;
		}

		switch (filestat.st_mode & S_IFMT)
		{
		case S_IFDIR:
			if (strcmp (".", d_file->d_name) && strcmp ("..", d_file->d_name))
			{
				if (nodes2fd (ab_path, fdd) < 0)
					return -1;
				current_did = file_nodes.other;
			}
			break;

		case S_IFREG:
			strcpy (file_nodes.name, basename (ab_path));
			file_nodes.mode = filestat.st_mode;
			file_nodes.uid = filestat.st_uid;
			file_nodes.gid = filestat.st_gid;
			file_nodes.size = filestat.st_size;
			file_nodes.atime = filestat.st_atime;
			file_nodes.mtime = filestat.st_mtime;
			file_nodes.did = current_did;
			//Selinux extended attributes
			if(getfilecon (ab_path, &context)<0)
			{
				strcpy(file_nodes.se_xattr, "");
			}
			else
			{
				strcpy (file_nodes.se_xattr, context);
			}

			if (filestat.st_nlink > 1)
			{
				// file is a hard link
				for (i = 0; i < n_nodes; i++)
					if (filestat.st_ino == nodes[i])
					{
						// this hard link is in the stack, and it's the number i
						file_nodes.mode = (filestat.st_mode & 0x0fff) | S_IFHLN;
						file_nodes.size = strlen (hard_links[i]);
						write (fdd, &file_nodes, sizeof (struct nodes_file));
						write (fdd, hard_links[i], file_nodes.size);
						break;
					}

				if ((n_nodes == i) && (filestat.st_ino != nodes[i]))
				{
					// this hard link wasn't already in the stack
					int fdfile;

					nodes[i] = filestat.st_ino;
					hard_links[i] = malloc (strlen (ab_path + prefix_size) + 1);
					strcpy (hard_links[i], ab_path + prefix_size);
					n_nodes++;

					if ((fdfile = open (ab_path, O_RDONLY)) < 0)
					{
						msgerror (_
								  ("Error while opening a file for read-only"));
						return -1;
					}

					if (write (fdd, &file_nodes, sizeof (struct nodes_file)) <
						0)
					{
						msgerror (_
								  ("Error while writting the file's metadata "
								   "on file/socket"));
						return -1;
					}

					if (lsendfile (fdd, fdfile, file_nodes.size, 0) !=
						file_nodes.size)
					{
						msgerror (_
								  ("Error while writting the image data "
								   "on file/socket"));
						return -1;
					}

					if (close (fdfile) < 0)
					{
						msgerror (_("WARNING: unable to close file"));
					}
				}
			}
			else
			{
				// It's a regular file
				int fdfile;

				if (write (fdd, &file_nodes, sizeof (struct nodes_file)) < 0)
				{
					msgerror (_("Error while writting on file/socket"));
					return -1;
				}

				if ((fdfile = open (ab_path, O_RDONLY)) < 0)
				{
					msgerror (_("Error while opening a file for read-only"));
					return -1;
				}


				if (lsendfile (fdd, fdfile, file_nodes.size, 0) !=
					file_nodes.size)
				{
					msgerror (_
							  ("Error while writting the image data "
							   "on file/socket"));
					return -1;
				}

				if (!quick_mode)
					update_slash (file_nodes.size, 0);

				if (close (fdfile) < 0)
				{
					msgerror (_("WARNING: unable to close file"));
				}
			}
			break;

		case S_IFCHR:
		case S_IFBLK:
			strcpy (file_nodes.name, basename (ab_path));
			file_nodes.mode = filestat.st_mode;
			file_nodes.uid = filestat.st_uid;
			file_nodes.gid = filestat.st_gid;
			file_nodes.size = 0;
			file_nodes.atime = filestat.st_atime;
			file_nodes.mtime = filestat.st_mtime;
			file_nodes.did = current_did;
			file_nodes.other = filestat.st_rdev;
			
			//Selinux extended attributes
			if(getfilecon (ab_path, &context)<0)
			{
				strcpy(file_nodes.se_xattr, "");
			}
			else
			{
				strcpy (file_nodes.se_xattr, context);
			}

			if (write (fdd, &file_nodes, sizeof (struct nodes_file)) < 0)
			{
				msgerror (_("Error while writting meta-data on file/socket"));
				return -1;
			}
			file_nodes.other = current_did;
			break;

		case S_IFLNK:
			strcpy (file_nodes.name, basename (ab_path));

			link_path = malloc (4096);
			if ((file_nodes.size = readlink (ab_path, link_path, 4096)) < 0)
				return -1;

			file_nodes.mode = filestat.st_mode;
			file_nodes.uid = filestat.st_uid;
			file_nodes.gid = filestat.st_gid;
			file_nodes.atime = filestat.st_atime;
			file_nodes.mtime = filestat.st_mtime;
			file_nodes.did = current_did;
			
			//Selinux extended attributes
			if(lgetfilecon (ab_path, &context)<0)
			{
				strcpy(file_nodes.se_xattr, "");
			}
			else
			{
				strcpy (file_nodes.se_xattr, context);
			}

			if (write (fdd, &file_nodes, sizeof (struct nodes_file)) < 0)
			{
				msgerror (_
						  ("Error while writting the image meta-data "
						   "on file/socket"));
				return -1;
			}

			if (write (fdd, link_path, file_nodes.size) < 0)
			{
				msgerror (_
						  ("Error while writting the image meta-data "
						   "on file/socket"));
				return -1;
			}

			free (link_path);
			break;

		case S_IFIFO:
		case S_IFSOCK:
			strcpy (file_nodes.name, basename (ab_path));
			file_nodes.mode = filestat.st_mode;
			file_nodes.uid = filestat.st_uid;
			file_nodes.gid = filestat.st_gid;
			file_nodes.size = 0;
			file_nodes.atime = filestat.st_atime;
			file_nodes.mtime = filestat.st_mtime;
			file_nodes.did = current_did;
			
			//Selinux extended attributes
			if(getfilecon (ab_path, &context)<0)
			{
				strcpy(file_nodes.se_xattr, "");
			}
			else
			{
				strcpy (file_nodes.se_xattr, context);
			}

			if (write (fdd, &file_nodes, sizeof (struct nodes_file)) < 0)
			{
				msgerror (_
						  ("Error while writting the image meta-data "
						   "on file/socket"));
				return -1;
			}
			break;

		default:
			msgerror (_("WARNING: Unknow file type"));
		}
		free (ab_path);
	}

	if (closedir (directory) < 0)
	{
		msgerror (_("WARNING: Unable to close directory"));
	}

	if (current_did == 1)
	{
		// Let's write the bound flag
		file_nodes.mode = EndOfImage;
		file_nodes.did = 0;
		if (write (fdd, &file_nodes, sizeof (struct nodes_file)) < 0)
		{
			msgerror (_("Error while writting the bound flag on descriptor"));
			return -1;
		}

		// Free the mallocs!
		for (i = 0; i < n_nodes; i++)
			free (hard_links[i]);

		// static vars are initialized again
		n_nodes = current_did = global_did = prefix_size = 0;
	}
	return 0;
}

/* This is the opposite funtion of nodes2fd. fd2nodes() reads from 'fdo' the
 * files (written with nodes2fd) and puts them on 'path' */
int
fd2nodes (struct params_dev_fd *params, const char *path)
{
	static unsigned int current_did, readed;
	static unsigned int prefix_size;

	if (!current_did)
	{
		prefix_size = strlen (path);
		readed = 0;
	}

	static struct nodes_file file_nodes;
	char *ab_path, *link_path;

	int fdfile;

	const unsigned int did_local = current_did;
	while (readed
		   ||
		   (get_nbytes
			(params->sockets.fdo, &file_nodes, sizeof (struct nodes_file))) > 0)
	{
		if (params->sockets.fdd && !readed)
		{
			if ((write
				 (params->sockets.fdd, &file_nodes,
				  sizeof (struct nodes_file))) < 0)
			{
				msgerror (_("Error in linker"));
				return -1;
			}
		}

		if (file_nodes.did != did_local)
		{
			// follow down to the parent directory
			readed = 1;
			return 0;
		}
		readed = 0;

		if (!strcmp (file_nodes.name, "stage2")
			|| !strcmp (file_nodes.name, "menu.lst")
			|| (!strcmp (file_nodes.name, "grub.cfg")))
		{
			grub_find = true;
		}

		asprintf (&ab_path, "%s/%s", path, file_nodes.name);

		switch (file_nodes.mode & S_IFMT)
		{
		case S_IFDIR:
		{
			current_did = file_nodes.other;

			if (mkdir (ab_path, 0) < 0)
				if (file_nodes.name[0] != '\0')
				{
					msgerror (_
							  ("Error in DNA image, unable to restore that image"));
					return -1;
				}
			if (chown (ab_path, file_nodes.uid, file_nodes.gid) < 0)
			{
				msgerror (_("Error while setting id's on file %s"), ab_path);
				return -1;
			}
			if (utime (ab_path, (struct utimbuf *) &file_nodes.atime) < 0)
			{
				msgerror (_("Error while setting timestamps on file %s"),
						  ab_path);
				return -1;
			}
			if (chmod (ab_path, file_nodes.mode) < 0)
			{
				msgerror (_("Error while setting permissions on file %s"),
						  ab_path);
				return -1;
			}
			
			if (setfilecon (ab_path, file_nodes.se_xattr) < 0)
			//Restore selinux extended attributes
			{
				msgerror (_
						  ("Error while setting seLinux attributtes on file %s"),
						  ab_path);
				return -1;
			}

			readed = 0;
			if (fd2nodes (params, ab_path) < 0)	// enter the directory
				return -1;
			break;
		}
		case S_IFREG:
		{
			if ((fdfile = open (ab_path, O_WRONLY | O_CREAT, 0)) < 0)
			{
				msgerror (_("Error while creating file %s"), ab_path);
				return -1;
			}
			if (chown (ab_path, file_nodes.uid, file_nodes.gid) < 0)
			{
				msgerror (_("Error while setting id's on file %s"), ab_path);
				return -1;
			}
			if (utime (ab_path, (struct utimbuf *) &file_nodes.atime) < 0)
			{
				msgerror (_("Error while setting timestamps on file %s"),
						  ab_path);
				return -1;
			}
			if (chmod (ab_path, file_nodes.mode) < 0)
			{
				msgerror (_("Error while setting permissions on file %s"),
						  ab_path);
				return -1;
			}
			
			if (setfilecon (ab_path, file_nodes.se_xattr) < 0)
			//Restore selinux extended attributes
			{
				msgerror (_
						  ("Error while setting seLinux attributtes on file %s"),
						  ab_path);
				return -1;
			}

			if (file_nodes.size != 0)
			{
				if (lsendfile
					(fdfile, params->sockets.fdo, file_nodes.size,
					 params->sockets.fdd) < 0)
				{
					msgerror (_("Error while writting data on file %s"),
							  ab_path);
					return -1;
				}

				if (!quick_mode)
					update_slash (file_nodes.size, 0);
			}
			if (close (fdfile) < 0)
			{
				fprintf (stderr, _("WARNING: Unable to close file %s"),
						 ab_path);
			}
			break;
		}
		case S_IFBLK:
		case S_IFCHR:
		{
			if (mknod (ab_path, file_nodes.mode, file_nodes.other) < 0)
			{
				msgerror (_("Unable to create file '%s'"),
						  ab_path + prefix_size);
				return -1;
			}
			if (chown (ab_path, file_nodes.uid, file_nodes.gid) < 0)
			{
				msgerror (_("Error while setting permissions on file %s"),
						  ab_path);
				return -1;
			}
			if (utime (ab_path, (struct utimbuf *) &file_nodes.atime) < 0)
			{
				msgerror (_("Error while setting timestamps on file %s"),
						  ab_path);
				return -1;
			}
			
			if (setfilecon (ab_path, file_nodes.se_xattr) < 0)
			//Restore selinux extended attributes
			{
				msgerror (_
						  ("Error while setting seLinux attributtes on file %s"),
						  ab_path);
				return -1;
			}

			break;
		}
		case S_IFLNK:
		{
			link_path = malloc (file_nodes.size + 1);	// +1 for '\0'
			link_path[file_nodes.size] = '\0';
			if (get_nbytes (params->sockets.fdo, link_path, file_nodes.size) <
				0)
			{
				msgerror (_("Error while getting the link path"));
				return -1;
			}

			if (params->sockets.fdd && !readed)
			{
				if ((write (params->sockets.fdd, link_path, file_nodes.size)) <
					0)
				{
					msgerror (_("Error in linker"));
					return -1;
				}
			}

			if (symlink (link_path, ab_path) < 0)
			{
				msgerror (_("Error while creating the symbolic link %s"),
						  ab_path);
				return -1;
			}
			if (lchown (ab_path, file_nodes.uid, file_nodes.gid) < 0)
			{
				msgerror (_("Error setting permissions on file %s"), ab_path);
				return -1;
			}
			
			if (lsetfilecon (ab_path, file_nodes.se_xattr) < 0)
			//Restore selinux extended attributes
			{
				msgerror (_
						  ("Error while setting seLinux attributtes on file %s"),
						  ab_path);
				return -1;
			}

			free (link_path);
			break;
		}
		case S_IFHLN:
		{
			link_path = malloc (prefix_size + file_nodes.size + 1);	// +1 for '\0'
			link_path = memcpy (link_path, path, prefix_size);

			if (get_nbytes
				(params->sockets.fdo, link_path + prefix_size,
				 file_nodes.size) < 0)
			{
				msgerror (_("Error while getting the link path"));
				return -1;
			}

			if (params->sockets.fdd && !readed)
			{
				if ((write
					 (params->sockets.fdd, link_path + prefix_size,
					  file_nodes.size)) < 0)
				{
					msgerror (_("Error in linker"));
					return -1;
				}
			}

			link_path[file_nodes.size + prefix_size] = '\0';

			if (link (link_path, ab_path) < 0)
			{
				msgerror (_("Error while creating the hard link"));
				return -1;
			}
			
			if (lsetfilecon (ab_path, file_nodes.se_xattr) < 0)
			//Restore selinux extended attributes
			{
				msgerror (_
						  ("Error while setting seLinux attributtes on file %s"),
						  ab_path);
				return -1;
			}

			free (link_path);
			break;
		}
		case S_IFIFO:
		case S_IFSOCK:
		{
			if (mknod (ab_path, file_nodes.mode, 0) < 0)
			{
				msgerror (_("Error while creating fifo/sock file %s"), ab_path);
				return -1;
			}
			if (chown (ab_path, file_nodes.uid, file_nodes.gid) < 0)
			{
				msgerror (_("Error setting permissions on file %s"), ab_path);
				return -1;
			}
			if (utime (ab_path, (struct utimbuf *) &file_nodes.atime) < 0)
			{
				msgerror (_("Error setting timestamps on file %s"), ab_path);
				return -1;
			}
			
			if (setfilecon (ab_path, file_nodes.se_xattr) < 0)
			//Restore selinux extended attributes
			{
				msgerror (_
						  ("Error while setting seLinux attributtes on file %s"),
						  ab_path);
				return -1;
			}

			break;
		}
		case EndOfImage:
		{
			readed = 1;
			free (ab_path);

			// static vars are initialized again
			current_did = prefix_size = 0;
			return 0;
			break;
		}
		default:
		{
			msgerror (_("WARNING: Unknow file type"));
		}
		}
		free (ab_path);
	}
	msgerror (_("This should never be printed."));
	return -1;
}

/* Introduces the data of a FAT partition in the dna image */
int
fat2fd (const char *path, int fdd)
{
	static int global_did;
	static int current_did;
	struct stat filestat;
	char *ab_path;

	DIR *directorio;
	struct dirent *fich;
	struct fat_file file_fat;

	if (lstat (path, &filestat) < 0 || !S_ISDIR (filestat.st_mode))
	{
		msgerror (_("Error, invalid path"));
		return -1;
	}

	// write the directory on descriptor
	strcpy (file_fat.name, basename (path));
	file_fat.mode = filestat.st_mode;
	file_fat.size = 0;
	file_fat.atime = filestat.st_atime;
	file_fat.mtime = filestat.st_mtime;

	global_did++;
	file_fat.did = current_did;
	file_fat.other = global_did;
	current_did = file_fat.other;

	if (write (fdd, &file_fat, sizeof (struct fat_file)) < 0)
	{
		msgerror (_("Unable to write on descriptor"));
		return -1;
	}

	// Follow down to directory
	if ((directorio = opendir (path)) == NULL)
		return -1;

	while ((fich = readdir (directorio)) != NULL)
	{
		asprintf (&ab_path, "%s/%s", path, fich->d_name);

		if (lstat (ab_path, &filestat) < 0)
		{
			msgerror (_("Error while getting file info"));
			return -1;
		}

		switch (filestat.st_mode & S_IFMT)
		{
		case S_IFDIR:
		{
			if (strcmp (".", fich->d_name) && strcmp ("..", fich->d_name))
				if (fat2fd (ab_path, fdd) < 0)
					return -1;
			current_did = file_fat.other;
			break;
		}
		case S_IFREG:
		{
			strcpy (file_fat.name, basename (ab_path));
			file_fat.mode = filestat.st_mode;
			file_fat.size = filestat.st_size;
			file_fat.atime = filestat.st_atime;
			file_fat.mtime = filestat.st_mtime;
			file_fat.did = current_did;

			int fdfile;

			if (write (fdd, &file_fat, sizeof (struct fat_file)) < 0)
			{
				msgerror (_("Unable to write on descriptor"));
				return -1;
			}

			if ((fdfile = open (ab_path, O_RDONLY)) < 0)
			{
				msgerror (_("Unable to read file %s"), ab_path);
				return -1;
			}


			if (lsendfile (fdd, fdfile, file_fat.size, 0) != file_fat.size)
			{
				msgerror (_("Error while copying file data"));
				return -1;
			}

			if (!quick_mode)
				update_slash (file_fat.size, 0);

			if (close (fdfile) < 0)
			{
				msgerror (_("WARNING: Unable to close file"));
			}
			break;
		}
		default:
		{
			msgerror (_("WARNING: Unknown file type"));
		}
		}
		free (ab_path);
	}

	if (closedir (directorio) < 0)
	{
		msgerror (_("WARNING: Unable to close directory"));
	}

	if (current_did == 1)
	{
		// Let's write the bound flag
		file_fat.mode = EndOfImage;
		file_fat.did = 0;
		if (write (fdd, &file_fat, sizeof (struct fat_file)) < 0)
		{
			msgerror (_("Error while writting the bound flag on descriptor"));
			return -1;
		}

		// Initialize again static vars, for the next call to fat2fd
		current_did = global_did = 0;
	}
	return 0;
}

/* This is the opposite funtion of fat2fd. fd2fat() reads from 'fdo' the
 * files (written with fat2fd) and puts them on 'path' */
int
fd2fat (struct params_dev_fd *params, const char *path)
{
	static unsigned int current_did, readed;

	static struct fat_file file_fat;
	char *ab_path;

	if (!current_did)
	{
		// We are in the root
		readed = 0;
	}

	int fdfile;

	const unsigned int did_local = current_did;
	while (readed
		   ||
		   (get_nbytes
			(params->sockets.fdo, &file_fat, sizeof (struct fat_file))) > 0)
	{
		if (params->sockets.fdd && !readed)
		{
			if ((write
				 (params->sockets.fdd, &file_fat,
				  sizeof (struct fat_file))) < 0)
			{
				msgerror (_("Error in linker"));
				return -1;
			}
		}

		if (file_fat.did != did_local)
		{
			// This directory is already restored
			readed = 1;
			return 0;
		}
		readed = 0;

		if (asprintf (&ab_path, "%s/%s", path, file_fat.name) < 0)
			return -1;

		switch (file_fat.mode & S_IFMT)
		{
		case S_IFDIR:
			current_did = file_fat.other;

			if (mkdir (ab_path, 0) < 0)
				if (file_fat.name[0] != '\0')
					return -1;
			if (utime (ab_path, (struct utimbuf *) &file_fat.atime) < 0)
				return -1;
			if (chmod (ab_path, file_fat.mode) < 0)
			{
				msgerror (_("Error while setting permissions on file %s"),
						  ab_path);
				return -1;
			}
			readed = 0;
			if (fd2fat (params, ab_path) < 0)	// enter the directory
				return -1;
			break;

		case S_IFREG:
			if ((fdfile = open (ab_path, O_WRONLY | O_CREAT, 0)) < 0)
				return -1;
			if (utime (ab_path, (struct utimbuf *) &file_fat.atime) < 0)
				return -1;
			if (chmod (ab_path, file_fat.mode) < 0)
			{
				msgerror (_("Error while setting permissions on file %s"),
						  ab_path);
				return -1;
			}
			if (file_fat.size != 0)
			{
				if (lsendfile
					(fdfile, params->sockets.fdo, file_fat.size,
					 params->sockets.fdd) < 0)
					return -1;

				if (!quick_mode)
					update_slash (file_fat.size, 0);
			}
			if (close (fdfile) < 0)
			{
				msgerror (_("WARNING: Unable to close file"));
			}
			break;

		case EndOfImage:
			readed = 1;
			free (ab_path);
			current_did = 0;
			return 0;
			break;

		default:
			msgerror (_("WARNING: %s: Unknown file type.\n"));
		}
		free (ab_path);
	}
	msgerror (_("This should never be printed."));
	return -1;
}

/* Saves the partition information in DNA image. */
int
partition_image_header_write (const char *dev_path, struct image_header *header)
{
	if (read_partition (dev_path, &header->partitions[0]) < 0)
	{
		msgerror (_("Error while reading partition information"));
		return -1;
	}

	header->image_size = header->partitions[0].min_size;
	bzero (&header->partitions[1], sizeof (struct part_info) * 4);
	header->image_type = part;

	return 0;
}

/* Saves the disk partition table in DNA image. */
int
disk_image_header_write (const char *dev_path, struct image_header *header)
{
	int i;

	//header->partitions
	if (read_disk (dev_path, header->partitions) < 0)
	{
		msgerror (_("Error while reading partitions"));
		return -1;
	}

	//header->image_size
	header->image_size = 0;
	for (i = 0; i < 4; i++)
		header->image_size += header->partitions[i].min_size;
	header->image_type = disk;

	//header->mbr
	if (read_disk_mbr (dev_path, header->mbr) < 0)
	{
		msgerror (_("Error reading the disk MBR"));
		return -1;
	}

	//header->no_data
	if (no_data)
	{
		header->no_data = true;
	}
	else
	{
		header->no_data = false;
	}

	return 0;
}

/* Saves the partition table of the disc in DNA image. */
int
image_header_write (const char *dev_path, struct image_header *header)
{
	if (is_disk(dev_path)) //Disk image
	{
		if (disk_image_header_write (dev_path, header) < 0)
		{
			msgerror (_("Error while creating the image header"));
			return -1;
		}
	}
	else				//Partition Image
	{
		if (partition_image_header_write (dev_path, header) < 0)
		{
			msgerror (_("Error while creating the image header"));
			return -1;
		}
	}

	return 0;
}

/* Verifies that the kind of image is correct */
int
partition_image_header_read (const char *dev_path, struct image_header *header)
{
	PedDevice *dev;
	PedDisk *disk;
	PedPartition *part;

	char disk_ab_path[256];
	int part_num = get_minor (dev_path) % (disk_max_partitions+1);
	
	get_disk_path_from_part (dev_path, disk_ab_path);

	if (!(dev = ped_device_get (disk_ab_path)))
	{
		msgerror (_("Unable to open device '%s'"), disk_ab_path);
		return -1;
	}

	if (!(disk = ped_disk_new (dev)))
	{
		msgerror (_("Error while reading partition table on '%s'"), dev_path);
		return -1;
	}

	if (!(part = ped_disk_get_partition (disk, part_num)))
	{
		msgerror (_
				  ("Error while reading the partition no. %i on '%s'"),
				  part_num, dev_path);
		return -1;
	}

	if (is_disk(dev_path))
	{
		msgerror (_
				  ("The origin image is from a partition, the output device"
				   " is a disk.  The output device should be something like"
				   " /dev/hda1, /dev/hdb3, etc."));
		return -1;
	}

	if (write_partition (dev_path, &header->partitions[0]) < 0)
	{
		msgerror (_("Error while creating the file system"));
		return -1;
	}

	if (header->partitions[0].bootable)
	{
		ped_partition_set_flag (part, PED_PARTITION_BOOT, 1);
	}
	if (header->partitions[0].lba)
	{
		ped_partition_set_flag (part, PED_PARTITION_LBA, 1);
	}

	return 0;
}

/* Verifies that the kind of image is correct */
int
disk_image_header_read (const char *dev_path, struct image_header *header)
{
	if (get_minor (dev_path) % (disk_max_partitions+1))
	{
		msgerror (_
				  ("The origin image is a full disk, the output device is "
				   "a partition. The output device should be something "
				   "like /dev/hda, /dev/sdb, etc."));
		return -1;
	}

	if (write_disk (dev_path, header->partitions) < 0)
	{
		msgerror (_("Error while creating partitions"));
		return -1;
	}

	return 0;
}

/* Read the table of partitions saved in image DNA and create a similar one *
 * in the hard disk */
int
image_header_read (char *dev_path, struct image_header *header)
{
	if (header->image_type)		// alone partition
	{
		if (partition_image_header_read (dev_path, header) < 0)
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}
	}
	else						// full disk
	{
		if (disk_image_header_read (dev_path, header) < 0)
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}
	}

	return 0;
}

/* It makes the preparations to send a nodes partition */
int
read_a_nodes_partition (const char *dev, int part_num, int fd,
						enum fs_codes fs_code)
{
	char path[64];
	if (mount_part_n (dev, part_num, fs_code, 0, path) < 0)
	{
		msgerror (_("Unable to mount the partition"));
		return -1;
	}
	strcpy ((char *) &path[strlen (path)], "/");
	printf (_("Reading...\n"));
	fflush(stdout);
	if (nodes2fd (path, fd) < 0)
	{
		msgerror (_("Error while creating the image"));
		umount_part ();
		return -1;
	}
	if (umount_part () < 0)
	{
		msgerror (_("WARNING: Unable to unmount the partition!"));
	}

	return 0;
}

/* This function removes a temporal file of windows */
int
remove_windows_swapfile (char *path, char *file)
{
	char swapfile_path[256];

	strcpy (swapfile_path, path);
	strcat (swapfile_path, file);
	unlink (swapfile_path);

	return 0;
}

/* This function removes a temporal file of windows */
int remove_swapfile(char *path)
{
	remove_windows_swapfile (path, "/pagefile.sys");
	remove_windows_swapfile (path, "/Pagefile.sys");
	remove_windows_swapfile (path, "/PAGEFILE.SYS");

	remove_windows_swapfile (path, "/hiberfil.sys");
	remove_windows_swapfile (path, "/Hiberfil.sys");
	remove_windows_swapfile (path, "/HIBERFIL.SYS");

	remove_windows_swapfile (path, "/windows/win386.swp");
	remove_windows_swapfile (path, "/windows/Win386.swp");
	remove_windows_swapfile (path, "/windows/WIN386.SWP");
	
	return 0;
}

/* It makes the preparations to send a fat partition */
int
read_a_fat_partition (const char *dev, int part_num, int fd,
					  enum fs_codes fs_code)
{
	char path[64];
	if (mount_part_n (dev, part_num, fs_code, 0, path) < 0)
	{
		msgerror (_("Unable to mount the partition"));
		return -1;
	}

	remove_swapfile(path); // This file is unnecessary and huge

	strcpy ((char *) &path[strlen (path)], "/");
	printf (_("Reading...\n"));
	fflush(stdout);
	if (fat2fd (path, fd) < 0)
	{
		msgerror (_("Error while creating the image"));
		umount_part ();
		return -1;
	}
	if (umount_part () < 0)
	{
		msgerror (_("WARNING: Unable to unmount the partition"));
	}

	return 0;
}

/* This function finds out which is the filesystem of the partition * 
 * and calls to the corresponding function */
int
x2fd (const char *dev, int part_num, int fd, enum fs_codes fs_code)
{
	set_fd_is_file ();
#ifdef DEBUG
	printf (_("Adding partition\n"));
	fflush(stdout);
#endif

	if (part_num > 64)
	{
		part_num -= 64;
	}
	printf ("\n>%s%d\n", dev, part_num);
	fflush (stdout);
	
	switch (get_fs_type (fs_code))
	{
	case nodes:
	{
		if (read_a_nodes_partition (dev, part_num, fd, fs_code) < 0)
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}

		break;
	}
	case fat:
	{
		if (read_a_fat_partition (dev, part_num, fd, fs_code) < 0)
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}

		break;
	}
	default:
	{
		msgerror (_("Unknown partition type"));
		return -1;
	}
	}
	return 0;
}

/* It makes the preparations to receive a nodes partition */
int
write_a_nodes_partition (const char *part, struct params_dev_fd *params,
						 enum fs_codes fs_code, unsigned char *bs,
						 unsigned char *label)
{
	char path[64];
	char lost_found[64];
	if (mount_part (part, fs_code, 0, path) < 0)
	{
		msgerror (_("Unable to mount the partition"));
		return -1;
	}
	sprintf (lost_found, "%s/lost+found", path);
	rmdir (lost_found);
	strcpy ((char *) &path[strlen (path)], "/");
	printf (_("Writing...\n"));
	fflush(stdout);
	if (fd2nodes (params, path) < 0)
	{
		msgerror (_("Error while restoring the image"));
		umount_part ();
		return -1;
	}
	if (umount_part () < 0)
	{
		msgerror (_("WARNING: Unable to unmount the partition!"));
	}

	if (fs_code != xfs)
	{
		if (write_bs (part, bs) < 0)
		{
			msgerror (_("WARNING: Error writing boot sector"));
		}
	}

	return 0;
}

/* It makes the preparations to receive a fat partition */
int
write_a_fat_partition (const char *part, struct params_dev_fd *params,
					   enum fs_codes fs_code, unsigned char *bs,
					   unsigned char *label)
{
	char path[64];

	if (fs_code == fat32)
	{
		char ntldr[6];
		strncpy (ntldr, &bs[0x170], 5);
		ntldr[5] = '\0';

		if (!(strcmp (ntldr, "NTLDR")))
		{
			if (write_bs_fatnt (part, bs) < 0)
			{
				msgerror (_("WARNING: Error writing boot sector"));
			}
		}
		else
		{
			if (write_bs_fat32 (part, bs) < 0)
			{
				msgerror (_("WARNING: Error writing boot sector"));
			}

		}
	}

	if (fs_code == ntfs)
	{
		if (write_bs_ntfs (part, bs) < 0)
		{
			msgerror (_("WARNING: Error writing boot sector"));
		}
	}

	if (fs_code == fat16)
	{
		if (write_bs_fat16 (part, bs) < 0)
		{
			msgerror (_("WARNING: Error writing boot sector"));
		}
	}

	if (mount_part (part, fs_code, 0, path) < 0)
	{
		msgerror (_("Unable to mount the partition"));
		return -1;
	}
	strcpy ((char *) &path[strlen (path)], "/");
	printf (_("Writing...\n"));
	fflush(stdout);
	if (fd2fat (params, path) < 0)
	{
		msgerror (_("Error while restoring the image"));
		umount_part ();
		return -1;
	}
	if (umount_part () < 0)
	{
		msgerror (_("WARNING: Unable to unmount the partition"));
	}

	return 0;
}

/* This function finds out which is the filesystem of the partition image* 
 * and calls to the corresponding function */
int
fd2x (const char *part, struct params_dev_fd *params, enum fs_codes fs_code,
	  unsigned char *bs, unsigned char *label)
{
	printf ("\n>%s\n", part);
	fflush (stdout);
	
	switch (get_fs_type (fs_code))
	{
	case nodes:
	{
		if (write_a_nodes_partition (part, params, fs_code, bs, label) < 0)
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}

		break;
	}
	case fat:
	{
		if (write_a_fat_partition (part, params, fs_code, bs, label) < 0)
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}

		break;
	}
	default:
	{
		msgerror (_("Unknown partition type"));
		return -1;
		break;
	}
	}
	return 0;
}

/* This function creates a dna image of a partition */
int
partition2fd (const struct params_dev_fd *params, struct image_header header)
{
	int part_num = get_minor (params->devicepath) % (disk_max_partitions+1);
	if (part_num < 0)
	{
		msgerror (_("Unable to get partition number"));
		return -1;
	}
	char dev_ab_path[256];

	if (get_disk_path_from_part (params->devicepath, dev_ab_path) < 0)
	{
		msgerror (_("Could not determine the disk path"));
		return -1;
	}

	if (x2fd
		(dev_ab_path, part_num, params->sockets.fdd,
		 header.partitions[0].fs_code))
	{
		msgerror (_("Unable to create DNA image"));
		return -1;
	}

	if (!quick_mode)
		update_slash (0, 1);

	return 0;
}

/* This function creates a dna image of a disk */
int
disk2fd (const struct params_dev_fd *params, struct image_header header)
{
	int i;
	for (i = 0; header.partitions[i].used_part || i < 4; i++)
	{
		if (!header.partitions[i].fs_code
			|| header.partitions[i].type == extended)
		{
			continue;
		}

		if (header.partitions[i].fs_code == swap)
		{
			continue;
		}

		if (header.no_data == false)
		{
			if (x2fd (params->devicepath, i + 1, params->sockets.fdd,
					  header.partitions[i].fs_code) < 0)
			{
				msgerror (_("Unable to create DNA image"));
				return -1;
			}
		}
	}

	if (!quick_mode)
		update_slash (0, 1);

	return 0;
}

/* It finds out if the origin is a disc or a partition and calls to *
 * the corresponding function */
int
dev2fd (const struct params_dev_fd *params)
{
	struct image_header header;

	if (image_header_write (params->devicepath, &header) < 0)
	{
		msgerror (_("Error while creating the image header"));
		return -1;
	}

	set_image_size (header.image_size);

	if (write (params->sockets.fdd, &header, sizeof (struct image_header)) < 0)
	{
		msgerror (_("Error while writing on descriptor"));
		return -1;
	}

	if (header.image_type == part)	//partition
	{
		if (partition2fd (params, header) < 0)
		{
			msgerror (_("Error creating a partition image"));
			return -1;
		}
	}
	else						//whole disk
	{
		if (disk2fd (params, header) < 0)
		{
			msgerror (_("Error creating a disk image"));
			return -1;
		}
	}

	return 0;
}

/* This function restores a dna image of a disk */
int
fd2partition (const struct params_dev_fd *params, struct image_header header)
{
	if (fd2x
		(params->devicepath, params,
		 header.partitions[0].fs_code, header.partitions[0].bs,
		 header.partitions[0].label) < 0)
	{
		msgerror (_("Error while copying data"));
		return -1;
	}

	if (!quick_mode)
		update_slash (0, 1);

	return 0;
}

/* This function restores a dna image of a disk */
int
fd2disk (const struct params_dev_fd *params, struct image_header header)
{
	int i;
	char device[256];
	
	system("swapoff -a"); // Disable the swap partition

	for (i = 0; header.partitions[i].used_part || i < 4; i++)
	{
		if (!header.partitions[i].used_part
			|| header.partitions[i].type == extended)
			continue;

		umount_part ();

		/*if (procpt
			(get_major (params->devicepath),
			 (get_minor (params->devicepath) + (i + 1)), partpath, 0) < 0)
		{
			msgerror (_("Error while reading a partition"));
			return -1;
		}

		sprintf (device, "/dev/%s", partpath);*/
		
		get_part_path_from_disk(params->devicepath, i+1, device);

		// a swap partition must be created but not restored
		if (header.partitions[i].fs_code == swap)
		{
			write_label (device, header.partitions[i].fs_code,
						 header.partitions[i].label);

			continue;
		}

		if (header.no_data == false)
		{
			if (fd2x
				(device, params, header.partitions[i].fs_code,
				 header.partitions[i].bs, header.partitions[i].label) < 0)
			{
				msgerror (_("Error while copying data"));
				return -1;
			}
			sync ();
			sleep (1);
		}

		if (grub_find && !info_grub.readed)
		{
			write_grub_info (params->devicepath, i + 1);

			grub_find = false;
		}
		else
		{
			grub_find = false;
		}
	}

	if (!quick_mode)
		update_slash (0, 1);

	if (write_disk_mbr (params->devicepath, header.mbr) < 0)
	{
		msgerror (_("WARNING: Error restoring the disk MBR"));
	}
	
	system("swapon -a"); // Enable the swap partition

	return 0;
}

/* It finds out if the origin is a disc or a partition and calls to *
 * the corresponding function */
int
fd2dev (const struct params_dev_fd *params)
{
	struct image_header header;

	if (get_nbytes (params->sockets.fdo, &header, sizeof (struct image_header))
		< 0)
	{
		msgerror (_("Error reading from file/socket"));
		return -1;
	}

	if (params->sockets.fdd)
	{
		if ((write
			 (params->sockets.fdd, &header, sizeof (struct image_header))) < 0)
		{
			msgerror (_("Error sending the partition table to the next link"));
			return -1;
		}
	}

	set_image_size (header.image_size);
	
	if (check_for_commands (header.partitions) < 0)
	{
		msgerror (_("Some needed command not found\n"));
		return -1;
	}

	if (image_header_read ((char *) params->devicepath, &header) < 0)
	{
		msgerror (_("Error while creating partition table"));
		return -1;
	}

	if (header.image_type == part)	//partition
	{
		if (fd2partition (params, header) < 0)
		{
			msgerror (_("Error restoring a partition image"));
			return -1;
		}
	}
	else						//whole disk
	{
		if (fd2disk (params, header) < 0)
		{
			msgerror (_("Error restoring a disk image"));
			return -1;
		}
	}
	return 0;
}
