/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* util.c implements usefull funtions for make easy the programming 
 * of gnetic.
 * Include:
 * - lsendfile function to send the data from a file descriptor to other.
 * - msgerror function to save the error messages with perror message.
 * - showbt funtion to show the backtrace of all errors happend in execution 
 * of gnetic.
 * These functions are commonly used all files *.c */

#include "util.h"
#include "slash.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>

#include <libintl.h>
#include <locale.h>

#define _(x) gettext(x)
#define N_(x) (x)

unsigned int bufsize = 4096;	//in bytes

/* It's a pointer to the function that is due to use that it can *
 * be recv_waitall or read */
ssize_t (*get_nbytes) (int, void *, size_t) = read;

/* Set the optimal size of buffer */
void
set_optimal_bufsize (unsigned int n)
{
	bufsize = n;
}

/* The recv function with waitall option */
ssize_t
recv_waitall (int s, void *buf, size_t len)
{
	return recv (s, buf, len, MSG_WAITALL);
}

/* Transfer "count" bytes from "in_fd" to "out_fd" */
off_t
lsendfile (int out_fd, int in_fd, off_t count, int string_fd)
{
	char buf[bufsize];
	unsigned int nbytes = bufsize;
	off_t written_bytes = 0;

	if (!count)
	{
		do
		{
			if ((nbytes = get_nbytes (in_fd, buf, bufsize)) <= 0)
			{
				if (nbytes == 0)
					return written_bytes;	// reached EOF
				else if (nbytes < 0)
				{
					msgerror (_("Error while reading from descriptor"));
					return -1;	// Read error
				}
			}

			if (string_fd)
			{
				if ((write (string_fd, buf, nbytes)) < 0)
				{
					msgerror (_("Error in linker"));
					return -1;
				}
			}

			if ((nbytes = write (out_fd, buf, nbytes)) < 0)
			{
				msgerror (_("Error while writing to descriptor"));
				return -1;
			}

			written_bytes += nbytes;
		}
		while (1);
	}
	else
	{
		do
		{
			if ((written_bytes + nbytes) > count)
			{
				// It is necessary to read less than tambuf, the end approaches
				nbytes = count - written_bytes;
			}

			if ((nbytes = get_nbytes (in_fd, buf, nbytes)) <= 0)
			{
				if (nbytes == 0)
					return written_bytes;	// EOF
				else if (nbytes < 0)
				{
					msgerror (_("Error while trying to read from descriptor"));
					return -1;	// read error
				}
			}

			if (string_fd)
			{
				if ((write (string_fd, buf, nbytes)) < 0)
				{
					msgerror (_("Error in linker"));
					return -1;
				}
			}

			if ((nbytes = write (out_fd, buf, nbytes)) < 0)
			{
				msgerror (_("Error while writing to descriptor"));
				return -1;
			}

			written_bytes += nbytes;
		}
		while (written_bytes < count);
	}

	return written_bytes;
}

struct
{
	char src_file[32];
	char function[32];
	int line;
	char message[256];
	int errnum;
} backtrace[32];

/* Make a backtrace of error messages */
int
__msgerror (char *msg, char *src_file, int line, char *function, ...)
{
	static int i;
	va_list args;

	if (i >= 32)
	{
		fprintf (stderr, _("Too much errors!"));
		return -1;
	}

	va_start (args, function);
	__vsnprintf (backtrace[i].message, 256, msg, args);
	va_end (args);

	strcpy (backtrace[i].src_file, src_file);
	strcpy (backtrace[i].function, function);
	backtrace[i].line = line;
	backtrace[i].errnum = errno;

#ifdef DEBUG
	fprintf (stderr, "%s: %s\n",
			 backtrace[i].message, strerror (backtrace[i].errnum));
#endif
	i++;
	return 0;
}

/* Show the backtrace of error messages */
void
showbt ()
{
	int i;
	if (backtrace[0].function[0])
	{
		fprintf (stderr, _("Error, showing debug information...\n"));
		for (i = 0; backtrace[i].function[0]; i++)
			fprintf (stderr, "#%2i: %s():%i: %s (%s)\n", i,
					 backtrace[i].function, backtrace[i].line,
					 backtrace[i].message, strerror (backtrace[i].errnum));
	}
}
