/*
* Copyright (C) 2011 Sunil Mohan Adapa <sunil@medhas.org>.
* Copyright (C) 2011 O S K Chaitanya <osk@medhas.org>.
*
* Author: Sunil Mohan Adapa <sunil@medhas.org>
*         O S K Chaitanya <osk@medhas.org>
*
* This file is part of GNOME Nonogram.
*
* GNOME Nonogram is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* GNOME Nonogram is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with GNOME Nonogram. If not, see <http://www.gnu.org/licenses/>.
*/

const Gtk = imports.gi.Gtk;
const GObject = imports.gi.GObject;

ClueTable = new GType(_ClueTable = {
    parent: Gtk.Table.type,
    name: "ClueTable",

    signals: [{name: "all_clues_satisfied"}],

    _BACKGROUND_COLOR: 0xffffffff,
    _NO_COLOR: 0,

    _solutionPuzzlePixels: null,
    _numPixelRows: null,
    _numPixelColumns: null,

    _clues: null,
    _numClueLines: null,
    _maxClueLineLength: null,
    _clueLineIsSatisfied: [],
    allClueLinesAreSatisfied: false,

    class_init: function(klass, prototype) {
        prototype._BACKGROUND_COLOR = _ClueTable._BACKGROUND_COLOR;
        prototype._NO_COLOR = _ClueTable._NO_COLOR;

        prototype._solutionPuzzlePixels = _ClueTable._solutionPuzzlePixels;
        prototype._numPixelRows = _ClueTable._numPixelRows;
        prototype._numPixelColumns = _ClueTable._numPixelColumns;

        prototype._clues = _ClueTable._clues;
        prototype._numClueLines = _ClueTable._numClueLines;
        prototype._maxClueLineLength = _ClueTable._maxClueLineLength;
        prototype._clueLineIsSatisfied = _ClueTable._clueLineIsSatisfied;
        prototype.allClueLinesAreSatisfied =
            _ClueTable.allClueLinesAreSatisfied;

        prototype.setSolutionPuzzlePixels = _ClueTable.setSolutionPuzzlePixels;
        prototype.reset = _ClueTable.reset;
        prototype._resize = _ClueTable._resize;
        prototype._addLabel = _ClueTable._addLabel;
        prototype._generateClues = _ClueTable._generateClues;
        prototype._generateSingleLineClues =
            _ClueTable._generateSingleLineClues;
        prototype._generateSingleLineCluesOriented =
            _ClueTable._generateSingleLineCluesOriented;
        prototype._populateClues = _ClueTable._populateClues;
        prototype.onPuzzleChanged = _ClueTable.onPuzzleChanged;
        prototype._onPuzzleLinesChanged = _ClueTable._onPuzzleLinesChanged;
        prototype._checkFinish = _ClueTable._checkFinish;
    },

    init: function() {
    },

    setSolutionPuzzlePixels: function(solutionPuzzlePixels) {
        this._solutionPuzzlePixels = solutionPuzzlePixels;
        this._numPixelRows = this._solutionPuzzlePixels.length;
        this._numPixelColumns = this._solutionPuzzlePixels[0].length;

        this._generateClues();

        // Remove existing children
        this.foreach(
            function(child) {
                child.get_parent().remove(child);
            }
        );

        // Find new dimensions
        this._numClueLines = this._clues.length;
        this._maxClueLineLength = 0;
        for (var i = 0; i < this._numClueLines; ++i) {
            if (this._maxClueLineLength < this._clues[i].length)
                this._maxClueLineLength = this._clues[i].length;
        }

        // Reset satisfied flags
        this.reset();

        // Resize
        this._resize();

        // Create new children
        this._populateClues();

        // Show the new children
        this.show_all();
    },

    reset: function() {
        this.allClueLinesAreSatisfied = false;
        this._clueLineIsSatisfied = [];
        for (var i = 0; i < this._numClueLines; ++i) {
            // Rows and columns which are empty should be marked as
            // satisfied by default
            if (this._clues[i].length == 0)
                this._clueLineIsSatisfied.push(true);
            else
                this._clueLineIsSatisfied.push(false);
        }
    },

    _resize: function() {
        // Virtual function: override in derived class
    },

    _addLabel: function(label, clueLineIndex, clueIndex) {
        // Virtual function: override in derived class
    },

    _generateClues: function() {
        // Virtual function: override in derived class
        this._clues = null;
    },

    _generateSingleLineCluesOriented: function () {
        // Virtual function: override in derived class
    },

    _generateSingleLineClues: function(puzzlePixels,
                                       rowStart, columnStart,
                                       rowIncrement, columnIncrement) {
        var singleLineClues = [];
        var currentColor = this._BACKGROUND_COLOR;
        var currentCount = 0;

        for (var x = columnStart, y = rowStart;
            x < this._numPixelColumns && y < this._numPixelRows;
            x += columnIncrement, y += rowIncrement) {
            if (puzzlePixels[y][x] == currentColor) {
                currentCount++;
            }
            else {
                if (currentColor != this._BACKGROUND_COLOR
                    && currentColor != this._NO_COLOR) {
                    singleLineClues.push({color: currentColor,
                                          count: currentCount});
                }

                currentColor = puzzlePixels[y][x];
                currentCount = 1;
            }
        }

        if (currentColor != this._BACKGROUND_COLOR
            && currentColor != this._NO_COLOR) {
            singleLineClues.push({color: currentColor,
                                  count: currentCount});
        }

        return singleLineClues;
    },

    _populateClues: function() {
        for (var clueLineIndex = 0; clueLineIndex < this._numClueLines; ++clueLineIndex) {
            var array = this._clues[clueLineIndex];

            for (var clueIndex = 0; clueIndex < this._maxClueLineLength; ++clueIndex) {
                var labelText = " ";
                var arrayIndex = clueIndex - this._maxClueLineLength + array.length;
                if (arrayIndex >= 0) {
                    labelText = array[arrayIndex].count.toString();

                    var labelColor = array[arrayIndex].color.toString(16);
                    labelColor = "00000000".slice(0, -labelColor.length)
                                    + labelColor;
                    labelColor = labelColor.slice(0, -2);

                    labelText = "<span color='#"
                                    + labelColor
                                    + "'>"
                                    + labelText
                                    + "</span>";
                }

                var label = new Gtk.Label({
                                              label: labelText,
                                              xalign: this.labelTextXAlign,
                                              use_markup: true
                                          });

                this._addLabel(label, clueLineIndex, clueIndex);
            }
        }
    },

    onPuzzleChanged: function(pixelGrid, pixelChanges) {
        // Virtual function: override in derived class
    },

    _onPuzzleLinesChanged: function(puzzlePixels, changedLines) {
        for (var line in changedLines) {
            // Keys of the hash are strings, we need integers
            line = parseInt(line);
            var singleLineClues =
                this._generateSingleLineCluesOriented(puzzlePixels, line);

            this._clueLineIsSatisfied[line] = true;
            if (singleLineClues.length != this._clues[line].length) {
                this._clueLineIsSatisfied[line] = false;
                continue;
            }

            for (var i = 0; i < this._clues[line].length; ++i) {
                if (this._clues[line][i].color != singleLineClues[i].color
                    || this._clues[line][i].count != singleLineClues[i].count) {
                    this._clueLineIsSatisfied[line] = false;
                    break;
                }
            }
        }
    },

    _checkFinish: function() {
        this.allClueLinesAreSatisfied = true;
        for (var i = 0; i < this._clueLineIsSatisfied.length; ++i) {
            if (this._clueLineIsSatisfied[i] == false) {
                this.allClueLinesAreSatisfied = false;
                break;
            }
        }

        if (this.allClueLinesAreSatisfied)
            this.signal.all_clues_satisfied.emit();
    }
});

RowClueTable = new GType(_RowClueTable = {
    parent: ClueTable.type,
    name: "RowClueTable",
    class_init: function(klass, prototype) {
        prototype.labelTextXAlign = 1.0;

        prototype._resize = _RowClueTable._resize;
        prototype._addLabel = _RowClueTable._addLabel;
        prototype._generateClues = _RowClueTable._generateClues;
        prototype._generateSingleLineCluesOriented =
            _RowClueTable._generateSingleLineCluesOriented;
        prototype.onPuzzleChanged = _RowClueTable.onPuzzleChanged;
    },

    _resize: function() {
        this.resize(this._numClueLines, this._maxClueLineLength);
    },

    _addLabel: function(label, clueLineIndex, clueIndex) {
        var x = clueIndex;
        var y = clueLineIndex;

        this.attach(label,
                    x, x + 1,
                    y, y + 1,
                    Gtk.AttachOptions.EXPAND |
                    Gtk.AttachOptions.FILL,
                    Gtk.AttachOptions.EXPAND |
                    Gtk.AttachOptions.FILL,
                    0, 0);
    },

    _generateClues: function() {
        this._clues = [];
        for (var y = 0; y < this._numPixelRows; ++y) {
            var singleRowClues =
                this._generateSingleLineCluesOriented(
                    this._solutionPuzzlePixels, y);

            this._clues.push(singleRowClues);
        }
    },

    _generateSingleLineCluesOriented: function(puzzlePixels, line) {
        return this._generateSingleLineClues(puzzlePixels, line, 0, 0, 1);
    },

    onPuzzleChanged: function(pixelGrid, pixelChanges) {
        var changeArray = pixelChanges.changeArray;

        var changedLines = {};
        changeArray.forEach(
            function (change) {
                changedLines[change.y] = true;
            }
        );

        this._onPuzzleLinesChanged(pixelChanges.puzzlePixels, changedLines);

        this._checkFinish();
    }
});

ColumnClueTable = new GType(_ColumnClueTable = {
    parent: ClueTable.type,
    name: "ColumnClueTable",
    class_init: function(klass, prototype) {
        prototype.labelTextXAlign = 0.5;

        prototype._resize = _ColumnClueTable._resize;
        prototype._addLabel = _ColumnClueTable._addLabel;
        prototype._generateClues = _ColumnClueTable._generateClues;
        prototype._generateSingleLineCluesOriented =
            _ColumnClueTable._generateSingleLineCluesOriented;
        prototype.onPuzzleChanged = _ColumnClueTable.onPuzzleChanged;
    },

    _resize: function() {
        this.resize(this._maxClueLineLength, this._numClueLines);
    },

    _addLabel: function(label, clueLineIndex, clueIndex) {
        var x = clueLineIndex;
        var y = clueIndex;

        this.attach(label,
                    x, x + 1,
                    y, y + 1,
                    Gtk.AttachOptions.EXPAND |
                    Gtk.AttachOptions.FILL,
                    Gtk.AttachOptions.EXPAND |
                    Gtk.AttachOptions.FILL,
                    0, 0);
    },

    _generateClues: function() {
        this._clues = [];
        for (var x = 0; x < this._numPixelColumns; ++x) {
            var singleColumnClues =
                this._generateSingleLineCluesOriented(
                    this._solutionPuzzlePixels, x);

            this._clues.push(singleColumnClues);
        }
    },

    _generateSingleLineCluesOriented: function(puzzlePixels, line) {
        return this._generateSingleLineClues(puzzlePixels, 0, line, 1, 0);
    },

    onPuzzleChanged: function(pixelGrid, pixelChanges) {
        var changeArray = pixelChanges.changeArray;

        var changedLines = {};
        changeArray.forEach(
            function (change) {
                changedLines[change.x] = true;
            }
        );

        this._onPuzzleLinesChanged(pixelChanges.puzzlePixels, changedLines);

        this._checkFinish();
    }
});
