/*
* Copyright (C) 2011 Sunil Mohan Adapa <sunil@medhas.org>.
* Copyright (C) 2011 O S K Chaitanya <osk@medhas.org>.
*
* Author: Sunil Mohan Adapa <sunil@medhas.org>
*         O S K Chaitanya <osk@medhas.org>
*
* This file is part of GNOME Nonogram.
*
* GNOME Nonogram is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* GNOME Nonogram is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with GNOME Nonogram. If not, see <http://www.gnu.org/licenses/>.
*/

const Lang = imports.lang;
const GLib = imports.gi.GLib;
const GObject = imports.gi.GObject;
const Gio = imports.gi.Gio;
const GdkPixbuf = imports.gi.GdkPixbuf;
const Gtk = imports.gi.Gtk;

const Config = imports.config;
const NonogramIO = imports.nonogramio;

DashboardWidget = new GType(_DashboardWidget = {
    parent: Gtk.HPaned.type,
    name: "DashboardWidget",

    signals: [{
        name: "puzzle_activated",
        parameters: [GObject.TYPE_STRING]
    }],

    _GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID: -2,

    _puzzleSetsListStore: null,
    _puzzleSetsTreeView: null,
    _puzzlesListStore: null,
    _puzzlesIconView: null,

    class_init: function(klass, prototype) {
        prototype._GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID =
            _DashboardWidget._GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID;

        prototype._puzzleSetsListStore = _DashboardWidget._puzzleSetsListStore;
        prototype._puzzleSetsTreeView = _DashboardWidget._puzzleSetsTreeView;
        prototype._puzzlesListStore = _DashboardWidget._puzzlesListStore;
        prototype._puzzlesIconView = _DashboardWidget._puzzlesIconView;

        prototype.connectSignalsAndChildren =
            _DashboardWidget.connectSignalsAndChildren;
        prototype._acquireChildren = _DashboardWidget._acquireChildren;
        prototype._onRealize = _DashboardWidget._onRealize;
        prototype._scanForPuzzleSets = _DashboardWidget._scanForPuzzleSets;
        prototype._getFilesInDirectory = _DashboardWidget._getFilesInDirectory;
        prototype._onPuzzleSetSelectionChanged =
            _DashboardWidget._onPuzzleSetSelectionChanged;
        prototype._scanForPuzzles =
            _DashboardWidget._scanForPuzzles;
        prototype._normalizeName = _DashboardWidget._normalizeName;
        prototype._getIndexFromFileName =
            _DashboardWidget._getIndexFromFileName;
        prototype._onPuzzleActivated = _DashboardWidget._onPuzzleActivated;
    },

    init: function() {
        this.signal.realize.connect(Lang.bind(this, this._onRealize));
    },

    connectSignalsAndChildren: function(builder) {
        this._acquireChildren(builder);

        var puzzleSetsSelection = this._puzzleSetsTreeView.get_selection();
        puzzleSetsSelection.set_mode(Gtk.SelectionMode.BROWSE);
        puzzleSetsSelection.signal.changed.connect(
            Lang.bind(this, this._onPuzzleSetSelectionChanged));

        this._puzzlesIconView.signal.item_activated.connect(
            Lang.bind(this, this._onPuzzleActivated));
    },

    _acquireChildren: function(builder) {
        this._puzzleSetsListStore =
            builder.get_object("puzzle_sets_liststore");
        this._puzzleSetsTreeView =
            builder.get_object("puzzle_sets_treeview");

        this._puzzlesListStore =
            builder.get_object("puzzles_liststore");
        this._puzzlesIconView =
            builder.get_object("puzzles_iconview");
    },

    _onRealize: function() {
        this._scanForPuzzleSets(Config.puzzlesDataDir);
    },

    _scanForPuzzleSets: function(directoryPath) {
        this._puzzleSetsListStore.clear();

        var files = this._getFilesInDirectory(directoryPath,
            function (directory, fileInfo) {
                var fileType = fileInfo.get_file_type();
                var accept = false;
                if (fileType == Gio.FileType.DIRECTORY)
                    accept = true;
                return accept;
            }
        );

        this._puzzleSetsListStore.set_sort_column_id(
            this._GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID);

        for (var rowNumber = 0; rowNumber < files.length; ++rowNumber) {
            var displayName = this._normalizeName(files[rowNumber].displayName);

            var newRow = this._puzzleSetsListStore.append();
            NonogramIO.list_store_set_value(this._puzzleSetsListStore,
                                            rowNumber.toString(),
                                            0,
                                            GObject.TYPE_STRING,
                                            displayName);

            NonogramIO.list_store_set_value(this._puzzleSetsListStore,
                                            rowNumber.toString(),
                                            1,
                                            GObject.TYPE_STRING,
                                            files[rowNumber].absolutePath);
            NonogramIO.list_store_set_value(this._puzzleSetsListStore,
                                            rowNumber.toString(),
                                            2,
                                            GObject.TYPE_UINT,
                                            files[rowNumber].index);
        }

        this._puzzleSetsListStore.set_sort_column_id(2,
            Gtk.SortType.ASCENDING);
    },

    _getFilesInDirectory: function (directoryPath, acceptChildCallback) {
        var directory = Gio.file_new_for_path(directoryPath);
        var attributes = Gio.FILE_ATTRIBUTE_STANDARD_DISPLAY_NAME + ","
                       + Gio.FILE_ATTRIBUTE_STANDARD_NAME + ","
                       + Gio.FILE_ATTRIBUTE_STANDARD_TYPE + ","
                       + Gio.FILE_ATTRIBUTE_STANDARD_CONTENT_TYPE;
        var enumerator = directory.enumerate_children(attributes, 0);

        var files = [];
        while (1) {
            var fileInfo = enumerator.next_file();
            if (fileInfo == null)
                break;

            if (typeof(acceptChildCallback) == "function") {
                if (acceptChildCallback(directory, fileInfo) == false)
                    continue;
            }

            var absolutePath = directory.resolve_relative_path(
                fileInfo.get_name()).get_path();

            var index =
                this._getIndexFromFileName(fileInfo.get_display_name());

            var file = {
                name: fileInfo.get_name(),
                displayName: fileInfo.get_display_name(),
                absolutePath: absolutePath,
                index: index
            };
            files.push(file);
        }

        return files;
    },

    _onPuzzleSetSelectionChanged: function(puzzleSetSelection) {
        puzzleSetSelection.selected_foreach(Lang.bind(this,
            function (model, path, iter) {
                var puzzleSetPath = NonogramIO.list_store_get_value(
                    this._puzzleSetsListStore, path.to_string(), 1);
                this._scanForPuzzles(puzzleSetPath);
            })
        );
    },

    _scanForPuzzles: function(puzzleSetPath) {
        this._puzzlesListStore.clear();

        var files = this._getFilesInDirectory(puzzleSetPath,
            function (directoryPath, fileInfo) {
                var accept = Gio.content_type_is_a(fileInfo.get_content_type(),
                                                   "image/*");
                return accept;
            }
        );

        this._puzzlesListStore.set_sort_column_id(
            this._GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID);

        var pixbuf = new GdkPixbuf.Pixbuf.from_file(
            Config.iconsDataDir + "/unsolved-puzzle.png");
        for (var rowNumber = 0; rowNumber < files.length; ++rowNumber) {
            var displayName = this._normalizeName(files[rowNumber].displayName);

            var newRow = this._puzzlesListStore.append();
            NonogramIO.list_store_set_value(this._puzzlesListStore,
                                            rowNumber.toString(),
                                            0,
                                            GObject.TYPE_STRING,
                                            displayName);
            NonogramIO.list_store_set_value(this._puzzlesListStore,
                                            rowNumber.toString(),
                                            1,
                                            GObject.TYPE_STRING,
                                            files[rowNumber].absolutePath);
            NonogramIO.list_store_set_value(this._puzzlesListStore,
                                            rowNumber.toString(),
                                            2,
                                            GdkPixbuf.Pixbuf.type,
                                            pixbuf);
            NonogramIO.list_store_set_value(this._puzzlesListStore,
                                            rowNumber.toString(),
                                            3,
                                            GObject.TYPE_UINT,
                                            files[rowNumber].index);
        }

        this._puzzlesListStore.set_sort_column_id(3, Gtk.SortType.ASCENDING);
    },

    _normalizeName: function(puzzleName) {
        return puzzleName.replace(new RegExp("\\.[^.]*$"), "")
                         .replace(new RegExp("^[0-9]*[_-]+"), "")
                         .replace(new RegExp("[-_]+", "g"), " ")
                         .replace(new RegExp("(\\b)(\\w)", "g"),
                            function(fullMatch, boundaryMatch, letterMatch) {
                                return boundaryMatch + letterMatch.toUpperCase();
                            }
                          );
    },

    _getIndexFromFileName: function(fileName) {
        var match = fileName.match(new RegExp("^(\\d+)"));
        if (match == null)
            return 0;

        return parseInt(match[0], 10);
    },

    _onPuzzleActivated: function(iconview, path) {
        var puzzlePath = NonogramIO.list_store_get_value(this._puzzlesListStore,
                                                         path.to_string(),
                                                         1);

        this.signal.puzzle_activated.emit(puzzlePath);
    }
});
