/*
* Copyright (C) 2011 Sunil Mohan Adapa <sunil@medhas.org>.
* Copyright (C) 2011 O S K Chaitanya <osk@medhas.org>.
*
* Author: Sunil Mohan Adapa <sunil@medhas.org>
*         O S K Chaitanya <osk@medhas.org>
*
* This file is part of GNOME Nonogram.
*
* GNOME Nonogram is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* GNOME Nonogram is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with GNOME Nonogram. If not, see <http://www.gnu.org/licenses/>.
*/

const Lang = imports.lang;
const Gtk = imports.gi.Gtk;
const GtkBuilder = imports.gtkbuilder;

const Config = imports.config;

MainWindow = new GType(_MainWindow = {
    parent: Gtk.Window.type,
    name: "MainWindow",

    _DASHBOARD_VIEW_PAGE: 0,
    _GAME_VIEW_PAGE: 1,

    _aboutDialog: null,
    _dashboardWidget: null,
    _gameWidget: null,
    _viewNotebook: null,
    _gameViewActionGroup: null,
    _undoAction: null,
    _redoAction: null,

    class_init: function(klass, prototype) {
        prototype._DASHBOARD_VIEW_PAGE = _MainWindow._DASHBOARD_VIEW_PAGE;
        prototype._GAME_VIEW_PAGE = _MainWindow._GAME_VIEW_PAGE;

        prototype._aboutDialog = _MainWindow._aboutDialog;
        prototype._dashboardWidget = _MainWindow._dashboardWidget;
        prototype._gameWidget = _MainWindow._gameWidget;
        prototype._viewNotebook = _MainWindow._viewNotebook;
        prototype._gameViewActionGroup = _MainWindow._gameViewActionGroup;
        prototype._undoAction = _MainWindow._undoAction;
        prototype._redoAction = _MainWindow._redoAction;

        prototype.connectSignalsAndChildren =
            _MainWindow.connectSignalsAndChildren;
        prototype._acquireChildren = _MainWindow._acquireChildren;
        prototype._tutorial = _MainWindow._tutorial;
        prototype._helpContents = _MainWindow._helpContents;
        prototype._howToPlay = _MainWindow._howToPlay;
        prototype._openHelp = _MainWindow._openHelp;
        prototype._about = _MainWindow._about;
        prototype._newGame = _MainWindow._newGame;
        prototype._startGame = _MainWindow._startGame;
        prototype._restartGame = _MainWindow._restartGame;
        prototype._switchToDashboardView = _MainWindow._switchToDashboardView;
        prototype._switchToGameView = _MainWindow._switchToGameView;
        prototype._onPuzzleActivated = _MainWindow._onPuzzleActivated;
        prototype._undo = _MainWindow._undo;
        prototype._redo = _MainWindow._redo;
        prototype._onUndoRedoAvailable = _MainWindow._onUndoRedoAvailable;
    },

    init: function() {
    },

    connectSignalsAndChildren: function(builder) {
        var signalHandlers = {
            tutorial: Lang.bind(this, this._tutorial),
            quit: Gtk.main_quit,
            helpContents: Lang.bind(this, this._helpContents),
            howToPlay: Lang.bind(this, this._howToPlay),
            about: Lang.bind(this, this._about),
            newGame: Lang.bind(this, this._newGame),
            restartGame: Lang.bind(this, this._restartGame),
            switchToDashboardView: Lang.bind(this,
                this._switchToDashboardView),
            undo: Lang.bind(this, this._undo),
            redo: Lang.bind(this, this._redo)
        };

        builder.connect_signals(signalHandlers);

        this._acquireChildren(builder);

        this._dashboardWidget.connectSignalsAndChildren(builder);
        this._gameWidget.connectSignalsAndChildren(builder);

        this._dashboardWidget.signal.puzzle_activated.connect(
            Lang.bind(this, this._onPuzzleActivated));
        this._gameWidget.signal.undo_redo_available.connect(
            Lang.bind(this, this._onUndoRedoAvailable));
    },

    _acquireChildren: function(builder) {
        this._viewNotebook = builder.get_object("view_notebook");

        builder.add_from_file(Config.uiDataDir + "/dashboard-view.ui");
        this._dashboardWidget = builder.get_object("dashboardwidget");
        this._viewNotebook.append_page(this._dashboardWidget);

        builder.add_from_file(Config.uiDataDir + "/game-view.ui");
        var gameViewVBox = builder.get_object("game_view_vbox");
        this._viewNotebook.append_page(gameViewVBox);

        this._gameWidget = builder.get_object("gamewidget");

        builder.add_from_file(Config.uiDataDir + "/about-dialog.ui");
        this._aboutDialog = builder.get_object("aboutdialog");
        this._aboutDialog.version = Config.version;
        this._aboutDialog.transient_for = this;

        // Some defaults that could not be set in the ui file
        this._gameViewActionGroup = builder.get_object("game_view_actiongroup");
        var actionList = this._gameViewActionGroup.list_actions();
        actionList.forEach(
            function(action) {
                var actionProxies = action.get_proxies();
                actionProxies.forEach(
                    function(proxyWidget) {
                        proxyWidget.no_show_all = true;
                    }
                );
            }
        );

        this._undoAction = builder.get_object("undo_action");
        this._redoAction = builder.get_object("redo_action");

        // Set white background for entire game area.
        // Viewport is the nearest ancestor for game widget which does
        // not have GTK_NO_WINDOW flag set.
        var viewPort = builder.get_object("game_viewport");
        viewPort.modify_bg(Gtk.StateType.NORMAL,
                           { red: 0xffff, green: 0xffff, blue: 0xffff });

        this._switchToDashboardView();
    },

    _tutorial: function() {
        this._openHelp("tutorial");
    },

    _helpContents: function() {
        this._openHelp();
    },

    _howToPlay: function() {
        this._openHelp();
    },

    _openHelp: function(section) {
        var url = "ghelp:gnome-nonogram";

        if (section != null)
            url += "?" + section;

        try {
            Gtk.show_uri(this.get_screen(), url, Gtk.get_current_event_time());
        } catch (e) {
            print("Unable to open help: " + url);
            print(e.message);
        }
    },

    _about: function() {
        this._aboutDialog.run();
        this._aboutDialog.hide();
    },

    _newGame: function() {
        var fileChooserDialog = new Gtk.FileChooserDialog();
        fileChooserDialog.set_title("Open image file for puzzle");
        fileChooserDialog.add_button("Cancel", Gtk.ResponseType.CANCEL);
        fileChooserDialog.add_button("Open", Gtk.ResponseType.ACCEPT);
        fileChooserDialog.set_action(Gtk.FileChooserAction.OPEN);
        fileChooserDialog.set_transient_for(this);

        var response = fileChooserDialog.run();

        if (response == Gtk.ResponseType.ACCEPT) {
            var filename = fileChooserDialog.get_filename();
            this._startGame(filename);
        }

        fileChooserDialog.destroy();
    },

    _startGame: function(puzzlePath) {
        this._switchToGameView();
        this._gameWidget.puzzleFile = puzzlePath;
    },

    _restartGame: function() {
        this._gameWidget.restart();
    },

    _switchToGameView: function() {
        this._viewNotebook.set_current_page(this._GAME_VIEW_PAGE);
        this._gameViewActionGroup.set_visible(true);
    },

    _switchToDashboardView: function() {
        this._viewNotebook.set_current_page(this._DASHBOARD_VIEW_PAGE);
        this._gameViewActionGroup.set_visible(false);
    },

    _onPuzzleActivated: function(object, puzzleFile) {
        this._startGame(puzzleFile);
    },

    _undo: function() {
        this._gameWidget.undo();
    },

    _redo: function() {
        this._gameWidget.redo();
    },

    _onUndoRedoAvailable: function(gameWidget, undoAvailable, redoAvailable) {
        this._undoAction.set_sensitive(undoAvailable);
        this._redoAction.set_sensitive(redoAvailable);
    }
});
