/*
* Copyright (C) 2011 Sunil Mohan Adapa <sunil@medhas.org>.
* Copyright (C) 2011 O S K Chaitanya <osk@medhas.org>.
*
* Author: Sunil Mohan Adapa <sunil@medhas.org>
*         O S K Chaitanya <osk@medhas.org>
*
* This file is part of GNOME Nonogram.
*
* GNOME Nonogram is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* GNOME Nonogram is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with GNOME Nonogram. If not, see <http://www.gnu.org/licenses/>.
*/

#include <glib.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>
#include <seed.h>

#define SUPPORTED_BITS_PER_SAMPLE 8
#define SUPPORTED_COLORSPACE GDK_COLORSPACE_RGB

static GdkPixbuf * load_puzzle_file(guchar *path,
                                    SeedContext ctx,
                                    SeedException *exception);
static SeedValue puzzle_new_from_pixbuf(const GdkPixbuf *pixbuf,
                                        SeedContext ctx,
                                        SeedException *exception);
static gboolean value_to_gvalue(SeedContext ctx,
                                SeedValue value,
                                GType type,
                                GValue *gvalue,
                                SeedException* exception);
static SeedValue value_from_gvalue(SeedContext ctx,
                                   GValue *gvalue,
                                   SeedException* exception);

SeedValue
load_puzzle(SeedContext ctx,
            SeedObject function,
            SeedObject this_object,
            gsize argument_count,
            const SeedValue arguments[],
            SeedException *exception)
{
  guchar *path;
  GdkPixbuf *pixbuf;
  SeedValue puzzle;

  // FIXME: Use CHECK_ARG_COUNT macro instead.
  // Currently, including seed-module.h results in build failure because
  // of mismatched include path and location of header file.
  if (argument_count != 1)
    {
      seed_make_exception(ctx, exception, "ArgumentError",
                          "Wrong number of arguments; expected %u, got %lu",
                          1, (unsigned long) argument_count);
      return seed_make_undefined(ctx);
    }

  path = seed_value_to_string(ctx, arguments[0], exception);
  if (path == NULL)
    return seed_make_undefined(ctx);

  pixbuf = load_puzzle_file(path, ctx, exception);
  if (pixbuf == NULL)
    return seed_make_undefined(ctx);

  puzzle = puzzle_new_from_pixbuf(pixbuf, ctx, exception);
  if (puzzle == NULL)
    return seed_make_undefined(ctx);

  g_free(path);

  return puzzle;
}

static GdkPixbuf *
load_puzzle_file(guchar * path,
                 SeedContext ctx,
                 SeedException *exception)
{
  GError *error = NULL;
  GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file(path, &error);
  if (pixbuf == NULL)
    {
      char *error_type = "PuzzleLoadFileError";
      if (error->domain == GDK_PIXBUF_ERROR)
        error_type = "PuzzleLoadImageError";

      seed_make_exception(ctx, exception, error_type, "%s", error->message);
      g_error_free(error);
      return NULL;
    }

  return pixbuf;
}

static SeedValue
puzzle_new_from_pixbuf(const GdkPixbuf *pixbuf,
                       SeedContext ctx,
                       SeedException *exception)
{
  int bits_per_sample = gdk_pixbuf_get_bits_per_sample(pixbuf);
  if (bits_per_sample != SUPPORTED_BITS_PER_SAMPLE)
    {
      seed_make_exception(ctx, exception, "PuzzleLoadImageError",
                          "Image has %d bits per sample, %d expected",
                          bits_per_sample, SUPPORTED_BITS_PER_SAMPLE);
      return NULL;
    }

  int num_channels = gdk_pixbuf_get_n_channels(pixbuf);
  if (num_channels != 3 && num_channels != 4)
    {
      seed_make_exception(ctx, exception, "PuzzleLoadImageError",
                          "Image has %d channels, 3 or 4 channels expected",
                          num_channels);
      return NULL;
    }

  GdkColorspace colorspace = gdk_pixbuf_get_colorspace(pixbuf);
  if (colorspace != SUPPORTED_COLORSPACE)
    {
      seed_make_exception(ctx, exception, "PuzzleLoadImageError",
                          "Image colorspace not supported, expected RGB");
      return NULL;
    }

  guchar *pixels = gdk_pixbuf_get_pixels(pixbuf);
  guint width = gdk_pixbuf_get_width(pixbuf);
  guint height = gdk_pixbuf_get_height(pixbuf);
  guint rowstride = gdk_pixbuf_get_rowstride(pixbuf);

  SeedValue puzzle_seed_array;
  SeedValue puzzle[height];
  int x, y;
  for (y = 0; y < height; ++y)
    {
      SeedValue row[width];
      for (x = 0; x < width; ++x)
        {
          guint32 color;
          guint32 r = pixels[y * rowstride + x * num_channels + 0];
          guint32 g = pixels[y * rowstride + x * num_channels + 1];
          guint32 b = pixels[y * rowstride + x * num_channels + 2];
          guint32 a = 255;
          if (num_channels == 4)
            a = pixels[y * rowstride + x * num_channels + 3];

          color = (r << 24) + (g << 16) + (b << 8) + a;

          row[x] = seed_value_from_uint(ctx, color, exception);
          if (row[x] == NULL)
            return NULL;
        }
      puzzle[y] = seed_make_array(ctx, row, width, exception);
      if (puzzle[y] == NULL)
        return NULL;
    }

  puzzle_seed_array = seed_make_array(ctx, puzzle, height, exception);
  return puzzle_seed_array;
}

SeedValue
list_store_set_value(SeedContext ctx,
                     SeedObject function,
                     SeedObject this_object,
                     gsize argument_count,
                     const SeedValue arguments[],
                     SeedException *exception)
{
  // FIXME: Use CHECK_ARG_COUNT macro instead.
  // Currently, including seed-module.h results in build failure because
  // of mismatched include path and location of header file.
  if (argument_count != 5)
    {
      seed_make_exception(ctx, exception, "ArgumentError",
                          "Wrong number of arguments; expected %u, got %lu",
                          5, (unsigned long) argument_count);
      return seed_make_undefined(ctx);
    }

  GtkListStore *list_store;
  list_store = GTK_LIST_STORE(seed_value_to_object(ctx,
                                                   arguments[0],
                                                   exception));
  if (list_store == NULL)
    return seed_make_undefined(ctx);

  gchar *iterString;
  iterString = seed_value_to_string(ctx, arguments[1], exception);
  if (iterString == NULL)
    return seed_make_undefined(ctx);

  guint column;
  column = seed_value_to_uint(ctx, arguments[2], exception);

  GType gtype;
  gtype = seed_value_to_int(ctx, arguments[3], exception);

  GValue *gvalue = g_slice_alloc0(sizeof(GValue));
  gboolean success = value_to_gvalue(ctx, arguments[4], gtype,
                                     gvalue, exception);
  if (success == FALSE)
    return seed_make_undefined(ctx);

  GtkTreeIter iter;
  success = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(list_store),
                                                &iter, iterString);
  if (success == FALSE)
    {
      seed_make_exception(ctx, exception, "ArgumentError",
                          "Invalid iterator string");

      return seed_make_undefined(ctx);
    }

  gtk_list_store_set_value(list_store, &iter, column, gvalue);

  g_slice_free(GValue, gvalue);

  return seed_make_null(ctx);
}

static gboolean
value_to_gvalue(SeedContext ctx,
                SeedValue value,
                GType type,
                GValue *gvalue,
                SeedException* exception)
{
  g_value_init(gvalue, type);

  switch(type)
    {
    case G_TYPE_STRING:
      {
        gchar *string = seed_value_to_string(ctx, value, exception);
        if (string == NULL)
          return FALSE;

        g_value_set_string(gvalue, string);
        return TRUE;
      }
    case G_TYPE_UINT:
      {
        guint ret = seed_value_to_uint(ctx, value, exception);
        g_value_set_uint(gvalue, ret);
        return TRUE;
      }
    }

  if (g_type_is_a(type, G_TYPE_OBJECT))
    {
      GObject *object = seed_value_to_object(ctx, value, exception);
      if (object == NULL)
        return FALSE;

      g_value_set_object(gvalue, object);
      return TRUE;
    }

  seed_make_exception(ctx, exception, "ArgumentError",
                      "Unsupported GValue type");

  return FALSE;
}

SeedValue
list_store_get_value(SeedContext ctx,
                     SeedObject function,
                     SeedObject this_object,
                     gsize argument_count,
                     const SeedValue arguments[],
                     SeedException *exception)
{
  // FIXME: Use CHECK_ARG_COUNT macro instead.
  // Currently, including seed-module.h results in build failure because
  // of mismatched include path and location of header file.
  if (argument_count != 3)
    {
      seed_make_exception(ctx, exception, "ArgumentError",
                          "Wrong number of arguments; expected %u, got %lu",
                          3, (unsigned long) argument_count);
      return seed_make_undefined(ctx);
    }

  GtkListStore *list_store;
  list_store = GTK_LIST_STORE(seed_value_to_object(ctx,
                                                   arguments[0],
                                                   exception));
  if (list_store == NULL)
    return seed_make_undefined(ctx);

  gchar *iterString;
  iterString = seed_value_to_string(ctx, arguments[1], exception);
  if (iterString == NULL)
    return seed_make_undefined(ctx);

  guint column;
  column = seed_value_to_uint(ctx, arguments[2], exception);

  GtkTreeIter iter;
  gboolean success;
  success = gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(list_store),
                                                &iter, iterString);
  if (success == FALSE)
    {
      seed_make_exception(ctx, exception, "ArgumentError",
                          "Invalid iterator string");

      return seed_make_undefined(ctx);
    }

  GValue *gvalue = g_slice_alloc0(sizeof(GValue));
  gtk_tree_model_get_value(GTK_TREE_MODEL(list_store), &iter, column, gvalue);

  SeedValue seed_value = value_from_gvalue(ctx, gvalue, exception);
  if (seed_value == NULL)
    return seed_make_undefined(ctx);

  g_value_unset(gvalue);
  g_slice_free(GValue, gvalue);

  return seed_value;
}

static SeedValue
value_from_gvalue(SeedContext ctx,
                  GValue *gvalue,
                  SeedException* exception)
{
  GType gtype = G_VALUE_TYPE(gvalue);

  switch(gtype)
    {
    case G_TYPE_STRING:
      {
        const gchar *string = g_value_get_string(gvalue);
        return seed_value_from_string(ctx, string, exception);
      }
    case G_TYPE_UINT:
      {
        const guint ret = g_value_get_uint(gvalue);
        return seed_value_from_uint(ctx, ret, exception);
      }
    }

  if (g_type_is_a(gtype, G_TYPE_OBJECT))
    {
      GObject *object = G_OBJECT(g_value_get_object(gvalue));
      return seed_value_from_object(ctx, object, exception);
    }

  seed_make_exception(ctx, exception, "ArgumentError",
                      "Unsupported GValue type");

  return NULL;
}

seed_static_function static_functions[] = {
  {"load_puzzle", load_puzzle, 0},
  {"list_store_set_value", list_store_set_value, 0},
  {"list_store_get_value", list_store_get_value, 0}
};

SeedObject
seed_module_init(SeedEngine *engine)
{
  SeedObject nonogram_io;
  SeedGlobalContext *ctx = engine->context;

  seed_class_definition class_def = seed_empty_class;
  class_def.static_functions = static_functions;

  SeedClass nonogram_io_class = seed_create_class(&class_def);

  nonogram_io = seed_make_object(ctx, nonogram_io_class, NULL);
  seed_value_protect(ctx, nonogram_io);

  return nonogram_io;
}
