# -*- coding: utf-8 -*-

#    This file is part of Gnomolicious.
#
#    Gnomolicious is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    Gnomolicious is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Gnomolicious; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    (C) 2005 Nicolas Évrard <nicoe@altern.org>

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import gnome.applet
import gnome.ui
import gconf
import gobject

import os.path
import re
import logging
import urllib
from time import time as now

import delicious
import conf

GCONF_PATH = "/apps/del.icio.us"
logger = logging.getLogger('gnomolicious')

def findMatchingTag(completion, key_string, cursor, column):
    lastword = key_string.split()[-1]
    model = completion.get_model()
    text = model.get_value(cursor, column)
    return text.startswith(lastword)

class GnomoliciousApplet(gnome.applet.Applet):
    
    title_re = re.compile('<title>(?P<title>.*)</title>',
            re.IGNORECASE|re.DOTALL)
    
    def __init__(self, applet, iid):
        self.last_update = 0

        #############################################
        #        GNOME initialization stuffs        #
        #############################################
        self.__gobject_init__()

        # gconf initilization
        self.conf = gconf.client_get_default()
        self.conf.add_dir(GCONF_PATH, gconf.CLIENT_PRELOAD_RECURSIVE)
        
        # Popup-menu definition and popup-menu callbacks
        self.propxml="""\
<popup name="button3">
  <menuitem name="post" verb="Post" label="P_ost on del.icio.us..."
    pixtype="stock" pixname="gtk-jump-to" />
  <menuitem name="pref" verb="Props" label="_Preferences..."
    pixtype="stock" pixname="gtk-properties" />
  <menuitem name="about" verb="About" label="_About..."
    pixtype="stock" pixname="gnome-stock-about" />
</popup>
"""
        self.verbs = [ ('Props', self.preferences),
                       ('Post', self.post),
                       ('About', self.about) ]
        
        
        # GNOME initialization
        gnome.init('Gnomolicious', conf.version)
        self.applet = applet
        self.applet.drag_dest_set(
                gtk.DEST_DEFAULT_MOTION
                | gtk.DEST_DEFAULT_HIGHLIGHT
                | gtk.DEST_DEFAULT_DROP,
                [ ('text/plain', 0, 1) ],
                gtk.gdk.ACTION_COPY)
        
        # resize_panel creates the self.visual from self.logo
        self.logo = gtk.gdk.pixbuf_new_from_file(
                os.path.join(conf.pixpath, 'gnomolicious.png'))
        self.on_resize_panel(self, self.applet.get_size())
        self.applet.add(self.visual)
        self.applet.show_all()
        
        # connecting signals
        logger.debug('Connecting the signals')
        self.applet.connect("button-press-event", self.on_button_press)
        self.applet.connect("drag-data-received", self.on_dragdata_received)
        self.applet.connect("delete-event", self.cleanup)
        self.applet.connect("change-size", self.on_resize_panel)

        # glade initialization
        wT = gtk.glade.XML(
                os.path.join(conf.gladepath, 'gnomolicious.glade'))
        
        self.postwin = wT.get_widget('window_post')
        self.postwin.entries = {
            'url' : wT.get_widget('entry_url'),
            'desc' : wT.get_widget('entry_desc'),
            'tags' : wT.get_widget('entry_tags'),
            'ext' : wT.get_widget('entry_ext'),
            'popular' : wT.get_widget('populartable') }
        self.populartags_buttons = {}
        self.populartags_nobuttons = gtk.Label(_('No popular tags retrieved for this URL'))
        self.postwin.completion = gtk.EntryCompletion()
        self.postwin.entries['tags'].set_completion(self.postwin.completion)
        self.tags = gtk.ListStore(str, str)
        self.update_tags()
        self.postwin.completion.set_model(self.tags)
        numtags = gtk.CellRendererText()
        numtags.set_property('foreground', 'grey')
        self.postwin.completion.pack_end(numtags)
        self.postwin.completion.add_attribute(numtags, 'text', 1)
        self.postwin.completion.set_text_column(0)
        self.postwin.completion.set_match_func(findMatchingTag, 0)
        self.postwin.completion.set_minimum_key_length(2)
        self.postwin.completion.connect('match-selected',
                self.on_match_selected)
        
        self.prefwin = wT.get_widget('window_pref')
        self.prefwin.entries = {
            'password' : wT.get_widget('entry_password'),
            'username' : wT.get_widget('entry_username') }
        
        wT.signal_autoconnect(self)

    def cleanup(self):
        logger.info('Shuting down gnomolicious')
        del self.applet

    def update_tags(self):
        username = self.conf.get_string(GCONF_PATH + '/username')
        password = self.conf.get_string(GCONF_PATH + '/password')
        logger.debug('Last update: %s' % (now() - self.last_update))
        if username and password and (now() - self.last_update) > 30*60 :
            logger.info('Updating your tags')
            self.last_update = now()
            self.tags.clear()
            taglist = delicious.get_tags(username, password)
            for tagdict in taglist:
                logger.debug('Adding: %s' % tagdict['tag'])
                if int(tagdict['count']) > 1:
                    numtag = '%s occurences' % tagdict['count']
                else:
                    numtag = '%s occurence' % tagdict['count']
                self.tags.append([tagdict['tag'], numtag])
            logger.info('Updating your tags: done')

    def clear_postwin(self):
        self.postwin.entries['url'].set_text('')
        self.postwin.entries['desc'].set_text('')
        self.postwin.entries['tags'].set_text('')
        buf = self.postwin.entries['ext'].get_buffer()
        buf.delete(buf.get_start_iter(), buf.get_end_iter())
        for tag, btn in self.populartags_buttons.items():
            self.postwin.entries['popular'].remove(btn)
        self.populartags_buttons = {}
        self.postwin.entries['popular'].attach(self.populartags_nobuttons,
                0, 15, 0, 5)

    def clear_prefwin(self):
        self.prefwin.entries['password'].set_text('')
        self.prefwin.entries['username'].set_text('')

    def get_poptags(self, url):
        tags = {}
        posts = delicious.get_urlposts(url)
        for post in posts:
            for tag in post['tags'].split():
                if tag in tags:
                    tags[tag] += 1
                else:
                    tags[tag] = 1
        lst = tags.items()
        lst.sort(lambda x,y: cmp(x[1], y[1]))
        lst.reverse()
        return lst

    def add_poptags(self, tags):
        row, col = 0, 0
        if tags:
            self.postwin.entries['popular'].remove(self.populartags_nobuttons)
        for tag, num in tags:
            btn = gtk.CheckButton(tag)
            btn.connect('toggled', self.on_poptoggled, tag)
            self.populartags_buttons[tag] = btn
            self.postwin.entries['popular'].attach(btn, col, col+1,
                    row, row+1)
            col = (col + 1) % 3
            if not col:
                row += 1

    ######################################################
    #               Callbacks for signals                #
    ######################################################
    
    def post(self, evt, data=None):
        self.update_tags()
        self.postwin.entries['popular'].attach(self.populartags_nobuttons,
                0, 14, 0, 5)
        self.postwin.show_all()

    def preferences(self, evt, data=None):
        username = self.conf.get_string(GCONF_PATH + '/username')
        password = self.conf.get_string(GCONF_PATH + '/password')
        self.prefwin.entries['username'].set_text(username)
        self.prefwin.entries['password'].set_text(password)
        self.prefwin.show()

    def about(self, evt, data=None):
        about = gnome.ui.About(_("Gnomolicious"),
                conf.version, "© 2005 Nicolas Évrard",
                _('GNOME applet to post on del.icio.us'),
                ['Nicolas Évrard <nicoe@altern.org>'], [], '', self.logo)
        about.show()

    def on_poptoggled(self, widget, data=None):
        tag = data
        toggled = widget.get_active()
        entry_tags = self.postwin.entries['tags'].get_text().split()
        if tag in entry_tags and not toggled:
            entry_tags.remove(tag)
            self.postwin.entries['tags'].set_text(' '.join(entry_tags))
        elif tag not in entry_tags and toggled:
            entry_tags.append(tag)
            self.postwin.entries['tags'].set_text(' '.join(entry_tags))

    def on_resize_panel(self, widget, data):
        logger.debug('New size: %s' % data)
        pixbuf = self.logo
        new_pixbuf = pixbuf.scale_simple(data-1, data-1,
                gtk.gdk.INTERP_BILINEAR)
        self.visual = gtk.Image()
        self.visual.set_from_pixbuf(new_pixbuf)
        
    def on_dragdata_received(self, widget, context, x, y, selection, ttype,
            time):
        url = selection.data
        logger.info('Received url: %s' % url)
        txt = urllib.urlopen(url).read()
        title = self.title_re.search(txt).groupdict()['title']
        title = ' '.join(title.split())
        self.postwin.entries['url'].set_text(url)
        self.postwin.entries['desc'].set_text(title)
        self.update_tags()
        logger.info('Retrieving popular tags')
        poptags = self.get_poptags(url)
        self.add_poptags(poptags[:15])
        self.postwin.show_all()

    def on_button_press(self, widget, evt):
        if evt.type == gtk.gdk.BUTTON_PRESS and evt.button == 3:
            self.applet.setup_menu(self.propxml, self.verbs, None)

    def on_post(self, evt, data=None):
        logger.info('Posting on del.icio.us')
        url = self.postwin.entries['url'].get_text()
        desc = self.postwin.entries['desc'].get_text()
        tags = self.postwin.entries['tags'].get_text()
        buff = self.postwin.entries['ext'].get_buffer()
        ext = buff.get_text(buff.get_start_iter(), buff.get_end_iter())
        logger.info('URL: %s' % url)
        logger.info('Tags: %s' % tags)
        username = self.conf.get_string(GCONF_PATH + '/username')
        password = self.conf.get_string(GCONF_PATH + '/password')
        delicious.add(username, password, url, desc, ext, tags)
        logger.info('Posting done')
        self.postwin.hide()
        self.clear_postwin()
        return gtk.TRUE

    def on_setpref(self, evt, data=None):
        username = self.prefwin.entries['username'].get_text()
        self.conf.set_string(GCONF_PATH + '/username', username)
        password = self.prefwin.entries['password'].get_text()
        self.conf.set_string(GCONF_PATH + '/password', password)
        self.conf.suggest_sync()
        self.prefwin.hide()
        return gtk.TRUE

    def on_postwin_delete_event(self, widget, evt):
        self.postwin.hide()
        self.clear_postwin()
        return gtk.TRUE

    def on_prefwin_delete_event(self, widget, evt):
        self.prefwin.hide()
        return gtk.TRUE
    
    def on_cancelpost(self, evt, data=None):
        self.postwin.hide()
        self.clear_postwin()
        return gtk.TRUE

    def on_match_selected(self, completion, model, cursor):
        match = model[cursor][0]
        etokens = self.postwin.entries['tags'].get_text().split()
        etokens[-1] = match
        self.postwin.entries['tags'].set_text(' '.join(etokens))
        return gtk.TRUE

    def on_postwin_tags(self, widget, event):
        if event.string == ' ':
            tags = self.postwin.entries['tags'].get_text().split()
            for tag in tags:
                if tag in self.populartags_buttons:
                    btn = self.populartags_buttons[tag]
                    btn.set_active(True)
        return gtk.FALSE
