# -*- coding: utf-8 -*-

#    This file is part of Gnomolicious.
#
#    Gnomolicious is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    Gnomolicious is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Gnomolicious; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#    (C) 2005 Nicolas Évrard <nicoe@nutellux.be>

__revision__ = "$Id: gnomolicious.py,v 1.20 2006/02/28 22:15:27 nicoe Exp $"

import os.path
import re
import logging
import urllib
import ConfigParser
import string
import datetime

import pygtk
pygtk.require('2.0')
import gtk
import gtk.glade
import gnomeapplet
import gnome.ui
import gconf
import gobject

import delicious
import conf

logger = logging.getLogger('gnomolicious')
VALID_CHAR = string.ascii_letters + string.punctuation + string.digits

def findMatchingTag(completion, key_string, cursor, column):
    lastword = key_string.split()[-1]
    model = completion.get_model()
    text = model.get_value(cursor, column)
    return text.startswith(lastword)

class GnomoliciousConfiguration(object):

    GCONF_PATH = "/apps/gnomolicious"
    PASSWORD_PATH = os.path.expanduser('~/.gnome2_private/gnomolicious')

    def __init__(self):
        self.privconf = ConfigParser.ConfigParser()
        self.privconf.read([self.PASSWORD_PATH])

        self.gconf = gconf.client_get_default()
        self.gconf.add_dir(self.GCONF_PATH, gconf.CLIENT_PRELOAD_RECURSIVE)

    def getUsername(self):
        try:
            return self.privconf.get('delicious', 'username')
        except (ConfigParser.NoSectionError, ConfigParser.NoOptionError):
            return ''

    def setUsername(self, user):
        if not self.privconf.has_section('delicious'):
            self.privconf.add_section('delicious')
        self.privconf.set('delicious', 'username', user)
        self.privconf.write(open(self.PASSWORD_PATH, 'w'))

    username = property(getUsername, setUsername)

    def getPassword(self):
        try:
            return self.privconf.get('delicious', 'password')
        except (ConfigParser.NoSectionError, ConfigParser.NoOptionError):
            return ''

    def setPassword(self, password):
        if not self.privconf.has_section('delicious'):
            self.privconf.add_section('delicious')
        self.privconf.set('delicious', 'password', password)
        self.privconf.write(open(self.PASSWORD_PATH, 'w'))

    password = property(getPassword, setPassword)

    def getCommonTagColor(self):
        return self.gconf.get_string(self.GCONF_PATH + '/common_tag_color')

    def setCommonTagColor(self, color):
        self.gconf.set_string(self.GCONF_PATH + '/common_tag_color',
                              color)
        self.gconf.suggest_sync()

    common_tag_color = property(getCommonTagColor, setCommonTagColor)

    def getPopTagNum(self):
        return self.gconf.get_int(self.GCONF_PATH + '/pop_tag_num')

    def setPopTagNum(self, num):
        self.gconf.set_int(self.GCONF_PATH + '/pop_tag_num', num)
        self.gconf.suggest_sync()

    pop_tag_num = property(getPopTagNum, setPopTagNum)

class GnomoliciousApplet(gnomeapplet.Applet):

    charset_re = re.compile('<meta[^>]*content="[^"]*charset=(?P<enc>[\w-]*)"',
                            re.IGNORECASE|re.DOTALL)
    title_re = re.compile('<title>(?P<title>.*?)</title>',
                          re.IGNORECASE|re.DOTALL)

    def __init__(self, applet, iid):
        self._last = datetime.datetime.min
        self.conf = GnomoliciousConfiguration()
        self.delicious_user = delicious.User(self.conf.username,
                                             self.conf.password)

        #############################################
        #        GNOME initialization stuffs        #
        #############################################
        self.__gobject_init__()

        # Popup-menu definition and popup-menu callbacks
        self.propxml = """\
                <popup name="button3">
        <menuitem name="post" verb="Post" label="%s"
        pixtype="stock" pixname="gtk-jump-to" />
        <menuitem name="pref" verb="Props" label="%s"
        pixtype="stock" pixname="gtk-properties" />
        <menuitem name="about" verb="About" label="%s"
        pixtype="stock" pixname="gnome-stock-about" />
        </popup>
        """ % (_('P_ost on del.icio.us...'), _('_Preferences...'),
               _('_About...'))
        self.verbs = [ ('Props', self.preferences),
                      ('Post', self.post),
                      ('About', self.about) ]


        # GNOME initialization
        gnome.init('Gnomolicious', conf.version)
        self.applet = applet
        self.applet.drag_dest_set(
                                  gtk.DEST_DEFAULT_MOTION
                                  | gtk.DEST_DEFAULT_HIGHLIGHT
                                  | gtk.DEST_DEFAULT_DROP,
                                  [ ('text/plain', 0, 1) ],
                                  gtk.gdk.ACTION_COPY)

        # resize_panel creates the self.visual from self.logo
        self.logo = gtk.gdk.pixbuf_new_from_file(os.path.join(conf.pixpath,
                                                              'gnomolicious.png'
                                                              ))
        self.on_resize_panel(self, self.applet.get_size())

        # connecting signals
        logger.debug('Connecting the signals')
        self.applet.connect("button-press-event", self.on_button_press)
        self.applet.connect("drag-data-received", self.on_dragdata_received)
        self.applet.connect("delete-event", self.cleanup)
        self.applet.connect("change-size", self.on_resize_panel)
        self.applet.connect("change-background", self.on_bckg_change)

        # glade initialization
        wT = gtk.glade.XML(
                           os.path.join(conf.gladepath, 'gnomolicious.glade'))

        self.postwin = wT.get_widget('window_post')
        self.postwin.entries = {
                                'url' : wT.get_widget('entry_url'),
                                'desc' : wT.get_widget('entry_desc'),
                                'tags' : wT.get_widget('entry_tags'),
                                'ext' : wT.get_widget('entry_ext'),
                                'popular' : wT.get_widget('populartable') }
        
        self.populartags_buttons = {}
        self.populartags_nobuttons = gtk.Label(\
                                _('No popular tags retrieved for this URL'))
        popular = self.postwin.entries['popular']
        popular.attach(self.populartags_nobuttons, 0, 3, 0,
                       (self.conf.pop_tag_num - 1) / 3 + 1, 2, 2)
        
        self.postwin.completion = gtk.EntryCompletion()
        self.postwin.entries['tags'].set_completion(self.postwin.completion)
        self.tags = gtk.ListStore(str, str)
        # TODO: prefetch tag ? can slow gnome startup when del.icio.us is
        # slow
        # self.update_tags()
        self.postwin.completion.set_model(self.tags)
        numtags = gtk.CellRendererText()
        numtags.set_property('foreground', 'grey')
        self.postwin.completion.pack_end(numtags)
        self.postwin.completion.add_attribute(numtags, 'text', 1)
        self.postwin.completion.set_text_column(0)
        self.postwin.completion.set_match_func(findMatchingTag, 0)
        self.postwin.completion.set_minimum_key_length(2)
        self.postwin.completion.connect('match-selected',
                                        self.on_match_selected)

        self.prefwin = wT.get_widget('window_pref')
        self.prefwin.entries = {'password' : wT.get_widget('entry_password'),
                                'username' : wT.get_widget('entry_username'),
                                'commontagcolor' : \
                                    wT.get_widget('entry_commontagcolor'),
                                'maxpoptag' : wT.get_widget('entry_numpoptag'),
                                }

        self.update_dialog = gtk.MessageDialog(parent = None,
                                               type = gtk.MESSAGE_INFO,
                                               message_format = _('Updating the list of your tags'))

        wT.signal_autoconnect(self)

    def cleanup(self):
        logger.info('Shuting down gnomolicious')
        del self.applet

    def update_tags(self):
        if (self.conf.username and self.conf.password) \
           and ((datetime.datetime.today() - self._last).seconds >= 900
                or not list(self.tags)):
            self._last = datetime.datetime.today()
            logger.info('Updating your tags')
            self.tags.clear()
            taglist = self.delicious_user.tags
            for tagdict in taglist:
                logger.debug('Adding: %s' % tagdict['tag'])
                if int(tagdict['count']) > 1:
                    numtag = '%s occurences' % tagdict['count']
                else:
                    numtag = '%s occurence' % tagdict['count']
                self.tags.append([tagdict['tag'], numtag])
            logger.info('Updating your tags: done')

    def clear_postwin(self):
        self.postwin.entries['url'].set_text('')
        self.postwin.entries['desc'].set_text('')
        self.postwin.entries['tags'].set_text('')
        buf = self.postwin.entries['ext'].get_buffer()
        buf.delete(buf.get_start_iter(), buf.get_end_iter())
        for tag, btn in self.populartags_buttons.items():
            self.postwin.entries['popular'].remove(btn)
            self.populartags_buttons = {}
            popular = self.postwin.entries['popular']
            popular.attach(self.populartags_nobuttons, 0, 3, 0,
                           (self.conf.pop_tag_num - 1) / 3 + 1, 2, 2)

    def clear_prefwin(self):
        pass

    def get_poptags(self, url):
        webpage = delicious.WebPage(url)
        lst = webpage.tags.items()
        lst.sort(lambda x, y: cmp(x[1], y[1]))
        lst.reverse()
        return lst

    def add_poptags(self, tags):
        row, col = 0, 0
        my_tags = [x[0] for x in self.tags]
        if tags:
            self.postwin.entries['popular'].remove(self.populartags_nobuttons)
        for tag, num in tags:
            if tag in my_tags:
                color = self.conf.common_tag_color
            else:
                color = "black"
            btn = gtk.CheckButton(tag, False)
            label = btn.get_children()[0]
            label.set_markup('<span foreground="%s">%s</span>' % (color, tag))
            btn.connect('toggled', self.on_poptoggled, tag)
            self.populartags_buttons[tag] = btn
            self.postwin.entries['popular'].attach(btn, col, col+1, row, row+1)
            col = (col + 1) % 3
            if not col:
                row += 1

    ######################################################
    #               Callbacks for signals                #
    ######################################################

    def post(self, evt, data=None):
        self.update_tags()
        self.postwin.entries['popular'].attach(self.populartags_nobuttons,
                                               0, 14, 0, 5)
        self.postwin.show_all()

    def preferences(self, evt, data=None):
        username = self.conf.username
        password = self.conf.password
        if username:
            self.prefwin.entries['username'].set_text(username)
        if password:
            self.prefwin.entries['password'].set_text(password)
        self.prefwin.entries['commontagcolor'].set_color(gtk.gdk.color_parse(self.conf.common_tag_color))
        self.prefwin.entries['maxpoptag'].set_value(self.conf.pop_tag_num)
        self.prefwin.show()

    def about(self, evt, data=None):
        about = gnome.ui.About("Gnomolicious",
                               conf.version, "© 2005 Nicolas Évrard",
                               _('GNOME applet to post on del.icio.us'),
                               ['Nicolas Évrard <nicoe@nutellux.be>'], [], '', self.logo)
        about.show()

    def on_poptoggled(self, widget, data=None):
        tag = data
        toggled = widget.get_active()
        entry_tags = self.postwin.entries['tags'].get_text().split()
        if tag in entry_tags and not toggled:
            entry_tags.remove(tag)
            self.postwin.entries['tags'].set_text(' '.join(entry_tags))
        elif tag not in entry_tags and toggled:
            entry_tags.append(tag)
            self.postwin.entries['tags'].set_text(' '.join(entry_tags))

    def on_resize_panel(self, widget, size):
        logger.debug('New size: %s' % size)
        pixbuf = self.logo
        new_pixbuf = pixbuf.scale_simple(size-1, size-1,
                                         gtk.gdk.INTERP_BILINEAR)
        self.visual = gtk.Image()
        self.visual.set_from_pixbuf(new_pixbuf)
        self.applet.add(self.visual)
        self.applet.show_all()

    def on_bckg_change(self, widget, background, colour, pixmap):
        logger.debug('New background: %s,%s,%s' % (background, colour, pixmap))
        if background == gnomeapplet.NO_BACKGROUND:
            #widget.modify_style(gtk.RcStyle())
            pass
        elif background == gnomeapplet.COLOR_BACKGROUND:
            widget.modify_bg(gtk.STATE_NORMAL, colour)
        elif background == gnomeapplet.PIXMAP_BACKGROUND:
            pass

    def on_dragdata_received(self, widget, context, x, y, selection, ttype,
                             time):
        url = ''.join([c for c in selection.data if c in VALID_CHAR])
        logger.info('Received url: %s' % url)
        txt = urllib.urlopen(url).read()
        title_match = self.title_re.search(txt)
        charset_match = self.charset_re.search(txt)
        if charset_match:
            charset = charset_match.groupdict()['enc']
        else:
            charset = 'utf-8'
        if title_match:
            try:
                title = unicode(' '.join(title_match.groupdict()['title'].split()),
                                charset)
            except UnicodeDecodeError:
                title = ' '.join(title_match.groupdict()['title'].split())
        else:
            title = ''
        self.postwin.entries['url'].set_text(url)
        self.postwin.entries['desc'].set_text(title)
        self.update_tags()
        logger.info('Retrieving popular tags')
        poptags = self.get_poptags(url)
        logger.info('Popular tags retrieved')
        self.add_poptags(poptags[:self.conf.pop_tag_num])
        self.postwin.show_all()

    def on_button_press(self, widget, evt):
        if evt.type == gtk.gdk.BUTTON_PRESS and evt.button == 3:
            self.applet.setup_menu(self.propxml, self.verbs, None)

    def on_post(self, evt, data=None):
        logger.info('Posting on del.icio.us')
        url = self.postwin.entries['url'].get_text()
        desc = self.postwin.entries['desc'].get_text()
        tags = self.postwin.entries['tags'].get_text()
        buff = self.postwin.entries['ext'].get_buffer()
        ext = buff.get_text(buff.get_start_iter(), buff.get_end_iter())
        if url and desc and tags:
            logger.info('Posting URL: %s' % url)
            self.delicious_user.do_post(url, desc, ext, tags)
            logger.info('Posting done')
            self.postwin.hide()
            self.clear_postwin()
        else:
            warn = gtk.MessageDialog(
                                     parent = self.postwin,
                                     flags = gtk.DIALOG_MODAL,
                                     type = gtk.MESSAGE_WARNING,
                                     buttons = gtk.BUTTONS_CLOSE)
            warn.set_markup(_('Nothing to post ! Make sure the URL, Description and Tags entries are not empty.'))
            warn.run()
            warn.destroy()
            return True

    def on_setpref(self, evt, data=None):
        self.conf.username = self.prefwin.entries['username'].get_text()
        self.conf.password = self.prefwin.entries['password'].get_text()
        self.delicious_user = delicious.User(self.conf.username,
                                             self.conf.password)
        color = self.prefwin.entries['commontagcolor'].get_color()
        self.conf.common_tag_color = '#%04x%04x%04x' % (color.red,
                                                        color.green, color.blue)
        self.conf.pop_tag_num = \
                self.prefwin.entries['maxpoptag'].get_value_as_int()

        self.prefwin.hide()
        return True

    def on_postwin_delete_event(self, widget, evt):
        self.postwin.hide()
        self.clear_postwin()
        return True

    def on_prefwin_delete_event(self, widget, evt):
        self.prefwin.hide()
        return True

    def on_cancelpost(self, evt, data=None):
        self.postwin.hide()
        self.clear_postwin()
        return True

    def on_match_selected(self, completion, model, cursor):
        match = model[cursor][0]
        etokens = self.postwin.entries['tags'].get_text().split()
        etokens[-1] = match
        self.postwin.entries['tags'].set_text(' '.join(etokens))
        return True

    def on_postwin_tags(self, widget):
        tags = self.postwin.entries['tags'].get_text().split()
        for tag, btn in self.populartags_buttons.items():
            btn.set_active(tag in tags)
        return True

    def on_register_clicked(self, evt, data=None):
        gnome.url_show('http://del.icio.us/register')
        return True
